<?php
/**
 * Locations Module - Add Work Location
 * Allows adding new work locations (offices, sites, warehouses, etc.)
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
    require_permission(70);
}

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

$page_title = 'Add Work Location';
$current_page = 'locations';
$errors = [];

// Get database connection
$pdo = get_database_connection();

// Get all active workers for site manager dropdown
$managers = [];
try {
    $stmt = $pdo->query("SELECT id, full_name, employee_number, job_title FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' ORDER BY full_name");
    $managers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Silently fail if table doesn't exist
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $type = $_POST['type'] ?? '';
    $address = trim($_POST['address'] ?? '');
    $city = trim($_POST['city'] ?? '');
    $postcode = trim($_POST['postcode'] ?? '');
    $capacity = $_POST['capacity'] ?? 0;
    $manager_id = $_POST['manager_id'] ?: null;
    $status = $_POST['status'] ?? 'active';
    $notes = trim($_POST['notes'] ?? '');

    // Shift patterns from POST (multiple shifts)
    $shift_patterns = isset($_POST['shift_patterns']) ? json_decode($_POST['shift_patterns'], true) : [];

    if (empty($name)) $errors[] = "Name is required";
    if (empty($shift_patterns)) $errors[] = "At least one shift pattern is required";

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Insert location (keeping old shift fields for backward compatibility)
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "work_locations
                (name, type, address, city, postcode, capacity, manager_id, shift_start_time, shift_end_time, shift_name, status, notes, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");

            // Use first shift pattern for backward compatibility
            $first_shift = $shift_patterns[0] ?? null;
            $stmt->execute([
                $name, $type, $address, $city, $postcode, $capacity, $manager_id,
                $first_shift['start_time'] ?? '09:00',
                $first_shift['end_time'] ?? '17:00',
                $first_shift['shift_name'] ?? 'Day Shift',
                $status, $notes
            ]);

            $location_id = $pdo->lastInsertId();

            // Insert all shift patterns
            $shift_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "location_shift_patterns
                (location_id, shift_name, shift_type, start_time, end_time, break_duration, days_of_week, is_default, is_active, valid_from, valid_until, notes)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

            foreach ($shift_patterns as $index => $shift) {
                $shift_stmt->execute([
                    $location_id,
                    $shift['shift_name'] ?? 'Shift ' . ($index + 1),
                    $shift['shift_type'] ?? 'day',
                    $shift['start_time'] ?? '09:00',
                    $shift['end_time'] ?? '17:00',
                    $shift['break_duration'] ?? 0,
                    $shift['days_of_week'] ?? 'Mon,Tue,Wed,Thu,Fri',
                    $index === 0 ? 1 : 0, // First shift is default
                    1, // is_active
                    $shift['valid_from'] ?? null,
                    $shift['valid_until'] ?? null,
                    $shift['notes'] ?? null
                ]);
            }

            $pdo->commit();
            $_SESSION['success'] = 'Location added successfully with ' . count($shift_patterns) . ' shift pattern(s)';

            // Handle redirect based on environment
            if (defined('MODULE_EMBEDDED')) {
                // Router mode - use JavaScript redirect
                echo '<script>window.location.href = "../../m/locations/list";</script>';
                echo '<noscript><meta http-equiv="refresh" content="0;url=../../m/locations/list"></noscript>';
            } else {
                // Direct access - use header redirect
                header('Location: list.php?success=1');
            }
            exit;
        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Only include header if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern"><i class="bi bi-plus-circle"></i> Add Work Location</h5>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0"><?php foreach ($errors as $error): ?><li><?php echo htmlspecialchars($error); ?></li><?php endforeach; ?></ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="row">
                            <div class="col-md-8 mb-3">
                                <label class="form-label">Location Name <span class="text-danger">*</span></label>
                                <input type="text" name="name" class="form-control" required>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Type</label>
                                <select name="type" class="form-select">
                                    <option value="">Select...</option>
                                    <option value="office">Office</option>
                                    <option value="warehouse">Warehouse</option>
                                    <option value="site">Site</option>
                                    <option value="remote">Remote</option>
                                </select>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Address</label>
                            <input type="text" name="address" class="form-control">
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">City</label>
                                <input type="text" name="city" class="form-control">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Postcode</label>
                                <input type="text" name="postcode" class="form-control">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Capacity</label>
                                <input type="number" name="capacity" class="form-control" value="0">
                                <small class="text-muted">Maximum number of workers</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Site Manager</label>
                                <select name="manager_id" class="form-select">
                                    <option value="">None</option>
                                    <?php foreach ($managers as $mgr): ?>
                                        <option value="<?php echo $mgr['id']; ?>">
                                            <?php echo htmlspecialchars($mgr['full_name']); ?>
                                            <?php if ($mgr['job_title']): ?>
                                                - <?php echo htmlspecialchars($mgr['job_title']); ?>
                                            <?php endif; ?>
                                            <?php if ($mgr['employee_number']): ?>
                                                (<?php echo htmlspecialchars($mgr['employee_number']); ?>)
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">Optional: Assign a site manager</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Status</label>
                                <select name="status" class="form-select">
                                    <option value="active">Active</option>
                                    <option value="inactive">Inactive</option>
                                </select>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Notes</label>
                            <textarea name="notes" class="form-control" rows="3" placeholder="Optional notes about this location..."><?php echo htmlspecialchars($_POST['notes'] ?? ''); ?></textarea>
                            <small class="text-muted">Optional: Add any relevant notes or information</small>
                        </div>

                        <hr>

                        <!-- Shift Patterns Section -->
                        <h6 class="mb-3">
                            <i class="bi bi-clock"></i> Shift Patterns
                            <button type="button" class="btn btn-sm btn-primary float-end" onclick="addShiftPattern()">
                                <i class="bi bi-plus-circle"></i> Add Shift Pattern
                            </button>
                        </h6>

                        <div id="shift-patterns-container" class="mb-3">
                            <!-- Shift patterns will be added here dynamically -->
                        </div>

                        <div class="alert alert-info small">
                            <i class="bi bi-info-circle"></i> Define shift patterns for this location. Workers will automatically use these times when marking attendance.
                        </div>

                        <input type="hidden" name="shift_patterns" id="shift_patterns_input" value="">

                        <hr>
                        <div class="d-flex justify-content-between">
                            <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php'; ?>"
                               class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle"></i> Add Location
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let shiftPatterns = [];
let shiftCounter = 0;

// Initialize with one default shift
document.addEventListener('DOMContentLoaded', function() {
    addShiftPattern({
        shift_name: 'Day Shift',
        shift_type: 'day',
        start_time: '09:00',
        end_time: '17:00',
        break_duration: 60,
        days_of_week: 'Mon,Tue,Wed,Thu,Fri'
    });
});

function addShiftPattern(data = {}) {
    const id = shiftCounter++;
    const pattern = {
        id: id,
        shift_name: data.shift_name || '',
        shift_type: data.shift_type || 'day',
        start_time: data.start_time || '09:00',
        end_time: data.end_time || '17:00',
        break_duration: data.break_duration || 0,
        days_of_week: data.days_of_week || 'Mon,Tue,Wed,Thu,Fri',
        valid_from: data.valid_from || '',
        valid_until: data.valid_until || '',
        notes: data.notes || ''
    };

    shiftPatterns.push(pattern);
    renderShiftPatterns();
}

function removeShiftPattern(id) {
    if (shiftPatterns.length === 1) {
        alert('At least one shift pattern is required');
        return;
    }
    shiftPatterns = shiftPatterns.filter(p => p.id !== id);
    renderShiftPatterns();
}

function updateShiftPattern(id, field, value) {
    const pattern = shiftPatterns.find(p => p.id === id);
    if (pattern) {
        pattern[field] = value;
        updateHiddenInput();
    }
}

function renderShiftPatterns() {
    const container = document.getElementById('shift-patterns-container');
    container.innerHTML = '';

    shiftPatterns.forEach((pattern, index) => {
        const card = document.createElement('div');
        card.className = 'card mb-3';
        card.innerHTML = `
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="mb-0">
                        <i class="bi bi-clock-fill text-primary"></i>
                        Shift Pattern ${index + 1}
                        ${index === 0 ? '<span class="badge bg-success ms-2">Default</span>' : ''}
                    </h6>
                    ${shiftPatterns.length > 1 ? `
                        <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeShiftPattern(${pattern.id})">
                            <i class="bi bi-trash"></i> Remove
                        </button>
                    ` : ''}
                </div>

                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Shift Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" value="${pattern.shift_name}"
                               onchange="updateShiftPattern(${pattern.id}, 'shift_name', this.value)"
                               placeholder="e.g., Day Shift, Night Shift" required>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Shift Type</label>
                        <select class="form-select" onchange="updateShiftPattern(${pattern.id}, 'shift_type', this.value)">
                            <option value="day" ${pattern.shift_type === 'day' ? 'selected' : ''}>Day Shift</option>
                            <option value="night" ${pattern.shift_type === 'night' ? 'selected' : ''}>Night Shift</option>
                            <option value="evening" ${pattern.shift_type === 'evening' ? 'selected' : ''}>Evening Shift</option>
                            <option value="split" ${pattern.shift_type === 'split' ? 'selected' : ''}>Split Shift</option>
                            <option value="rotating" ${pattern.shift_type === 'rotating' ? 'selected' : ''}>Rotating Shift</option>
                            <option value="flexible" ${pattern.shift_type === 'flexible' ? 'selected' : ''}>Flexible Hours</option>
                            <option value="temporary" ${pattern.shift_type === 'temporary' ? 'selected' : ''}>Temporary Shift</option>
                            <option value="special" ${pattern.shift_type === 'special' ? 'selected' : ''}>Special Shift</option>
                        </select>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-4 mb-3">
                        <label class="form-label">Start Time <span class="text-danger">*</span></label>
                        <input type="time" class="form-control" value="${pattern.start_time}"
                               onchange="updateShiftPattern(${pattern.id}, 'start_time', this.value)" required>
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="form-label">End Time <span class="text-danger">*</span></label>
                        <input type="time" class="form-control" value="${pattern.end_time}"
                               onchange="updateShiftPattern(${pattern.id}, 'end_time', this.value)" required>
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="form-label">Break Duration (minutes)</label>
                        <input type="number" class="form-control" value="${pattern.break_duration}" min="0"
                               onchange="updateShiftPattern(${pattern.id}, 'break_duration', this.value)">
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-12 mb-3">
                        <label class="form-label">Days of Week</label>
                        <div class="btn-group w-100" role="group">
                            ${['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'].map(day => `
                                <input type="checkbox" class="btn-check" id="day_${pattern.id}_${day}"
                                       ${pattern.days_of_week.includes(day) ? 'checked' : ''}
                                       onchange="updateDaysOfWeek(${pattern.id})">
                                <label class="btn btn-outline-primary btn-sm" for="day_${pattern.id}_${day}">${day}</label>
                            `).join('')}
                        </div>
                    </div>
                </div>

                ${pattern.shift_type === 'temporary' ? `
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Valid From</label>
                            <input type="date" class="form-control" value="${pattern.valid_from}"
                                   onchange="updateShiftPattern(${pattern.id}, 'valid_from', this.value)">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Valid Until</label>
                            <input type="date" class="form-control" value="${pattern.valid_until}"
                                   onchange="updateShiftPattern(${pattern.id}, 'valid_until', this.value)">
                        </div>
                    </div>
                ` : ''}

                <div class="mb-3">
                    <label class="form-label">Notes</label>
                    <textarea class="form-control" rows="2"
                              onchange="updateShiftPattern(${pattern.id}, 'notes', this.value)"
                              placeholder="Optional notes about this shift pattern">${pattern.notes}</textarea>
                </div>
            </div>
        `;
        container.appendChild(card);
    });

    updateHiddenInput();
}

function updateDaysOfWeek(patternId) {
    const days = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    const selected = days.filter(day =>
        document.getElementById(`day_${patternId}_${day}`).checked
    );
    updateShiftPattern(patternId, 'days_of_week', selected.join(','));
}

function updateHiddenInput() {
    const data = shiftPatterns.map(({id, ...rest}) => rest); // Remove id before sending
    document.getElementById('shift_patterns_input').value = JSON.stringify(data);
}

// Validate form before submission
document.querySelector('form').addEventListener('submit', function(e) {
    if (shiftPatterns.length === 0) {
        e.preventDefault();
        alert('Please add at least one shift pattern');
        return false;
    }

    // Ensure all required fields are filled
    for (let pattern of shiftPatterns) {
        if (!pattern.shift_name || !pattern.start_time || !pattern.end_time) {
            e.preventDefault();
            alert('Please fill in all required fields for all shift patterns');
            return false;
        }
    }

    updateHiddenInput();
});
</script>

<?php
// Only include footer if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
