<?php
require __DIR__ . '/../../config/database.php';
require __DIR__ . '/../../includes/functions.php';

require_login();
require_permission(50);
$page_title = 'Request Holiday';

$errors = [];
$success = '';

// Get workers for dropdown (if HR+)
$workers = [];
if ($_SESSION['permission_level'] >= 70) {
    $stmt = $pdo->prepare("SELECT id, CONCAT(first_name, ' ', last_name) as name, 
                          employee_number, annual_leave_entitlement
                          FROM sponsored_workers 
                          WHERE is_active = 1 AND business_profile_id = ?
                          ORDER BY first_name, last_name");
    $stmt->execute([$_SESSION['business_profile_id']]);
    $workers = $stmt->fetchAll();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $worker_id = $_POST['worker_id'] ?? null;
    $leave_type = trim($_POST['leave_type'] ?? '');
    $start_date = trim($_POST['start_date'] ?? '');
    $end_date = trim($_POST['end_date'] ?? '');
    $reason = trim($_POST['reason'] ?? '');
    
    // Validation
    if (empty($worker_id)) $errors[] = "Please select a worker";
    if (empty($leave_type)) $errors[] = "Please select leave type";
    if (empty($start_date)) $errors[] = "Start date is required";
    if (empty($end_date)) $errors[] = "End date is required";
    
    if (empty($errors)) {
        // Calculate working days (excluding weekends)
        $start = new DateTime($start_date);
        $end = new DateTime($end_date);
        $interval = $start->diff($end);
        $days = $interval->days + 1;
        
        // Simple working days calculation (excluding Saturdays and Sundays)
        $working_days = 0;
        $current = clone $start;
        while ($current <= $end) {
            $day_of_week = $current->format('N');
            if ($day_of_week < 6) { // Monday = 1, Sunday = 7
                $working_days++;
            }
            $current->modify('+1 day');
        }
        
        // Check for overlapping requests
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM holiday_requests 
                              WHERE worker_id = ? 
                              AND approval_status != 'Rejected'
                              AND business_profile_id = ?
                              AND (
                                  (start_date <= ? AND end_date >= ?) OR
                                  (start_date <= ? AND end_date >= ?) OR
                                  (start_date >= ? AND end_date <= ?)
                              )");
        $stmt->execute([$worker_id, $_SESSION['business_profile_id'], 
                       $start_date, $start_date, $end_date, $end_date,
                       $start_date, $end_date]);
        $overlapping = $stmt->fetchColumn();
        
        if ($overlapping > 0) {
            $errors[] = "This worker already has a holiday request for these dates";
        }
        
        if (empty($errors)) {
            // Insert holiday request
            $stmt = $pdo->prepare("INSERT INTO holiday_requests 
                                  (worker_id, business_profile_id, leave_type, start_date, end_date, 
                                   total_days, reason, approval_status, requested_by, created_at)
                                  VALUES (?, ?, ?, ?, ?, ?, ?, 'Pending', ?, NOW())");
            
            if ($stmt->execute([$worker_id, $_SESSION['business_profile_id'], $leave_type, 
                               $start_date, $end_date, $working_days, $reason, $_SESSION['user_id']])) {
                
                // Log activity
                log_activity($pdo, 'holiday_request_created', 'holiday_requests', 
                           $pdo->lastInsertId(), "Holiday request created for worker ID: $worker_id");
                
                $_SESSION['success'] = "Holiday request submitted successfully";
                header('Location: list.php');
                exit;
            } else {
                $errors[] = "Failed to create holiday request";
            }
        }
    }
}

require __DIR__ . '/../../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-8">
        <h2><i class="bi bi-calendar-plus"></i> Request Holiday</h2>
        <p class="text-muted">Submit a new holiday request for approval</p>
    </div>
    <div class="col-md-4 text-end">
        <a href="list.php" class="btn btn-outline-secondary">
            <i class="bi bi-arrow-left"></i> Back to List
        </a>
    </div>
</div>

<?php if (!empty($errors)): ?>
<div class="alert alert-danger">
    <strong>Please fix the following errors:</strong>
    <ul class="mb-0 mt-2">
        <?php foreach ($errors as $error): ?>
        <li><?= htmlspecialchars($error) ?></li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="stat-card">
            <form method="POST" action="">
                
                <!-- Worker Selection -->
                <?php if ($_SESSION['permission_level'] >= 70): ?>
                <div class="mb-3">
                    <label class="form-label">Worker *</label>
                    <select name="worker_id" class="form-select" required id="workerSelect">
                        <option value="">Select Worker</option>
                        <?php foreach ($workers as $worker): ?>
                        <option value="<?= $worker['id'] ?>" 
                                data-entitlement="<?= $worker['annual_leave_entitlement'] ?>">
                            <?= htmlspecialchars($worker['name']) ?> 
                            (<?= htmlspecialchars($worker['employee_number']) ?>)
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div id="leaveBalance" class="alert alert-info" style="display: none;">
                    <strong>Annual Leave Entitlement:</strong> <span id="entitlementDays"></span> days
                </div>
                <?php endif; ?>
                
                <!-- Leave Type -->
                <div class="mb-3">
                    <label class="form-label">Leave Type *</label>
                    <select name="leave_type" class="form-select" required>
                        <option value="">Select Type</option>
                        <option value="Annual Leave">Annual Leave</option>
                        <option value="Sick Leave">Sick Leave</option>
                        <option value="Unpaid Leave">Unpaid Leave</option>
                        <option value="Maternity/Paternity">Maternity/Paternity</option>
                        <option value="Other">Other</option>
                    </select>
                </div>
                
                <!-- Date Range -->
                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Start Date *</label>
                        <input type="date" name="start_date" class="form-control" required
                               min="<?= date('Y-m-d') ?>" id="startDate">
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="form-label">End Date *</label>
                        <input type="date" name="end_date" class="form-control" required
                               min="<?= date('Y-m-d') ?>" id="endDate">
                    </div>
                </div>
                
                <div id="daysCalculation" class="alert alert-secondary" style="display: none;">
                    <strong>Working Days:</strong> <span id="workingDays">0</span> days
                </div>
                
                <!-- Reason -->
                <div class="mb-3">
                    <label class="form-label">Reason</label>
                    <textarea name="reason" class="form-control" rows="3" 
                              placeholder="Optional: Provide additional details"></textarea>
                </div>
                
                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-check-circle"></i> Submit Request
                    </button>
                    <a href="list.php" class="btn btn-outline-secondary">Cancel</a>
                </div>
            </form>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="stat-card">
            <h5 class="mb-3"><i class="bi bi-info-circle"></i> Important Notes</h5>
            <ul class="small">
                <li>Holiday requests require manager approval</li>
                <li>Submit requests at least 2 weeks in advance</li>
                <li>Working days calculation excludes weekends</li>
                <li>Check team calendar for conflicts</li>
                <li>Approved holidays will appear on your calendar</li>
            </ul>
        </div>
        
        <div class="stat-card mt-3">
            <h6 class="mb-2">Leave Types</h6>
            <small class="text-muted">
                <strong>Annual Leave:</strong> Paid time off from annual entitlement<br>
                <strong>Sick Leave:</strong> Medical absence<br>
                <strong>Unpaid Leave:</strong> Time off without pay<br>
                <strong>Maternity/Paternity:</strong> Parental leave
            </small>
        </div>
    </div>
</div>

<script>
// Show worker's leave entitlement
document.getElementById('workerSelect')?.addEventListener('change', function() {
    const entitlement = this.options[this.selectedIndex]?.dataset.entitlement;
    const balanceDiv = document.getElementById('leaveBalance');
    const entitlementSpan = document.getElementById('entitlementDays');
    
    if (entitlement && balanceDiv && entitlementSpan) {
        entitlementSpan.textContent = entitlement;
        balanceDiv.style.display = 'block';
    } else if (balanceDiv) {
        balanceDiv.style.display = 'none';
    }
});

// Calculate working days
function calculateWorkingDays() {
    const startInput = document.getElementById('startDate');
    const endInput = document.getElementById('endDate');
    const calcDiv = document.getElementById('daysCalculation');
    const workingDaysSpan = document.getElementById('workingDays');
    
    if (!startInput || !endInput || !calcDiv || !workingDaysSpan) return;
    
    const start = new Date(startInput.value);
    const end = new Date(endInput.value);
    
    if (start && end && start <= end) {
        let workingDays = 0;
        let current = new Date(start);
        
        while (current <= end) {
            const day = current.getDay();
            if (day !== 0 && day !== 6) { // Not Sunday (0) or Saturday (6)
                workingDays++;
            }
            current.setDate(current.getDate() + 1);
        }
        
        workingDaysSpan.textContent = workingDays;
        calcDiv.style.display = 'block';
    } else {
        calcDiv.style.display = 'none';
    }
}

document.getElementById('startDate')?.addEventListener('change', calculateWorkingDays);
document.getElementById('endDate')?.addEventListener('change', calculateWorkingDays);

// Set minimum end date based on start date
document.getElementById('startDate')?.addEventListener('change', function() {
    const endInput = document.getElementById('endDate');
    if (endInput) {
        endInput.min = this.value;
    }
});
</script>

<?php require __DIR__ . '/../../includes/footer.php'; ?>
