<?php
require __DIR__ . '/../../config/database.php';
require __DIR__ . '/../../includes/functions.php';

require_login();
$page_title = 'Holiday Requests';

$status_filter = $_GET['status'] ?? '';
$where = ["hr.business_profile_id = ?"];
$params = [$_SESSION['business_profile_id']];

if ($status_filter) {
    $where[] = "hr.approval_status = ?";
    $params[] = $status_filter;
}

$where_clause = implode(' AND ', $where);

// Fetch holiday requests
$sql = "SELECT hr.*, 
        CONCAT(sw.first_name, ' ', sw.last_name) as worker_name,
        sw.employee_number,
        CONCAT(u.full_name) as approved_by_name
        FROM holiday_requests hr
        JOIN sponsored_workers sw ON hr.worker_id = sw.id
        LEFT JOIN users u ON hr.approved_by = u.id
        WHERE $where_clause
        ORDER BY hr.created_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$requests = $stmt->fetchAll();

// Count by status
$stmt = $pdo->prepare("SELECT approval_status, COUNT(*) as count 
                      FROM holiday_requests 
                      WHERE business_profile_id = ? 
                      GROUP BY approval_status");
$stmt->execute([$_SESSION['business_profile_id']]);
$status_counts = [];
foreach ($stmt->fetchAll() as $row) {
    $status_counts[$row['approval_status']] = $row['count'];
}

require __DIR__ . '/../../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-8">
        <h2><i class="bi bi-calendar-check"></i> Holiday Requests</h2>
        <p class="text-muted">Manage staff holiday requests and approvals</p>
    </div>
    <div class="col-md-4 text-end">
        <a href="request.php" class="btn btn-primary">
            <i class="bi bi-plus-circle"></i> New Request
        </a>
    </div>
</div>

<!-- Statistics Cards -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3 class="text-warning"><?= $status_counts['Pending'] ?? 0 ?></h3>
            <small>Pending</small>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3 class="text-success"><?= $status_counts['Approved'] ?? 0 ?></h3>
            <small>Approved</small>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3 class="text-danger"><?= $status_counts['Rejected'] ?? 0 ?></h3>
            <small>Rejected</small>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3><?= count($requests) ?></h3>
            <small>Total Requests</small>
        </div>
    </div>
</div>

<!-- Filter Tabs -->
<ul class="nav nav-tabs mb-3">
    <li class="nav-item">
        <a class="nav-link <?= $status_filter === '' ? 'active' : '' ?>" href="list.php">All</a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $status_filter === 'Pending' ? 'active' : '' ?>" href="list.php?status=Pending">
            Pending <span class="badge bg-warning"><?= $status_counts['Pending'] ?? 0 ?></span>
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $status_filter === 'Approved' ? 'active' : '' ?>" href="list.php?status=Approved">
            Approved <span class="badge bg-success"><?= $status_counts['Approved'] ?? 0 ?></span>
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $status_filter === 'Rejected' ? 'active' : '' ?>" href="list.php?status=Rejected">
            Rejected <span class="badge bg-danger"><?= $status_counts['Rejected'] ?? 0 ?></span>
        </a>
    </li>
</ul>

<!-- Requests Table -->
<div class="stat-card">
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>Worker</th>
                    <th>Leave Type</th>
                    <th>Start Date</th>
                    <th>End Date</th>
                    <th>Days</th>
                    <th>Status</th>
                    <th>Requested</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($requests)): ?>
                <tr>
                    <td colspan="8" class="text-center py-4">
                        <i class="bi bi-inbox" style="font-size: 48px; opacity: 0.3;"></i>
                        <p class="text-muted mt-2">No holiday requests found</p>
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($requests as $request): ?>
                <tr>
                    <td>
                        <strong><?= htmlspecialchars($request['worker_name']) ?></strong><br>
                        <small class="text-muted"><?= htmlspecialchars($request['employee_number']) ?></small>
                    </td>
                    <td><?= htmlspecialchars($request['leave_type']) ?></td>
                    <td><?= date('d/m/Y', strtotime($request['start_date'])) ?></td>
                    <td><?= date('d/m/Y', strtotime($request['end_date'])) ?></td>
                    <td><strong><?= $request['total_days'] ?></strong></td>
                    <td>
                        <?php
                        $badge_class = match($request['approval_status']) {
                            'Approved' => 'bg-success',
                            'Rejected' => 'bg-danger',
                            'Pending' => 'bg-warning',
                            default => 'bg-secondary'
                        };
                        ?>
                        <span class="badge <?= $badge_class ?>">
                            <?= htmlspecialchars($request['approval_status']) ?>
                        </span>
                    </td>
                    <td><?= date('d/m/Y', strtotime($request['created_at'])) ?></td>
                    <td>
                        <a href="view.php?id=<?= $request['id'] ?>" class="btn btn-sm btn-outline-primary" title="View">
                            <i class="bi bi-eye"></i>
                        </a>
                        <?php if ($request['approval_status'] === 'Pending' && $_SESSION['permission_level'] >= 70): ?>
                        <a href="approve.php?id=<?= $request['id'] ?>" class="btn btn-sm btn-outline-success" title="Review">
                            <i class="bi bi-check-circle"></i>
                        </a>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php require __DIR__ . '/../../includes/footer.php'; ?>