<?php
/**
 * Holidays Module - Delete Bank Holiday
 * Role-based deletion:
 * - Permission 70-89: Can delete bank holidays
 * - Permission 90-99: Can delete bank holidays
 * - Permission 100+: Can PERMANENT delete with PIN
 *
 * Note: Bank holidays are simple records without soft delete
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }
}

// Check minimum permission - delete requires level 70
if ($current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete bank holidays.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
    exit;
}

$holiday_id = (int)($_POST['id'] ?? 0);
$pin = $_POST['pin'] ?? '';

if ($holiday_id <= 0) {
    $_SESSION['error'] = 'Invalid holiday ID.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
    exit;
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if holiday exists
    $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "bank_holidays WHERE id = ?");
    $stmt->execute([$holiday_id]);
    $holiday = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$holiday) {
        $_SESSION['error'] = 'Bank holiday not found.';
        header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
        exit;
    }

    $permission_level = $current_user['permission_level'];

    // For permission 70-99: Allow deletion with audit log
    if ($permission_level >= 70 && $permission_level < 100) {
        // Log the deletion for audit trail
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'DELETE', 'bank_holidays', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'holiday_name' => $holiday['name'],
            'date' => $holiday['date'],
            'region' => $holiday['region'],
            'reason' => 'User deleted bank holiday'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $holiday_id, $details, $ip]);

        // Delete the record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "bank_holidays WHERE id = ?");
        $stmt->execute([$holiday_id]);

        $_SESSION['success'] = 'Bank holiday "' . htmlspecialchars($holiday['name']) . '" on ' . date('d/m/Y', strtotime($holiday['date'])) . ' has been deleted.';
    }

    // For permission 100+: Require PIN for deletion
    elseif ($permission_level >= 100) {
        // REQUIRE PIN for admin deletion
        if (empty($pin)) {
            $_SESSION['error'] = 'PIN is required for deletion.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
            exit;
        }

        // Verify PIN
        if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
            $_SESSION['error'] = 'Invalid PIN. Deletion cancelled.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
            exit;
        }

        // Log the deletion for audit trail
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'ADMIN_DELETE', 'bank_holidays', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'holiday_name' => $holiday['name'],
            'date' => $holiday['date'],
            'region' => $holiday['region'],
            'reason' => 'Admin deletion with PIN verification'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $holiday_id, $details, $ip]);

        // Delete the record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "bank_holidays WHERE id = ?");
        $stmt->execute([$holiday_id]);

        $_SESSION['success'] = 'Bank holiday "' . htmlspecialchars($holiday['name']) . '" on ' . date('d/m/Y', strtotime($holiday['date'])) . ' has been PERMANENTLY DELETED.';
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting bank holiday: ' . $e->getMessage();
}

// Redirect back to list
header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/holidays/list' : 'list.php'));
exit;
?>
