<?php
/**
 * Holidays Module - Add Bank Holiday
 * Allows admin users to add UK statutory bank holidays
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
    require_permission(90); // Only senior staff can manage holidays
}

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

$page_title = 'Add Bank Holiday';
$current_page = 'holidays';
$errors = [];

// Get database connection
$pdo = get_database_connection();

// Get year from URL or use current year
$year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
$year = max(2020, min(2030, $year));

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $date = $_POST['date'] ?? '';
    $name = trim($_POST['name'] ?? '');
    $region = $_POST['region'] ?? 'UK';

    // Validation
    if (empty($date)) {
        $errors[] = "Date is required";
    }
    if (empty($name)) {
        $errors[] = "Holiday name is required";
    }

    // Check if holiday already exists for this date and region
    if (!empty($date)) {
        $check_stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "bank_holidays WHERE date = ? AND region = ?");
        $check_stmt->execute([$date, $region]);
        if ($check_stmt->fetch()) {
            $errors[] = "A bank holiday already exists for this date and region";
        }
    }

    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "bank_holidays (date, name, region) VALUES (?, ?, ?)");
            $stmt->execute([$date, $name, $region]);

            $_SESSION['success'] = 'Bank holiday added successfully';

            // Redirect based on how the page was accessed
            if (defined('MODULE_EMBEDDED')) {
                header('Location: ../../m/holidays/list?year=' . $year);
            } else {
                header('Location: list.php?year=' . $year);
            }
            exit;
        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Only include header if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<style>
.form-section {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
}

.form-section-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 2px solid var(--primary-color);
    display: flex;
    align-items: center;
    gap: 8px;
}
</style>

<div class="row justify-content-center">
    <div class="col-12 col-lg-8">
        <div class="modern-card">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-calendar-event"></i> Add UK Bank Holiday
                </h5>
            </div>
            <div class="card-body-modern">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <i class="bi bi-exclamation-triangle"></i>
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <form method="POST">
                    <!-- Holiday Information -->
                    <div class="form-section">
                        <div class="form-section-title">
                            <i class="bi bi-info-circle"></i> Holiday Information
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Date <span class="text-danger">*</span></label>
                                <input type="date" name="date" class="form-control" required
                                       value="<?php echo htmlspecialchars($_POST['date'] ?? $year . '-01-01'); ?>"
                                       min="<?php echo $year; ?>-01-01"
                                       max="<?php echo $year; ?>-12-31">
                                <small class="text-muted">Select the bank holiday date</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Region</label>
                                <select name="region" class="form-select">
                                    <option value="UK" <?php echo ($_POST['region'] ?? 'UK') === 'UK' ? 'selected' : ''; ?>>UK (All)</option>
                                    <option value="England" <?php echo ($_POST['region'] ?? '') === 'England' ? 'selected' : ''; ?>>England</option>
                                    <option value="Scotland" <?php echo ($_POST['region'] ?? '') === 'Scotland' ? 'selected' : ''; ?>>Scotland</option>
                                    <option value="Wales" <?php echo ($_POST['region'] ?? '') === 'Wales' ? 'selected' : ''; ?>>Wales</option>
                                    <option value="Northern Ireland" <?php echo ($_POST['region'] ?? '') === 'Northern Ireland' ? 'selected' : ''; ?>>Northern Ireland</option>
                                </select>
                                <small class="text-muted">Regional holidays may differ</small>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-12 mb-3">
                                <label class="form-label">Holiday Name <span class="text-danger">*</span></label>
                                <input type="text" name="name" class="form-control" required
                                       value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>"
                                       placeholder="e.g., New Year's Day, Christmas Day">
                                <small class="text-muted">Official name of the bank holiday</small>
                            </div>
                        </div>
                    </div>

                    <!-- Common UK Bank Holidays -->
                    <div class="alert alert-info">
                        <h6><i class="bi bi-info-circle"></i> Common UK Bank Holidays for <?php echo $year; ?>:</h6>
                        <ul class="small mb-0">
                            <li><strong>New Year's Day</strong> - 1 January</li>
                            <li><strong>Good Friday</strong> - Varies (typically March/April)</li>
                            <li><strong>Easter Monday</strong> - Varies (typically March/April)</li>
                            <li><strong>Early May Bank Holiday</strong> - First Monday in May</li>
                            <li><strong>Spring Bank Holiday</strong> - Last Monday in May</li>
                            <li><strong>Summer Bank Holiday</strong> - Last Monday in August</li>
                            <li><strong>Christmas Day</strong> - 25 December</li>
                            <li><strong>Boxing Day</strong> - 26 December</li>
                        </ul>
                        <small class="text-muted mt-2 d-block">Note: When a bank holiday falls on a weekend, a substitute weekday becomes a bank holiday.</small>
                    </div>

                    <hr class="my-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/holidays/list?year=' . $year : 'list.php?year=' . $year; ?>"
                           class="btn btn-secondary">
                            <i class="bi bi-x-circle"></i> Cancel
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-circle"></i> Add Bank Holiday
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<?php
// Only include footer if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
