<?php
/**
 * Documents Module - Upload Document
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

// Check permission (60+ can upload documents)
if (($_SESSION['permission_level'] ?? 0) < 60) {
    die('Access denied. You do not have permission to upload documents.');
}

$page_title = 'Upload Document';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

$errors = [];
$success = '';

// Get workers for dropdown
$workers_sql = "SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' ORDER BY full_name";
$workers_stmt = $pdo->query($workers_sql);
$workers = $workers_stmt->fetchAll(PDO::FETCH_ASSOC);

// Document categories
$categories = [
    'passport' => 'Passport',
    'visa' => 'Visa',
    'cos' => 'Certificate of Sponsorship',
    'contract' => 'Employment Contract',
    'rtw' => 'Right to Work',
    'qualification' => 'Qualifications',
    'reference' => 'References',
    'company' => 'Company Document',
    'policy' => 'Policy/Procedure',
    'other' => 'Other'
];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $worker_id = $_POST['worker_id'] ?: null;
    $document_type = $_POST['document_type'] ?? '';
    $document_name = trim($_POST['document_name'] ?? '');
    $expiry_date = $_POST['expiry_date'] ?: null;
    $notes = trim($_POST['notes'] ?? '');

    // Validation
    if (empty($document_type)) {
        $errors[] = "Please select a document type";
    }

    if (empty($document_name)) {
        $errors[] = "Please enter a document name";
    }

    if (!isset($_FILES['document']) || $_FILES['document']['error'] !== UPLOAD_ERR_OK) {
        if ($_FILES['document']['error'] === UPLOAD_ERR_NO_FILE) {
            $errors[] = "Please select a file to upload";
        } else {
            $errors[] = "File upload error. Please try again.";
        }
    }

    if (empty($errors)) {
        // Create upload directory if not exists
        $upload_dir = __DIR__ . '/../../uploads/documents/';
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }

        // Generate unique filename
        $file_extension = strtolower(pathinfo($_FILES['document']['name'], PATHINFO_EXTENSION));
        $allowed_extensions = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx', 'xls', 'xlsx'];

        if (!in_array($file_extension, $allowed_extensions)) {
            $errors[] = "Invalid file type. Allowed: " . implode(', ', $allowed_extensions);
        } else {
            $unique_filename = uniqid() . '_' . time() . '.' . $file_extension;
            $file_path = 'uploads/documents/' . $unique_filename;
            $full_path = $upload_dir . $unique_filename;

            if (move_uploaded_file($_FILES['document']['tmp_name'], $full_path)) {
                try {
                    // Insert into database
                    $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "documents
                                          (worker_id, document_type, document_name, file_path,
                                          file_size, expiry_date, notes, uploaded_by, uploaded_at)
                                          VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");

                    $stmt->execute([
                        $worker_id,
                        $document_type,
                        $document_name,
                        $file_path,
                        $_FILES['document']['size'],
                        $expiry_date,
                        $notes,
                        $_SESSION['user_id']
                    ]);

                    $_SESSION['success_message'] = 'Document uploaded successfully';
                    header('Location: list.php');
                    exit;

                } catch (Exception $e) {
                    $errors[] = 'Database error: ' . $e->getMessage();
                    // Delete uploaded file if database insert fails
                    if (file_exists($full_path)) {
                        unlink($full_path);
                    }
                }
            } else {
                $errors[] = "Failed to move uploaded file";
            }
        }
    }
}
?>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <strong>Error!</strong>
        <ul class="mb-0">
            <?php foreach ($errors as $error): ?>
                <li><?= htmlspecialchars($error) ?></li>
            <?php endforeach; ?>
        </ul>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<div class="modern-card">
    <div class="card-body-modern">
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h5><i class="bi bi-upload"></i> Upload Document</h5>
            <a href="list.php" class="btn btn-secondary btn-sm">
                <i class="bi bi-arrow-left"></i> Back to List
            </a>
        </div>
        <hr>

        <form method="POST" enctype="multipart/form-data">
            <div class="row">
                <div class="col-md-6">
                    <h6 class="text-primary mb-3">Document Information</h6>

                    <div class="mb-3">
                        <label class="form-label">Document Type <span class="text-danger">*</span></label>
                        <select name="document_type" class="form-select" required>
                            <option value="">-- Select Type --</option>
                            <?php foreach ($categories as $value => $label): ?>
                                <option value="<?= $value ?>" <?= (isset($_POST['document_type']) && $_POST['document_type'] === $value) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($label) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="text-muted">Select the category that best describes this document</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Document Name <span class="text-danger">*</span></label>
                        <input type="text" name="document_name" class="form-control" required
                               value="<?= htmlspecialchars($_POST['document_name'] ?? '') ?>"
                               placeholder="e.g., John Smith Passport">
                        <small class="text-muted">A descriptive name for this document</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Associated Worker</label>
                        <select name="worker_id" class="form-select">
                            <option value="">-- Company/General Document --</option>
                            <?php foreach ($workers as $worker): ?>
                                <option value="<?= $worker['id'] ?>" <?= (isset($_POST['worker_id']) && $_POST['worker_id'] == $worker['id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($worker['full_name']) ?> (<?= htmlspecialchars($worker['employee_number']) ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <small class="text-muted">Leave empty for company-wide documents (policies, procedures, etc.)</small>
                    </div>
                </div>

                <div class="col-md-6">
                    <h6 class="text-primary mb-3">File & Details</h6>

                    <div class="mb-3">
                        <label class="form-label">Select File <span class="text-danger">*</span></label>
                        <input type="file" name="document" class="form-control" required accept=".pdf,.jpg,.jpeg,.png,.doc,.docx,.xls,.xlsx">
                        <small class="text-muted">Allowed: PDF, JPG, PNG, DOC, DOCX, XLS, XLSX (Max 10MB)</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Expiry Date</label>
                        <input type="date" name="expiry_date" class="form-control"
                               value="<?= htmlspecialchars($_POST['expiry_date'] ?? '') ?>">
                        <small class="text-muted">Set expiry date for time-sensitive documents (passports, visas, etc.)</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Notes</label>
                        <textarea name="notes" class="form-control" rows="3"
                                  placeholder="Any additional information..."><?= htmlspecialchars($_POST['notes'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>

            <hr>

            <div class="alert alert-info">
                <i class="bi bi-info-circle"></i> <strong>Upload Guidelines:</strong>
                <ul class="mb-0 small mt-2">
                    <li><strong>Worker Documents:</strong> Passports, visas, contracts, qualifications linked to specific workers</li>
                    <li><strong>Company Documents:</strong> Policies, procedures, licenses not linked to any worker</li>
                    <li>Maximum file size: 10 MB</li>
                    <li>Use clear, descriptive names for easy identification</li>
                    <li>Set expiry dates for documents that need renewal tracking</li>
                </ul>
            </div>

            <div class="d-flex justify-content-between">
                <a href="list.php" class="btn btn-secondary">
                    <i class="bi bi-x-circle"></i> Cancel
                </a>
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-upload"></i> Upload Document
                </button>
            </div>
        </form>
    </div>
</div>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>
