<?php
/**
 * Documents Module - List View
 */

$page_title = 'Document Management';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

// Filters
$worker_id = $_GET['worker_id'] ?? '';
$category = $_GET['category'] ?? '';
$search = $_GET['search'] ?? '';

// Build query
$where = ["1=1"];
$params = [];

if ($worker_id) {
    $where[] = "d.worker_id = ?";
    $params[] = $worker_id;
}

if ($category) {
    $where[] = "d.document_type = ?";
    $params[] = $category;
}

if ($search) {
    $where[] = "(d.document_name LIKE ? OR d.notes LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
}

$where_clause = implode(' AND ', $where);

// Fetch documents
$sql = "SELECT d.*,
        sw.full_name as worker_name,
        sw.employee_number,
        u1.full_name as uploaded_by_name
        FROM " . DB_PREFIX . "documents d
        LEFT JOIN " . DB_PREFIX . "sponsored_workers sw ON d.worker_id = sw.id
        LEFT JOIN " . DB_PREFIX . "users u1 ON d.uploaded_by = u1.id
        WHERE $where_clause
        ORDER BY d.uploaded_at DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$documents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get categories
$categories = [
    'passport' => 'Passport',
    'visa' => 'Visa',
    'cos' => 'Certificate of Sponsorship',
    'contract' => 'Employment Contract',
    'rtw' => 'Right to Work',
    'qualification' => 'Qualifications',
    'reference' => 'References',
    'other' => 'Other'
];

// Get workers for filter
$workers_sql = "SELECT id, full_name, employee_number
                FROM " . DB_PREFIX . "sponsored_workers
                WHERE status = 'active'
                ORDER BY full_name";
$workers_stmt = $pdo->query($workers_sql);
$workers = $workers_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$stats_sql = "SELECT
    COUNT(*) as total_docs,
    SUM(CASE WHEN expiry_date IS NOT NULL AND expiry_date < CURDATE() THEN 1 ELSE 0 END) as expired,
    SUM(CASE WHEN expiry_date IS NOT NULL AND expiry_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as expiring_soon
FROM " . DB_PREFIX . "documents";
$stats_stmt = $pdo->query($stats_sql);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>

<div class="stat-card">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h5><i class="bi bi-files"></i> Document Management</h5>
            <p class="text-muted mb-0">Manage worker documents and compliance files</p>
        </div>
        <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/documents/upload' : 'upload.php' ?>" class="btn btn-primary">
            <i class="bi bi-upload"></i> Upload Document
        </a>
    </div>

    <!-- Quick Actions -->
    <div class="quick-actions-grid mb-4">
        <!-- Total Documents -->
        <div class="quick-action-btn gradient-pink">
            <div class="quick-action-icon">
                <i class="ri-file-list-3-fill"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['total_docs'] ?? 0); ?> Total Documents</span>
        </div>

        <!-- Expiring Soon -->
        <div class="quick-action-btn gradient-orange">
            <div class="quick-action-icon">
                <span class="material-symbols-outlined" style="font-size: 1.75rem;">schedule</span>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['expiring_soon'] ?? 0); ?> Expiring Soon</span>
        </div>

        <!-- Expired -->
        <div class="quick-action-btn gradient-red">
            <div class="quick-action-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['expired'] ?? 0); ?> Expired</span>
        </div>

        <!-- Document Types -->
        <div class="quick-action-btn gradient-cyan">
            <div class="quick-action-icon">
                <i class="ri-folder-3-fill"></i>
            </div>
            <span class="quick-action-label"><?php echo count($categories); ?> Categories</span>
        </div>
    </div>

    <!-- Search and Filters -->
    <form method="GET" class="row g-3 mb-4">
        <div class="col-md-4">
            <input type="text" name="search" class="form-control"
                   placeholder="Search documents..."
                   value="<?= htmlspecialchars($search) ?>">
        </div>
        <div class="col-md-3">
            <select name="worker_id" class="form-select">
                <option value="">All Workers</option>
                <?php foreach ($workers as $worker): ?>
                    <option value="<?= $worker['id'] ?>" <?= $worker_id == $worker['id'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($worker['full_name']) ?> (<?= htmlspecialchars($worker['employee_number']) ?>)
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-3">
            <select name="category" class="form-select">
                <option value="">All Categories</option>
                <?php foreach ($categories as $value => $label): ?>
                    <option value="<?= $value ?>" <?= $category === $value ? 'selected' : '' ?>>
                        <?= htmlspecialchars($label) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="submit" class="btn btn-secondary w-100">
                <i class="bi bi-search"></i> Search
            </button>
        </div>
    </form>

    <!-- Documents Table -->
    <?php if (empty($documents)): ?>
        <div class="alert alert-info text-center">
            <i class="bi bi-inbox" style="font-size: 48px;"></i>
            <p class="mt-3 mb-2"><strong>No Documents Found</strong></p>
            <p class="text-muted">Upload documents to get started</p>
            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/documents/upload' : 'upload.php' ?>" class="btn btn-primary mt-2">
                <i class="bi bi-upload"></i> Upload First Document
            </a>
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-hover">
                <thead class="table-light">
                    <tr>
                        <th>Document Name</th>
                        <th>Type</th>
                        <th>Worker</th>
                        <th>Uploaded</th>
                        <th>Expiry</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($documents as $doc): ?>
                    <tr>
                        <td>
                            <i class="bi bi-file-earmark-pdf text-danger"></i>
                            <strong><?= htmlspecialchars($doc['document_name']) ?></strong>
                            <?php if ($doc['notes']): ?>
                                <br><small class="text-muted"><?= htmlspecialchars($doc['notes']) ?></small>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="badge bg-info">
                                <?= htmlspecialchars($categories[$doc['document_type']] ?? $doc['document_type']) ?>
                            </span>
                        </td>
                        <td>
                            <?php if ($doc['worker_name']): ?>
                                <?= htmlspecialchars($doc['worker_name']) ?>
                                <br><small class="text-muted"><?= htmlspecialchars($doc['employee_number']) ?></small>
                            <?php else: ?>
                                <span class="text-muted">General</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?= date('d M Y', strtotime($doc['uploaded_at'])) ?><br>
                            <small class="text-muted">by <?= htmlspecialchars($doc['uploaded_by_name'] ?? 'Unknown') ?></small>
                        </td>
                        <td>
                            <?php if ($doc['expiry_date']):
                                $expiry = new DateTime($doc['expiry_date']);
                                $now = new DateTime();
                                $days_to_expiry = $now->diff($expiry)->days;
                                $is_expired = $expiry < $now;
                            ?>
                                <?= $expiry->format('d M Y') ?><br>
                                <?php if ($is_expired): ?>
                                    <span class="badge bg-danger">Expired</span>
                                <?php elseif ($days_to_expiry <= 30): ?>
                                    <span class="badge bg-warning text-dark"><?= $days_to_expiry ?> days</span>
                                <?php else: ?>
                                    <span class="badge bg-success">Valid</span>
                                <?php endif; ?>
                            <?php else: ?>
                                <span class="text-muted">No expiry</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <a href="../../<?= htmlspecialchars($doc['file_path']) ?>"
                               class="btn btn-sm btn-outline-success"
                               download
                               title="Download">
                                <i class="bi bi-download"></i>
                            </a>
                            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/documents/view?id=' . $doc['id'] : 'view.php?id=' . $doc['id'] ?>"
                               class="btn btn-sm btn-outline-primary"
                               title="View Details">
                                <i class="bi bi-eye"></i>
                            </a>
                            <?php if (($_SESSION['permission_level'] ?? 0) >= 70): ?>
                            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/documents/delete?id=' . $doc['id'] : 'delete.php?id=' . $doc['id'] ?>"
                               class="btn btn-sm btn-outline-danger"
                               title="Delete"
                               onclick="return confirm('Are you sure you want to delete this document?')">
                                <i class="bi bi-trash"></i>
                            </a>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- Information Box -->
    <div class="alert alert-info mt-4">
        <h6><i class="bi bi-info-circle"></i> Document Management Tips</h6>
        <small>
            <ul class="mb-0">
                <li>Upload all worker-related documents for easy access and compliance tracking</li>
                <li>Set expiry dates for time-sensitive documents (visas, passports, certificates)</li>
                <li>System will automatically alert you of expiring documents</li>
                <li>Keep documents organized by type for quick retrieval during audits</li>
            </ul>
        </small>
    </div>
</div>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>
