<?php
/**
 * Documents Module - Delete Document
 * Role-based deletion:
 * - Permission 70-89: Can delete documents (removes file from disk)
 * - Permission 90-99: Can delete documents
 * - Permission 100+: Can PERMANENT delete with PIN
 *
 * Note: Documents are deleted from database and file system
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }
}

// Check minimum permission - delete requires level 70
if ($current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete documents.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
    exit;
}

$document_id = (int)($_POST['id'] ?? 0);
$pin = $_POST['pin'] ?? '';

if ($document_id <= 0) {
    $_SESSION['error'] = 'Invalid document ID.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
    exit;
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if document exists
    $stmt = $pdo->prepare("SELECT d.*, sw.full_name, sw.employee_number
                          FROM " . DB_PREFIX . "documents d
                          LEFT JOIN " . DB_PREFIX . "sponsored_workers sw ON d.worker_id = sw.id
                          WHERE d.id = ?");
    $stmt->execute([$document_id]);
    $document = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$document) {
        $_SESSION['error'] = 'Document not found.';
        header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
        exit;
    }

    $permission_level = $current_user['permission_level'];

    // For permission 70-99: Allow deletion with audit log
    if ($permission_level >= 70 && $permission_level < 100) {
        // Log the deletion for audit trail
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'DELETE', 'documents', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'document_name' => $document['document_name'],
            'worker_name' => $document['full_name'],
            'employee_number' => $document['employee_number'],
            'document_type' => $document['document_type'],
            'file_path' => $document['file_path'],
            'reason' => 'User deleted document'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $document_id, $details, $ip]);

        // Delete the physical file if it exists
        if (!empty($document['file_path'])) {
            $file_full_path = __DIR__ . '/../../' . $document['file_path'];
            if (file_exists($file_full_path)) {
                @unlink($file_full_path);
            }
        }

        // Delete the database record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "documents WHERE id = ?");
        $stmt->execute([$document_id]);

        $_SESSION['success'] = 'Document "' . htmlspecialchars($document['document_name']) . '" has been deleted.';
    }

    // For permission 100+: Require PIN for deletion
    elseif ($permission_level >= 100) {
        // REQUIRE PIN for admin deletion
        if (empty($pin)) {
            $_SESSION['error'] = 'PIN is required for deletion.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
            exit;
        }

        // Verify PIN
        if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
            $_SESSION['error'] = 'Invalid PIN. Deletion cancelled.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
            exit;
        }

        // Log the deletion for audit trail
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'ADMIN_DELETE', 'documents', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'document_name' => $document['document_name'],
            'worker_name' => $document['full_name'],
            'employee_number' => $document['employee_number'],
            'document_type' => $document['document_type'],
            'file_path' => $document['file_path'],
            'reason' => 'Admin deletion with PIN verification'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $document_id, $details, $ip]);

        // Delete the physical file if it exists
        if (!empty($document['file_path'])) {
            $file_full_path = __DIR__ . '/../../' . $document['file_path'];
            if (file_exists($file_full_path)) {
                @unlink($file_full_path);
            }
        }

        // Delete the database record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "documents WHERE id = ?");
        $stmt->execute([$document_id]);

        $_SESSION['success'] = 'Document "' . htmlspecialchars($document['document_name']) . '" has been PERMANENTLY DELETED.';
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting document: ' . $e->getMessage();
}

// Redirect back to list
header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/documents/list' : 'list.php'));
exit;
?>
