<?php
/**
 * CoS Module - Delete CoS Record
 * Role-based deletion:
 * - Permission 70-89: Can soft delete only
 * - Permission 90-99: Can soft delete + restore
 * - Permission 100+: Can soft delete, restore, or PERMANENT delete (requires PIN)
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }
}

// Check minimum permission - delete requires level 70
if ($current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete CoS records.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
    exit;
}

$cos_id = (int)($_POST['id'] ?? 0);
$delete_type = $_POST['delete_type'] ?? 'soft'; // 'soft' or 'permanent'
$pin = $_POST['pin'] ?? '';

if ($cos_id <= 0) {
    $_SESSION['error'] = 'Invalid CoS ID.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
    exit;
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if CoS exists and get current status
    $stmt = $pdo->prepare("SELECT id, cos_number, status FROM " . DB_PREFIX . "cos_records WHERE id = ?");
    $stmt->execute([$cos_id]);
    $cos = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$cos) {
        $_SESSION['error'] = 'CoS record not found.';
        header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
        exit;
    }

    // Determine what type of delete to perform based on role and request
    $permission_level = $current_user['permission_level'];
    $current_status = $cos['status'];

    // CASE 1: Non-admin users (70-89) - Can only soft delete active records
    if ($permission_level >= 70 && $permission_level < 90) {
        if ($current_status === 'deleted') {
            $_SESSION['error'] = 'This CoS is already deleted. Only admins can permanently remove deleted records.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
            exit;
        }

        // Soft delete
        $stmt = $pdo->prepare("
            UPDATE " . DB_PREFIX . "cos_records
            SET status = 'deleted',
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$cos_id]);
        $_SESSION['success'] = 'CoS "' . htmlspecialchars($cos['cos_number']) . '" has been deleted.';
    }

    // CASE 2: Senior users (90-99) - Can soft delete and restore
    elseif ($permission_level >= 90 && $permission_level < 100) {
        if ($current_status === 'deleted') {
            // Restore deleted CoS (set to 'assigned' status)
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "cos_records
                SET status = 'assigned',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$cos_id]);
            $_SESSION['success'] = 'CoS "' . htmlspecialchars($cos['cos_number']) . '" has been restored.';
        } else {
            // Soft delete
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "cos_records
                SET status = 'deleted',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$cos_id]);
            $_SESSION['success'] = 'CoS "' . htmlspecialchars($cos['cos_number']) . '" has been deleted.';
        }
    }

    // CASE 3: Admins (100+) - Can do everything including permanent delete
    elseif ($permission_level >= 100) {
        // If requesting permanent delete
        if ($delete_type === 'permanent') {
            // REQUIRE PIN for permanent delete
            if (empty($pin)) {
                $_SESSION['error'] = 'PIN is required for permanent deletion.';
                header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
                exit;
            }

            // Verify PIN (check against user's PIN)
            if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
                $_SESSION['error'] = 'Invalid PIN. Permanent deletion cancelled.';
                header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
                exit;
            }

            // Log the permanent deletion for audit trail
            $audit_stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "audit_log
                (user_id, action, table_name, record_id, details, ip_address, created_at)
                VALUES (?, 'PERMANENT_DELETE', 'cos_records', ?, ?, ?, NOW())
            ");
            $details = json_encode([
                'cos_number' => $cos['cos_number'],
                'reason' => 'Admin permanent deletion with PIN verification'
            ]);
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $audit_stmt->execute([$current_user['id'], $cos_id, $details, $ip]);

            // PERMANENT DELETE - Remove from database
            $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "cos_records WHERE id = ?");
            $stmt->execute([$cos_id]);

            $_SESSION['success'] = 'CoS "' . htmlspecialchars($cos['cos_number']) . '" has been PERMANENTLY DELETED from the database. This action cannot be undone.';
        } else {
            // Soft delete or restore
            if ($current_status === 'deleted') {
                // Restore
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "cos_records
                    SET status = 'assigned',
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$cos_id]);
                $_SESSION['success'] = 'CoS "' . htmlspecialchars($cos['cos_number']) . '" has been restored.';
            } else {
                // Soft delete
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "cos_records
                    SET status = 'deleted',
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$cos_id]);
                $_SESSION['success'] = 'CoS "' . htmlspecialchars($cos['cos_number']) . '" has been deleted (soft delete).';
            }
        }
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting CoS: ' . $e->getMessage();
}

// Redirect back to list
header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/cos/list' : 'list.php'));
exit;
?>
