-- Certificate of Sponsorship (CoS) Management Module
-- Database Schema for SCMS v3.0

-- Table: scms_cos_records
-- Tracks all Certificates of Sponsorship (CoS) issued by the organization
CREATE TABLE IF NOT EXISTS `scms_cos_records` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `cos_number` varchar(100) NOT NULL COMMENT 'CoS reference number from Home Office',
  `cos_type` enum('defined','undefined') DEFAULT 'defined' COMMENT 'Defined or Undefined CoS',
  `issue_date` date NOT NULL COMMENT 'Date CoS was issued by Home Office',
  `assigned_to` int(11) DEFAULT NULL COMMENT 'Worker ID this CoS is assigned to',
  `status` enum('assigned','used','withdrawn','expired') DEFAULT 'assigned' COMMENT 'Current status of the CoS',
  `cost` decimal(10,2) DEFAULT 0.00 COMMENT 'Cost of the CoS in GBP',
  `expiry_date` date DEFAULT NULL COMMENT 'Date when CoS expires if not used',
  `job_title` varchar(255) DEFAULT NULL COMMENT 'Job title on the CoS',
  `soc_code` varchar(20) DEFAULT NULL COMMENT 'Standard Occupational Classification code',
  `salary` decimal(10,2) DEFAULT NULL COMMENT 'Salary specified on the CoS',
  `start_date` date DEFAULT NULL COMMENT 'Intended start date on the CoS',
  `end_date` date DEFAULT NULL COMMENT 'Intended end date on the CoS',
  `notes` text DEFAULT NULL COMMENT 'Additional notes and comments',
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT NULL ON UPDATE current_timestamp(),
  `created_by` int(11) DEFAULT NULL COMMENT 'User ID who created the record',
  PRIMARY KEY (`id`),
  UNIQUE KEY `cos_number` (`cos_number`),
  KEY `assigned_to` (`assigned_to`),
  KEY `status` (`status`),
  KEY `expiry_date` (`expiry_date`),
  KEY `created_by` (`created_by`),
  CONSTRAINT `fk_cos_worker` FOREIGN KEY (`assigned_to`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_cos_user` FOREIGN KEY (`created_by`) REFERENCES `scms_users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Certificate of Sponsorship tracking';

-- Table: scms_cos_history
-- Audit trail of all CoS status changes and assignments
CREATE TABLE IF NOT EXISTS `scms_cos_history` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `cos_id` int(11) NOT NULL COMMENT 'Reference to cos_records',
  `action` varchar(50) NOT NULL COMMENT 'Action performed (assigned, used, withdrawn, etc.)',
  `previous_status` varchar(50) DEFAULT NULL COMMENT 'Status before change',
  `new_status` varchar(50) DEFAULT NULL COMMENT 'Status after change',
  `previous_worker_id` int(11) DEFAULT NULL COMMENT 'Previous worker if reassigned',
  `new_worker_id` int(11) DEFAULT NULL COMMENT 'New worker if reassigned',
  `reason` text DEFAULT NULL COMMENT 'Reason for the change',
  `performed_by` int(11) DEFAULT NULL COMMENT 'User who performed the action',
  `performed_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `cos_id` (`cos_id`),
  KEY `new_worker_id` (`new_worker_id`),
  KEY `performed_by` (`performed_by`),
  CONSTRAINT `fk_cos_history_cos` FOREIGN KEY (`cos_id`) REFERENCES `scms_cos_records` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_cos_history_worker` FOREIGN KEY (`new_worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_cos_history_user` FOREIGN KEY (`performed_by`) REFERENCES `scms_users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='CoS change history and audit trail';

-- Table: scms_cos_documents
-- Documents related to CoS (PDF copies, Home Office correspondence, etc.)
CREATE TABLE IF NOT EXISTS `scms_cos_documents` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `cos_id` int(11) NOT NULL COMMENT 'Reference to cos_records',
  `document_type` varchar(100) NOT NULL COMMENT 'Type of document (CoS PDF, Assignment letter, etc.)',
  `filename` varchar(255) NOT NULL COMMENT 'Original filename',
  `file_path` varchar(500) NOT NULL COMMENT 'Path to stored file',
  `file_size` int(11) DEFAULT NULL COMMENT 'File size in bytes',
  `mime_type` varchar(100) DEFAULT NULL COMMENT 'MIME type of the file',
  `uploaded_by` int(11) DEFAULT NULL COMMENT 'User who uploaded the document',
  `uploaded_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `cos_id` (`cos_id`),
  KEY `uploaded_by` (`uploaded_by`),
  CONSTRAINT `fk_cos_doc_cos` FOREIGN KEY (`cos_id`) REFERENCES `scms_cos_records` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_cos_doc_user` FOREIGN KEY (`uploaded_by`) REFERENCES `scms_users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Documents associated with CoS records';

-- Index for performance optimization
CREATE INDEX idx_cos_status_expiry ON scms_cos_records(status, expiry_date);
CREATE INDEX idx_cos_assigned_status ON scms_cos_records(assigned_to, status);
CREATE INDEX idx_cos_history_date ON scms_cos_history(performed_at);

-- Initial data: Example CoS types for dropdown reference
-- This would typically be in a settings table, but included here for reference
INSERT IGNORE INTO `scms_system_lists` (`list_type`, `list_value`, `list_label`, `sort_order`) VALUES
('cos_type', 'defined', 'Defined CoS', 1),
('cos_type', 'undefined', 'Undefined CoS', 2),
('cos_status', 'assigned', 'Assigned', 1),
('cos_status', 'used', 'Used', 2),
('cos_status', 'withdrawn', 'Withdrawn', 3),
('cos_status', 'expired', 'Expired', 4),
('cos_document_type', 'cos_pdf', 'CoS PDF from Home Office', 1),
('cos_document_type', 'assignment_letter', 'CoS Assignment Letter', 2),
('cos_document_type', 'withdrawal_notice', 'Withdrawal Notice', 3),
('cos_document_type', 'other', 'Other Document', 99);
