<?php
/**
 * Attendance Module Settings
 * Configure conflict detection, roster management, and attendance features
 */
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();
require_permission(90); // Admin only

$pdo = get_database_connection();
$page_title = "Attendance Settings";

// Handle settings save
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $settings = [
            // Conflict Management
            'enable_night_shift_conflicts' => isset($_POST['enable_night_shift_conflicts']) ? 1 : 0,
            'enable_same_day_conflicts' => isset($_POST['enable_same_day_conflicts']) ? 1 : 0,
            'auto_resolve_conflicts' => isset($_POST['auto_resolve_conflicts']) ? 1 : 0,

            // Roster Management
            'enable_roster_overrides' => isset($_POST['enable_roster_overrides']) ? 1 : 0,
            'enable_roster_leave_auth' => isset($_POST['enable_roster_leave_auth']) ? 1 : 0,
            'auto_mark_roster_leaves' => isset($_POST['auto_mark_roster_leaves']) ? 1 : 0,

            // Leave Authorization
            'require_auth_pl' => isset($_POST['require_auth_pl']) ? 1 : 0,
            'require_auth_fo' => isset($_POST['require_auth_fo']) ? 1 : 0,
            'require_auth_ho' => isset($_POST['require_auth_ho']) ? 1 : 0,
            'require_auth_ul' => isset($_POST['require_auth_ul']) ? 1 : 0,
            'require_auth_ml' => isset($_POST['require_auth_ml']) ? 1 : 0,

            // Tracking & Audit
            'enable_tracking_badges' => isset($_POST['enable_tracking_badges']) ? 1 : 0,
            'show_authorization_details' => isset($_POST['show_authorization_details']) ? 1 : 0,

            // Localization
            'date_format' => $_POST['date_format'] ?? 'DD/MM/YYYY',
            'use_british_english' => isset($_POST['use_british_english']) ? 1 : 0,
        ];

        // Save settings to database
        foreach ($settings as $key => $value) {
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "module_settings (module_name, setting_key, setting_value)
                VALUES ('attendance', ?, ?)
                ON DUPLICATE KEY UPDATE setting_value = ?");
            $stmt->execute([$key, $value, $value]);
        }

        $success_message = "Settings saved successfully";
    } catch (Exception $e) {
        $error_message = "Error saving settings: " . $e->getMessage();
    }
}

// Load current settings
$current_settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "module_settings WHERE module_name = 'attendance'");
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $current_settings[$row['setting_key']] = $row['setting_value'];
}

// Set defaults if not exists
$defaults = [
    'enable_night_shift_conflicts' => 1,
    'enable_same_day_conflicts' => 1,
    'auto_resolve_conflicts' => 0,
    'enable_roster_overrides' => 1,
    'enable_roster_leave_auth' => 1,
    'auto_mark_roster_leaves' => 1,
    'require_auth_pl' => 1,
    'require_auth_fo' => 1,
    'require_auth_ho' => 1,
    'require_auth_ul' => 1,
    'require_auth_ml' => 1,
    'enable_tracking_badges' => 1,
    'show_authorization_details' => 1,
    'date_format' => 'DD/MM/YYYY',
    'use_british_english' => 1,
];

foreach ($defaults as $key => $default) {
    if (!isset($current_settings[$key])) {
        $current_settings[$key] = $default;
    }
}
?>
<!DOCTYPE html>
<html lang="en-GB">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - SCMS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        :root {
            --primary-color: #6366f1;
        }
        .settings-section {
            background: white;
            border-radius: 8px;
            padding: 24px;
            margin-bottom: 24px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .settings-section h3 {
            color: var(--primary-color);
            font-size: 18px;
            font-weight: 600;
            margin-bottom: 16px;
            padding-bottom: 12px;
            border-bottom: 2px solid #e5e7eb;
        }
        .setting-item {
            padding: 12px 0;
            border-bottom: 1px solid #f3f4f6;
        }
        .setting-item:last-child {
            border-bottom: none;
        }
        .form-check-input:checked {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        .alert-custom {
            border-left: 4px solid var(--primary-color);
        }
    </style>
</head>
<body class="bg-light">
    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col">
                <h1 class="h3 mb-0">
                    <i class="bi bi-gear text-primary"></i> Attendance Module Settings
                </h1>
                <p class="text-muted mb-0">Configure conflict detection, roster management, and attendance features</p>
            </div>
            <div class="col-auto">
                <a href="calendar_by_location.php" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left"></i> Back to Calendar
                </a>
            </div>
        </div>

        <?php if (isset($success_message)): ?>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle"></i> <?php echo $success_message; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($error_message)): ?>
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="bi bi-exclamation-triangle"></i> <?php echo $error_message; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <form method="POST" action="">
            <div class="row">
                <div class="col-lg-6">
                    <!-- Conflict Management Section -->
                    <div class="settings-section">
                        <h3><i class="bi bi-exclamation-triangle"></i> Conflict Management</h3>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="enable_night_shift_conflicts"
                                       name="enable_night_shift_conflicts" value="1"
                                       <?php echo $current_settings['enable_night_shift_conflicts'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="enable_night_shift_conflicts">
                                    <strong>Night Shift Conflict Detection</strong>
                                    <div class="small text-muted">Prevent scheduling workers for day shifts immediately after night shifts</div>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="enable_same_day_conflicts"
                                       name="enable_same_day_conflicts" value="1"
                                       <?php echo $current_settings['enable_same_day_conflicts'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="enable_same_day_conflicts">
                                    <strong>Same-Day Multi-Location Conflicts</strong>
                                    <div class="small text-muted">Prevent scheduling workers at multiple locations on the same day</div>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="auto_resolve_conflicts"
                                       name="auto_resolve_conflicts" value="1"
                                       <?php echo $current_settings['auto_resolve_conflicts'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="auto_resolve_conflicts">
                                    <strong>Auto-Resolve Conflicts</strong>
                                    <div class="small text-muted">Automatically remove conflicting shifts without confirmation (not recommended)</div>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Roster Management Section -->
                    <div class="settings-section">
                        <h3><i class="bi bi-calendar3"></i> Roster Management</h3>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="enable_roster_overrides"
                                       name="enable_roster_overrides" value="1"
                                       <?php echo $current_settings['enable_roster_overrides'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="enable_roster_overrides">
                                    <strong>Enable Roster Overrides</strong>
                                    <div class="small text-muted">Allow adding, removing, or modifying shifts via roster management</div>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="enable_roster_leave_auth"
                                       name="enable_roster_leave_auth" value="1"
                                       <?php echo $current_settings['enable_roster_leave_auth'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="enable_roster_leave_auth">
                                    <strong>Roster Leave Authorisation</strong>
                                    <div class="small text-muted">Allow authorising leaves (PL, FO, HO, UL, ML) via roster page</div>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="auto_mark_roster_leaves"
                                       name="auto_mark_roster_leaves" value="1"
                                       <?php echo $current_settings['auto_mark_roster_leaves'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="auto_mark_roster_leaves">
                                    <strong>Auto-Mark Roster Leaves</strong>
                                    <div class="small text-muted">Automatically mark attendance when leaves are authorised via roster</div>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Tracking & Audit Section -->
                    <div class="settings-section">
                        <h3><i class="bi bi-shield-check"></i> Tracking & Audit Trail</h3>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="enable_tracking_badges"
                                       name="enable_tracking_badges" value="1"
                                       <?php echo $current_settings['enable_tracking_badges'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="enable_tracking_badges">
                                    <strong>Show Tracking Badges</strong>
                                    <div class="small text-muted">Display ROSTER/AUTH badges on calendar to show who authorised changes</div>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="show_authorization_details"
                                       name="show_authorization_details" value="1"
                                       <?php echo $current_settings['show_authorization_details'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="show_authorization_details">
                                    <strong>Show Authorisation Details</strong>
                                    <div class="small text-muted">Allow clicking badges to view detailed authorisation information</div>
                                </label>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-6">
                    <!-- Leave Authorization Section -->
                    <div class="settings-section">
                        <h3><i class="bi bi-file-text"></i> Leave Authorisation Requirements</h3>
                        <p class="small text-muted mb-3">Select which leave types require management authorisation</p>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="require_auth_pl"
                                       name="require_auth_pl" value="1"
                                       <?php echo $current_settings['require_auth_pl'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="require_auth_pl">
                                    <strong>Paid Leave (PL)</strong>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="require_auth_fo"
                                       name="require_auth_fo" value="1"
                                       <?php echo $current_settings['require_auth_fo'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="require_auth_fo">
                                    <strong>Day Off (FO)</strong>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="require_auth_ho"
                                       name="require_auth_ho" value="1"
                                       <?php echo $current_settings['require_auth_ho'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="require_auth_ho">
                                    <strong>Half Day Off (HO)</strong>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="require_auth_ul"
                                       name="require_auth_ul" value="1"
                                       <?php echo $current_settings['require_auth_ul'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="require_auth_ul">
                                    <strong>Unpaid Leave (UL)</strong>
                                </label>
                            </div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="require_auth_ml"
                                       name="require_auth_ml" value="1"
                                       <?php echo $current_settings['require_auth_ml'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="require_auth_ml">
                                    <strong>Maternity Leave (ML)</strong>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Localization Section -->
                    <div class="settings-section">
                        <h3><i class="bi bi-globe"></i> Localisation</h3>

                        <div class="setting-item">
                            <label for="date_format" class="form-label"><strong>Date Format</strong></label>
                            <select class="form-select" id="date_format" name="date_format">
                                <option value="DD/MM/YYYY" <?php echo $current_settings['date_format'] === 'DD/MM/YYYY' ? 'selected' : ''; ?>>DD/MM/YYYY (British)</option>
                                <option value="MM/DD/YYYY" <?php echo $current_settings['date_format'] === 'MM/DD/YYYY' ? 'selected' : ''; ?>>MM/DD/YYYY (American)</option>
                                <option value="YYYY-MM-DD" <?php echo $current_settings['date_format'] === 'YYYY-MM-DD' ? 'selected' : ''; ?>>YYYY-MM-DD (ISO)</option>
                            </select>
                            <div class="small text-muted mt-1">Display format for dates throughout the module</div>
                        </div>

                        <div class="setting-item">
                            <div class="form-check form-switch">
                                <input class="form-check-input" type="checkbox" id="use_british_english"
                                       name="use_british_english" value="1"
                                       <?php echo $current_settings['use_british_english'] ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="use_british_english">
                                    <strong>Use British English</strong>
                                    <div class="small text-muted">Use British spellings (authorise, colour, etc.)</div>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Information Panel -->
                    <div class="alert alert-custom alert-info">
                        <h6 class="alert-heading"><i class="bi bi-info-circle"></i> About These Settings</h6>
                        <p class="small mb-0">
                            These settings control the behaviour of the Attendance module. Changes take effect immediately.
                            Disabling conflict detection may allow scheduling errors. It's recommended to keep safety features enabled.
                        </p>
                    </div>
                </div>
            </div>

            <div class="row mt-4">
                <div class="col">
                    <div class="d-flex gap-2">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Save Settings
                        </button>
                        <a href="calendar_by_location.php" class="btn btn-outline-secondary">
                            Cancel
                        </a>
                    </div>
                </div>
            </div>
        </form>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
