<?php
/**
 * Roster Management - Calendar View with Drag-to-Select
 * Manage worker shifts with visual calendar interface
 */
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication('../../login.php');

$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

// Check permission
if (($_SESSION['permission_level'] ?? 0) < 30) {
    die('Access denied.');
}

try {
    $pdo = get_database_connection();
} catch (Exception $e) {
    error_log('Roster.php - Database connection error: ' . $e->getMessage());
    die('Database connection error. Please contact administrator.');
}

$page_title = 'Roster Management';
$current_page = 'attendance';

// Get current month/year or from URL
$current_month = isset($_GET['month']) ? intval($_GET['month']) : date('n');
$current_year = isset($_GET['year']) ? intval($_GET['year']) : date('Y');

// Calculate previous/next month
$prev_month = $current_month - 1;
$prev_year = $current_year;
if ($prev_month < 1) {
    $prev_month = 12;
    $prev_year--;
}

$next_month = $current_month + 1;
$next_year = $current_year;
if ($next_month > 12) {
    $next_month = 1;
    $next_year++;
}

// Get first and last day of month
$first_day = date('Y-m-01', strtotime("$current_year-$current_month-01"));
$last_day = date('Y-m-t', strtotime("$current_year-$current_month-01"));

// Get all active locations
try {
    $locations = $pdo->query("SELECT id, name FROM " . DB_PREFIX . "work_locations WHERE status = 'active' ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log('Roster.php - Error fetching locations: ' . $e->getMessage());
    die('Error loading locations. Please contact administrator.');
}

// Check if any locations exist
if (empty($locations)) {
    die('No active locations found. Please create at least one active location to use the roster management.');
}

// Get selected location (default to first)
$selected_location = isset($_GET['location']) ? intval($_GET['location']) : ($locations[0]['id'] ?? 0);

// Get all active workers
try {
    $workers = $pdo->query("SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' ORDER BY full_name")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log('Roster.php - Error fetching workers: ' . $e->getMessage());
    $workers = [];
}

// Get shift patterns for selected location
$shift_patterns = [];
if ($selected_location) {
    try {
        $stmt = $pdo->prepare("SELECT id, shift_name, shift_type, start_time, end_time FROM " . DB_PREFIX . "location_shift_patterns WHERE location_id = ? ORDER BY start_time");
        $stmt->execute([$selected_location]);
        $shift_patterns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log('Roster.php - Error fetching shift patterns: ' . $e->getMessage());
    }
}

// Get regular worker assignments for the location
$worker_assignments = [];
if ($selected_location) {
    try {
        $stmt = $pdo->prepare("SELECT
            wla.worker_id,
            wla.days_of_week,
            sp.id as shift_pattern_id,
            sp.shift_name,
            sp.shift_type,
            sp.start_time,
            sp.end_time
            FROM " . DB_PREFIX . "worker_location_assignments wla
            LEFT JOIN " . DB_PREFIX . "location_shift_patterns sp ON wla.shift_pattern_id = sp.id
            WHERE wla.location_id = ?
            AND (wla.end_date IS NULL OR wla.end_date >= ?)
            ORDER BY wla.worker_id");
        $stmt->execute([$selected_location, $first_day]);

        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $assignment) {
            $worker_assignments[$assignment['worker_id']][] = $assignment;
        }
    } catch (Exception $e) {
        error_log('Roster.php - Error fetching worker assignments: ' . $e->getMessage());
    }
}

// Get roster overrides for the month
$roster_overrides = [];
if ($selected_location) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "roster_overrides
            WHERE location_id = ? AND roster_date >= ? AND roster_date <= ?
            ORDER BY roster_date");
        $stmt->execute([$selected_location, $first_day, $last_day]);

        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $override) {
            $key = $override['worker_id'] . '_' . $override['roster_date'];
            $roster_overrides[$key] = $override;
        }
    } catch (Exception $e) {
        error_log('Roster.php - Error fetching roster overrides: ' . $e->getMessage());
    }
}

// Generate calendar days
$days_in_month = date('t', strtotime("$current_year-$current_month-01"));
$calendar_days = [];
for ($day = 1; $day <= $days_in_month; $day++) {
    $date = sprintf('%04d-%02d-%02d', $current_year, $current_month, $day);
    $calendar_days[] = [
        'date' => $date,
        'day' => $day,
        'day_name' => date('D', strtotime($date)),
        'is_today' => $date === date('Y-m-d'),
        'is_weekend' => in_array(date('N', strtotime($date)), [6, 7])
    ];
}

$month_name = date('F Y', strtotime("$current_year-$current_month-01"));

include '../../includes/header.php';
?>

<style>
.roster-container {
    background: #f8f9fa;
    min-height: calc(100vh - 120px);
    padding: 20px;
}

.roster-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 25px;
    border-radius: 12px;
    margin-bottom: 25px;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.location-selector {
    background: white;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.05);
}

.calendar-grid {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.08);
    overflow-x: auto;
}

.roster-table {
    width: 100%;
    border-collapse: separate;
    border-spacing: 0;
    min-width: 1400px;
}

.roster-table th {
    background: #f8f9fa;
    padding: 12px 8px;
    font-weight: 600;
    font-size: 13px;
    border-bottom: 2px solid #dee2e6;
    position: sticky;
    top: 0;
    z-index: 10;
}

.roster-table th.worker-col {
    min-width: 180px;
    text-align: left;
    background: #667eea;
    color: white;
}

.roster-table th.date-col {
    min-width: 50px;
    text-align: center;
    writing-mode: vertical-rl;
    transform: rotate(180deg);
    padding: 8px 10px;
}

.roster-table th.date-col.weekend {
    background: #ffe5e5;
}

.roster-table th.date-col.today {
    background: #fff3cd;
    font-weight: 700;
}

.roster-table td {
    border: 1px solid #e9ecef;
    padding: 0;
    height: 60px;
    position: relative;
}

.roster-table td.worker-info {
    padding: 12px;
    background: #f8f9fa;
    font-weight: 500;
}

.roster-table td.worker-info small {
    display: block;
    color: #6c757d;
    font-weight: 400;
    font-size: 11px;
}

.roster-cell {
    height: 100%;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    transition: all 0.2s;
    position: relative;
}

.roster-cell:hover {
    background: #f0f0f0;
}

.roster-cell.scheduled {
    background: #d1ecf1;
}

.roster-cell.override-add {
    background: #d4edda;
    border: 2px solid #28a745;
}

.roster-cell.override-remove {
    background: #f8d7da;
    border: 2px solid #dc3545;
}

.roster-cell.override-leave {
    background: #fff3cd;
    border: 2px solid #ffc107;
}

.roster-cell.selected {
    background: #667eea !important;
    color: white;
}

.roster-cell.weekend {
    background: #fafafa;
}

.shift-badge {
    font-size: 10px;
    padding: 2px 6px;
    border-radius: 3px;
    font-weight: 600;
    white-space: nowrap;
}

.shift-badge.regular {
    background: #cce5ff;
    color: #004085;
}

.shift-badge.override {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffc107;
}

.shift-badge.leave {
    background: #fffbea;
    color: #856404;
    border: 1px solid #ffc107;
    font-weight: 700;
}

.bulk-actions {
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: white;
    padding: 20px;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.15);
    display: none;
    z-index: 1000;
    min-width: 300px;
}

.bulk-actions.active {
    display: block;
    animation: slideUp 0.3s ease;
}

@keyframes slideUp {
    from {
        transform: translateY(20px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

.legend {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
    margin-top: 15px;
    padding: 12px;
    background: #f8f9fa;
    border-radius: 8px;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 13px;
}

.legend-box {
    width: 24px;
    height: 24px;
    border-radius: 4px;
}

/* Mobile Responsiveness */
@media (max-width: 768px) {
    .roster-container {
        padding: 10px;
    }

    .roster-header {
        padding: 15px;
    }

    .roster-header h3 {
        font-size: 18px;
    }

    .roster-header p {
        font-size: 12px;
    }

    .roster-table td {
        min-height: 70px;
        height: 70px;
    }

    .roster-cell {
        touch-action: manipulation;
        -webkit-tap-highlight-color: rgba(0, 0, 0, 0.1);
        min-height: 70px;
        padding: 5px;
    }

    .shift-badge {
        font-size: 9px;
        padding: 3px 5px;
    }

    .bulk-actions {
        bottom: 10px;
        right: 10px;
        left: 10px;
        min-width: auto;
        max-height: 70vh;
        overflow-y: auto;
    }

    .location-selector .row {
        gap: 10px;
    }

    .location-selector .col-md-4,
    .location-selector .col-md-5,
    .location-selector .col-md-3 {
        width: 100%;
    }

    .table-responsive {
        -webkit-overflow-scrolling: touch;
    }

    .calendar-grid {
        padding: 10px;
    }
}
</style>

<div class="roster-container">
    <!-- Header -->
    <div class="roster-header">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <h3 class="mb-1"><i class="bi bi-calendar-check"></i> Roster Management</h3>
                <p class="mb-0 opacity-90">Drag to select dates • Click to add/remove shifts</p>
            </div>
            <a href="calendar_by_location.php" class="btn btn-light">
                <i class="bi bi-arrow-left"></i> Back to Calendar
            </a>
        </div>
    </div>

    <!-- Location & Month Selector -->
    <div class="location-selector">
        <div class="row align-items-center">
            <div class="col-md-4">
                <label class="form-label fw-bold mb-2">Location</label>
                <select id="locationSelect" class="form-select" onchange="changeLocation()">
                    <?php foreach ($locations as $location): ?>
                        <option value="<?php echo $location['id']; ?>" <?php echo $location['id'] == $selected_location ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($location['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-5 text-center">
                <label class="form-label fw-bold mb-2">Month</label>
                <div class="btn-group">
                    <a href="?location=<?php echo $selected_location; ?>&month=<?php echo $prev_month; ?>&year=<?php echo $prev_year; ?>" class="btn btn-outline-secondary">
                        <i class="bi bi-chevron-left"></i>
                    </a>
                    <button class="btn btn-outline-primary" style="min-width: 200px;" disabled>
                        <strong><?php echo $month_name; ?></strong>
                    </button>
                    <a href="?location=<?php echo $selected_location; ?>&month=<?php echo $next_month; ?>&year=<?php echo $next_year; ?>" class="btn btn-outline-secondary">
                        <i class="bi bi-chevron-right"></i>
                    </a>
                </div>
            </div>
            <div class="col-md-3 text-end">
                <label class="form-label fw-bold mb-2">Quick Actions</label>
                <div>
                    <button type="button" class="btn btn-success btn-sm" onclick="clearSelection()">
                        <i class="bi bi-arrow-clockwise"></i> Clear Selection
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Calendar Grid -->
    <div class="calendar-grid">
        <div class="table-responsive">
            <table class="roster-table" id="rosterTable">
                <thead>
                    <tr>
                        <th class="worker-col">Worker</th>
                        <?php foreach ($calendar_days as $day): ?>
                            <th class="date-col <?php echo $day['is_weekend'] ? 'weekend' : ''; ?> <?php echo $day['is_today'] ? 'today' : ''; ?>">
                                <div><?php echo $day['day_name']; ?></div>
                                <div style="font-size: 16px; font-weight: 700; margin-top: 5px;"><?php echo $day['day']; ?></div>
                            </th>
                        <?php endforeach; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($workers as $worker): ?>
                        <?php
                        // Get regular assignments for this worker
                        $assignments = $worker_assignments[$worker['id']] ?? [];
                        ?>
                        <tr>
                            <td class="worker-info">
                                <div><?php echo htmlspecialchars($worker['full_name']); ?></div>
                                <small><?php echo htmlspecialchars($worker['employee_number']); ?></small>
                            </td>
                            <?php foreach ($calendar_days as $day): ?>
                                <?php
                                $override_key = $worker['id'] . '_' . $day['date'];
                                $override = $roster_overrides[$override_key] ?? null;

                                // Check regular schedule
                                $day_abbr = $day['day_name'];
                                $has_regular_shift = false;
                                $shift_info = '';

                                foreach ($assignments as $assignment) {
                                    $days = array_map('trim', explode(',', $assignment['days_of_week'] ?? ''));
                                    if (in_array($day_abbr, $days)) {
                                        $has_regular_shift = true;
                                        $shift_info = $assignment['shift_name'] ?? '';
                                        break;
                                    }
                                }

                                // Determine cell state
                                $cell_class = 'roster-cell';
                                $cell_content = '';
                                $is_scheduled = false;

                                if ($override) {
                                    if ($override['override_type'] === 'add') {
                                        $cell_class .= ' override-add';
                                        $cell_content = '<span class="shift-badge override">' . htmlspecialchars($override['shift_name'] ?? 'Override') . '</span>';
                                        $is_scheduled = true;
                                    } else if ($override['override_type'] === 'leave') {
                                        $cell_class .= ' override-leave';
                                        $leave_abbr = $override['absence_type'] ?? 'LEAVE';
                                        $cell_content = '<span class="shift-badge leave">' . htmlspecialchars($leave_abbr) . '</span>';
                                        $is_scheduled = false;
                                    } else {
                                        $cell_class .= ' override-remove';
                                        $cell_content = '<i class="bi bi-x-circle text-danger"></i>';
                                        $is_scheduled = false;
                                    }
                                } else if ($has_regular_shift) {
                                    $cell_class .= ' scheduled';
                                    $cell_content = '<span class="shift-badge regular">' . htmlspecialchars(substr($shift_info, 0, 3)) . '</span>';
                                    $is_scheduled = true;
                                }

                                if ($day['is_weekend'] && !$is_scheduled) {
                                    $cell_class .= ' weekend';
                                }
                                ?>
                                <td>
                                    <div class="<?php echo $cell_class; ?>"
                                         data-worker-id="<?php echo $worker['id']; ?>"
                                         data-worker-name="<?php echo htmlspecialchars($worker['full_name']); ?>"
                                         data-date="<?php echo $day['date']; ?>"
                                         data-has-regular="<?php echo $has_regular_shift ? '1' : '0'; ?>"
                                         data-has-override="<?php echo $override ? '1' : '0'; ?>"
                                         data-is-scheduled="<?php echo $is_scheduled ? '1' : '0'; ?>"
                                         onmousedown="startSelection(this, event)"
                                         onmouseover="continueSelection(this)"
                                         onmouseup="endSelection()">
                                        <?php echo $cell_content; ?>
                                    </div>
                                </td>
                            <?php endforeach; ?>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Legend -->
        <div class="legend">
            <div class="legend-item">
                <div class="legend-box" style="background: #d1ecf1;"></div>
                <span>Regular Shift</span>
            </div>
            <div class="legend-item">
                <div class="legend-box" style="background: #d4edda; border: 2px solid #28a745;"></div>
                <span>Override - Added</span>
            </div>
            <div class="legend-item">
                <div class="legend-box" style="background: #fff3cd; border: 2px solid #ffc107;"></div>
                <span>Override - Leave</span>
            </div>
            <div class="legend-item">
                <div class="legend-box" style="background: #f8d7da; border: 2px solid #dc3545;"></div>
                <span>Override - Removed</span>
            </div>
            <div class="legend-item">
                <div class="legend-box" style="background: #667eea;"></div>
                <span>Selected</span>
            </div>
        </div>
    </div>

    <!-- Bulk Actions Panel -->
    <div id="bulkActionsPanel" class="bulk-actions">
        <h6 class="mb-3"><i class="bi bi-lightning-fill text-warning"></i> Bulk Actions</h6>
        <p class="small text-muted mb-3"><span id="selectedCount">0</span> dates selected</p>

        <div class="mb-3">
            <label class="form-label fw-bold">Action</label>
            <select id="bulkAction" class="form-select form-select-sm" onchange="handleActionChange()">
                <option value="">-- Select Action --</option>
                <option value="add">Add to Shift</option>
                <option value="remove">Remove from Shift</option>
                <option value="leave">Authorize Leave</option>
                <option value="delete">Delete Override (Reinstate Original)</option>
            </select>
        </div>

        <div id="shiftSelectionPanel" style="display: none;" class="mb-3">
            <label class="form-label fw-bold">Shift</label>
            <select id="bulkShiftPattern" class="form-select form-select-sm">
                <option value="">-- Select Shift --</option>
                <?php foreach ($shift_patterns as $pattern): ?>
                    <option value="<?php echo $pattern['id']; ?>"
                            data-name="<?php echo htmlspecialchars($pattern['shift_name']); ?>"
                            data-start="<?php echo $pattern['start_time']; ?>"
                            data-end="<?php echo $pattern['end_time']; ?>">
                        <?php echo htmlspecialchars($pattern['shift_name']); ?>
                        (<?php echo date('H:i', strtotime($pattern['start_time'])); ?> - <?php echo date('H:i', strtotime($pattern['end_time'])); ?>)
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <!-- Leave Selection Panel -->
        <div id="leaveSelectionPanel" style="display: none;">
            <div class="mb-3">
                <label class="form-label fw-bold">Leave Type</label>
                <select id="bulkLeaveType" class="form-select form-select-sm" onchange="handleLeaveTypeChange()">
                    <option value="">-- Select Leave Type --</option>
                    <option value="PL">Paid Leave (PL)</option>
                    <option value="FO">Day Off (FO)</option>
                    <option value="HO">Half Day Off (HO)</option>
                    <option value="UL">Unpaid Leave (UL)</option>
                    <option value="ML">Maternity Leave (ML)</option>
                </select>
            </div>

            <div id="halfDayPanel" style="display: none;" class="mb-3">
                <label class="form-label fw-bold">Half-Day Period</label>
                <select id="bulkHalfDayPeriod" class="form-select form-select-sm" onchange="handleHalfDayPeriodChange()">
                    <option value="">-- Select Period --</option>
                    <option value="first_half">First Half of Day</option>
                    <option value="second_half">Second Half of Day</option>
                    <option value="custom">Custom Time Range</option>
                </select>
            </div>

            <div id="customTimePanel" style="display: none;" class="mb-3">
                <div class="row">
                    <div class="col-6">
                        <label class="form-label small">From</label>
                        <input type="time" id="bulkTimeFrom" class="form-control form-control-sm">
                    </div>
                    <div class="col-6">
                        <label class="form-label small">To</label>
                        <input type="time" id="bulkTimeTo" class="form-control form-control-sm">
                    </div>
                </div>
            </div>
        </div>

        <div class="mb-3">
            <label class="form-label fw-bold">Notes / Reason</label>
            <textarea id="bulkNotes" class="form-control form-control-sm" rows="2" placeholder="Reason for override..."></textarea>
        </div>

        <?php if ($_SESSION['permission_level'] >= 90): ?>
            <div class="mb-3">
                <label class="form-label fw-bold">Authorize As</label>
                <select id="bulkAuthorizeAs" class="form-select form-select-sm">
                    <option value="<?php echo $_SESSION['user_id']; ?>">Myself (<?php echo htmlspecialchars($_SESSION['username']); ?>)</option>
                    <?php
                    $users_stmt = $pdo->query("SELECT id, username, full_name FROM " . DB_PREFIX . "users WHERE permission_level >= 30 AND id != " . $_SESSION['user_id'] . " ORDER BY username");
                    foreach ($users_stmt->fetchAll(PDO::FETCH_ASSOC) as $user):
                    ?>
                        <option value="<?php echo $user['id']; ?>">
                            <?php echo htmlspecialchars($user['username']); ?>
                            <?php echo $user['full_name'] ? ' (' . htmlspecialchars($user['full_name']) . ')' : ''; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        <?php endif; ?>

        <div class="d-grid gap-2">
            <button type="button" class="btn btn-primary btn-sm" onclick="applyBulkAction()">
                <i class="bi bi-check-circle"></i> Apply to Selected Dates
            </button>
            <button type="button" class="btn btn-outline-secondary btn-sm" onclick="clearSelection()">
                Cancel
            </button>
        </div>
    </div>

    <!-- Conflict Resolution Modal -->
    <div class="modal fade" id="conflictModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header bg-warning">
                    <h5 class="modal-title"><i class="bi bi-exclamation-triangle"></i> Schedule Conflict Detected</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <strong>Warning:</strong> The following worker(s) are already scheduled at other locations on the selected dates.
                    </div>

                    <div id="conflictList" class="mb-3">
                        <!-- Conflicts will be populated here -->
                    </div>

                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> <strong>What would you like to do?</strong>
                        <p class="mb-0 mt-2">Click "Remove & Add Here" to automatically remove them from the other location(s) and add them to this shift.</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        Cancel
                    </button>
                    <button type="button" class="btn btn-warning" onclick="resolveConflicts()">
                        <i class="bi bi-arrow-left-right"></i> Remove & Add Here
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let isSelecting = false;
let selectedCells = new Set();
let currentWorker = null;

// Change location
function changeLocation() {
    const locationId = document.getElementById('locationSelect').value;
    window.location = '?location=' + locationId + '&month=<?php echo $current_month; ?>&year=<?php echo $current_year; ?>';
}

// Start drag selection
function startSelection(cell, event) {
    event.preventDefault();
    isSelecting = true;
    currentWorker = cell.dataset.workerId;
    selectedCells.clear();

    // Clear all previous selections
    document.querySelectorAll('.roster-cell.selected').forEach(c => {
        c.classList.remove('selected');
    });

    // Add first cell
    cell.classList.add('selected');
    selectedCells.add(cell);
    updateBulkPanel();
}

// Continue selection (drag)
function continueSelection(cell) {
    if (!isSelecting) return;

    // Only select cells from same worker
    if (cell.dataset.workerId !== currentWorker) return;

    if (!cell.classList.contains('selected')) {
        cell.classList.add('selected');
        selectedCells.add(cell);
        updateBulkPanel();
    }
}

// End selection
function endSelection() {
    isSelecting = false;
}

// Global mouse up listener
document.addEventListener('mouseup', function() {
    endSelection();
});

// Global touch end listener
document.addEventListener('touchend', function() {
    endSelection();
}, { passive: true });

// Update bulk actions panel
function updateBulkPanel() {
    const panel = document.getElementById('bulkActionsPanel');
    const count = selectedCells.size;

    document.getElementById('selectedCount').textContent = count;

    if (count > 0) {
        panel.classList.add('active');
    } else {
        panel.classList.remove('active');
    }
}

// Clear selection
function clearSelection() {
    selectedCells.clear();
    document.querySelectorAll('.roster-cell.selected').forEach(cell => {
        cell.classList.remove('selected');
    });
    document.getElementById('bulkActionsPanel').classList.remove('active');
    document.getElementById('bulkAction').value = '';
    document.getElementById('shiftSelectionPanel').style.display = 'none';
}

// Handle action change
function handleActionChange() {
    const action = document.getElementById('bulkAction').value;
    const shiftPanel = document.getElementById('shiftSelectionPanel');
    const leavePanel = document.getElementById('leaveSelectionPanel');

    // Hide all panels first
    shiftPanel.style.display = 'none';
    leavePanel.style.display = 'none';

    // Show relevant panel
    if (action === 'add') {
        shiftPanel.style.display = 'block';
    } else if (action === 'leave') {
        leavePanel.style.display = 'block';
    }
}

// Handle leave type change
function handleLeaveTypeChange() {
    const leaveType = document.getElementById('bulkLeaveType').value;
    const halfDayPanel = document.getElementById('halfDayPanel');

    if (leaveType === 'HO') {
        halfDayPanel.style.display = 'block';
    } else {
        halfDayPanel.style.display = 'none';
        document.getElementById('customTimePanel').style.display = 'none';
    }
}

// Handle half-day period change
function handleHalfDayPeriodChange() {
    const period = document.getElementById('bulkHalfDayPeriod').value;
    const customTimePanel = document.getElementById('customTimePanel');

    if (period === 'custom') {
        customTimePanel.style.display = 'block';
    } else {
        customTimePanel.style.display = 'none';
    }
}

// Store current bulk action data for conflict resolution
let currentBulkData = null;
let conflictModalInstance = null;

// Apply bulk action
async function applyBulkAction(resolveConflicts = false) {
    const action = document.getElementById('bulkAction').value;
    const notes = document.getElementById('bulkNotes').value;
    const authorizeAs = document.getElementById('bulkAuthorizeAs')?.value || <?php echo $_SESSION['user_id']; ?>;

    if (!action) {
        alert('Please select an action');
        return;
    }

    if (selectedCells.size === 0) {
        alert('Please select dates');
        return;
    }

    let shiftPatternId = null;
    let shiftName = null;
    let startTime = null;
    let endTime = null;
    let absenceType = null;
    let halfDayPeriod = null;
    let timeFrom = null;
    let timeTo = null;

    if (action === 'delete') {
        // For delete action, no additional data needed - just delete the override
        // Check if selected cells actually have overrides
        let hasOverride = false;
        selectedCells.forEach(cell => {
            if (cell.dataset.hasOverride === '1') {
                hasOverride = true;
            }
        });

        if (!hasOverride) {
            alert('No roster overrides found in selected dates to delete');
            return;
        }
    } else if (action === 'add') {
        const shiftSelect = document.getElementById('bulkShiftPattern');
        shiftPatternId = shiftSelect.value;

        if (!shiftPatternId) {
            alert('Please select a shift');
            return;
        }

        const selectedOption = shiftSelect.options[shiftSelect.selectedIndex];
        shiftName = selectedOption.dataset.name;
        startTime = selectedOption.dataset.start;
        endTime = selectedOption.dataset.end;
    } else if (action === 'leave') {
        absenceType = document.getElementById('bulkLeaveType').value;

        if (!absenceType) {
            alert('Please select a leave type');
            return;
        }

        if (absenceType === 'HO') {
            halfDayPeriod = document.getElementById('bulkHalfDayPeriod').value;
            if (!halfDayPeriod) {
                alert('Please select half-day period for Half Day Off');
                return;
            }

            if (halfDayPeriod === 'custom') {
                timeFrom = document.getElementById('bulkTimeFrom').value;
                timeTo = document.getElementById('bulkTimeTo').value;
                if (!timeFrom || !timeTo) {
                    alert('Please specify time range for custom half-day');
                    return;
                }
            }
        }

        if (!notes) {
            alert('Please provide a reason for the leave');
            return;
        }
    }

    // Collect all selected dates
    const overrides = [];
    selectedCells.forEach(cell => {
        overrides.push({
            worker_id: cell.dataset.workerId,
            worker_name: cell.dataset.workerName,
            date: cell.dataset.date
        });
    });

    // Store data for conflict resolution
    currentBulkData = {
        location_id: <?php echo $selected_location; ?>,
        action: action,
        overrides: overrides,
        shift_pattern_id: shiftPatternId,
        shift_name: shiftName,
        start_time: startTime,
        end_time: endTime,
        absence_type: absenceType,
        half_day_period: halfDayPeriod,
        time_from: timeFrom,
        time_to: timeTo,
        notes: notes,
        authorized_by: authorizeAs,
        resolve_conflicts: resolveConflicts
    };

    // Show loading indicator
    const applyBtn = document.querySelector('#bulkActionsPanel .btn-primary');
    const originalText = applyBtn?.innerHTML || 'Apply to Selected Dates';
    if (applyBtn) {
        applyBtn.disabled = true;
        applyBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Processing...';
    }

    try {
        const response = await fetch('ajax/bulk_roster_override.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(currentBulkData)
        });

        const result = await response.json();

        if (result.success) {
            // Show success message with debug info
            alert(result.message || 'Success');
            location.reload();
        } else if (result.has_conflicts) {
            // Show conflict modal
            showConflictModal(result.conflicts);
            if (applyBtn) {
                applyBtn.disabled = false;
                applyBtn.innerHTML = originalText;
            }
        } else {
            alert('Error: ' + (result.message || 'Unknown error'));
            if (applyBtn) {
                applyBtn.disabled = false;
                applyBtn.innerHTML = originalText;
            }
        }
    } catch (error) {
        console.error('Error:', error);
        alert('Error applying bulk action: ' + error.message);
        if (applyBtn) {
            applyBtn.disabled = false;
            applyBtn.innerHTML = originalText;
        }
    }
}

// Show conflict modal with details
function showConflictModal(conflicts) {
    const conflictList = document.getElementById('conflictList');
    conflictList.innerHTML = '';

    const table = document.createElement('table');
    table.className = 'table table-sm table-bordered';
    table.innerHTML = `
        <thead class="table-light">
            <tr>
                <th>Worker</th>
                <th>Date</th>
                <th>Currently At</th>
                <th>Shift</th>
            </tr>
        </thead>
        <tbody>
        </tbody>
    `;

    const tbody = table.querySelector('tbody');
    conflicts.forEach(conflict => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td><strong>${conflict.worker_name}</strong></td>
            <td>${new Date(conflict.date).toLocaleDateString('en-GB')}</td>
            <td><span class="badge bg-danger">${conflict.conflict_location_name}</span></td>
            <td>${conflict.conflict_shift_name || '-'}</td>
        `;
        tbody.appendChild(row);
    });

    conflictList.appendChild(table);

    // Show modal
    conflictModalInstance = new bootstrap.Modal(document.getElementById('conflictModal'));
    conflictModalInstance.show();
}

// Resolve conflicts and apply action
async function resolveConflicts() {
    if (!currentBulkData) {
        alert('No pending action');
        return;
    }

    // Hide conflict modal
    if (conflictModalInstance) {
        conflictModalInstance.hide();
    }

    // Re-apply with resolve_conflicts = true
    await applyBulkAction(true);
}

// Prevent text selection while dragging
document.getElementById('rosterTable').addEventListener('selectstart', function(e) {
    if (isSelecting) {
        e.preventDefault();
    }
});

// Mobile touch support for swipe-to-select
function initMobileTouchSupport() {
    const rosterCells = document.querySelectorAll('.roster-cell');

    rosterCells.forEach(cell => {
        // Touch start - begin selection
        cell.addEventListener('touchstart', function(e) {
            e.preventDefault();
            startSelection(this, e.touches[0]);
        }, { passive: false });

        // Touch move - continue selection (swipe)
        cell.addEventListener('touchmove', function(e) {
            e.preventDefault();
            // Get the element at touch position
            const touch = e.touches[0];
            const element = document.elementFromPoint(touch.clientX, touch.clientY);

            // Find the roster-cell element (might be touching a child element)
            const targetCell = element?.closest('.roster-cell');
            if (targetCell) {
                continueSelection(targetCell);
            }
        }, { passive: false });

        // Touch end - end selection
        cell.addEventListener('touchend', function(e) {
            e.preventDefault();
            endSelection();
        }, { passive: false });
    });
}

// Initialize mobile touch support when page loads
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(initMobileTouchSupport, 100);
});
</script>

<?php include '../../includes/footer.php'; ?>
