<?php
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

// Check permission level
if (($_SESSION['permission_level'] ?? 0) < 50) {
    die('Access denied. You do not have permission to view attendance reports.');
}

$page_title = 'Attendance Report Generator';

// Fetch company details from system settings
$settings_stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings");
$settings = [];
while ($row = $settings_stmt->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Company details from settings
$company_name = $settings['company_name'] ?? "Company Name";
$company_reg = $settings['company_reg'] ?? '';
$company_address = $settings['address'] ?? '';
$company_city = $settings['city'] ?? '';
$company_postcode = $settings['postcode'] ?? '';
$company_phone = $settings['phone'] ?? '';
$company_email = $settings['email'] ?? '';
$company_logo = $settings['company_logo'] ?? ''; // Logo path if exists

// Check if generating report
$generate_report = isset($_POST['generate_report']) || isset($_GET['generate']);

if ($generate_report) {
    // Get form data
    $date_from = $_POST['date_from'] ?? $_GET['date_from'] ?? date('Y-m-01');
    $date_to = $_POST['date_to'] ?? $_GET['date_to'] ?? date('Y-m-t');
    $selected_workers = $_POST['workers'] ?? $_GET['workers'] ?? [];

    // Convert to array if single value
    if (!is_array($selected_workers)) {
        $selected_workers = [$selected_workers];
    }

    // Validate dates
    $date_from = date('Y-m-d', strtotime($date_from));
    $date_to = date('Y-m-d', strtotime($date_to));

    // Get selected workers
    if (empty($selected_workers)) {
        $workers = $pdo->query("SELECT id, employee_number, full_name, department
                               FROM " . DB_PREFIX . "sponsored_workers
                               WHERE status = 'active'
                               ORDER BY employee_number")->fetchAll(PDO::FETCH_ASSOC);
    } else {
        $placeholders = str_repeat('?,', count($selected_workers) - 1) . '?';
        $stmt = $pdo->prepare("SELECT id, employee_number, full_name, department
                              FROM " . DB_PREFIX . "sponsored_workers
                              WHERE id IN ($placeholders) AND status = 'active'
                              ORDER BY employee_number");
        $stmt->execute($selected_workers);
        $workers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get attendance data for date range
    $attendance_stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "attendance_records
                                      WHERE attendance_date >= ? AND attendance_date <= ?");
    $attendance_stmt->execute([$date_from, $date_to]);
    $attendance_data = [];
    foreach ($attendance_stmt->fetchAll(PDO::FETCH_ASSOC) as $record) {
        $key = $record['worker_id'] . '_' . $record['attendance_date'];
        $attendance_data[$key] = $record;
    }

    // Calculate statistics
    $period_start = new DateTime($date_from);
    $period_end = new DateTime($date_to);
    $period_end->modify('+1 day');
    $interval = new DateInterval('P1D');
    $date_range = new DatePeriod($period_start, $interval, $period_end);

    $worker_stats = [];
    $all_dates = [];

    foreach ($date_range as $date) {
        $all_dates[] = $date->format('Y-m-d');
    }

    foreach ($workers as $worker) {
        $stats = [
            'P' => 0, 'A' => 0, 'PL' => 0, 'UL' => 0, 'FO' => 0, 'HO' => 0, 'ML' => 0, 'WE' => 0,
            'total_days' => count($all_dates),
            'working_days' => 0,
            'present_days' => 0,
            'attendance_rate' => 0,
            'details' => []
        ];

        foreach ($all_dates as $date) {
            $key = $worker['id'] . '_' . $date;
            $day_of_week = date('N', strtotime($date));

            // Count working days (Mon-Fri)
            if ($day_of_week <= 5) {
                $stats['working_days']++;
            }

            if (isset($attendance_data[$key])) {
                $record = $attendance_data[$key];
                $status = $record['status'];
                if (isset($stats[$status])) {
                    $stats[$status]++;
                }
                if ($status === 'P') {
                    $stats['present_days']++;
                }
                $stats['details'][$date] = $status;
            } else {
                $stats['details'][$date] = '-';
            }
        }

        if ($stats['working_days'] > 0) {
            $stats['attendance_rate'] = round(($stats['present_days'] / $stats['working_days']) * 100, 1);
        }

        $worker_stats[$worker['id']] = $stats;
    }
}

// If not generating report, show the filter form
if (!$generate_report) {
    include '../../includes/header.php';

    // Get all active workers for the checkbox list
    $all_workers = $pdo->query("SELECT id, employee_number, full_name, department
                                FROM " . DB_PREFIX . "sponsored_workers
                                WHERE status = 'active'
                                ORDER BY employee_number")->fetchAll(PDO::FETCH_ASSOC);
    ?>

    <div class="dashboard-container">
        <div class="row justify-content-center">
            <div class="col-lg-10">
                <div class="modern-card">
                    <div class="card-header-modern">
                        <div class="d-flex justify-content-between align-items-center w-100">
                            <h5 class="card-title-modern mb-0">
                                <i class="bi bi-file-earmark-bar-graph"></i> Attendance Report Generator
                            </h5>
                            <div class="ms-auto">
                                <a href="calendar.php" class="btn btn-sm btn-outline-secondary">
                                    <i class="bi bi-arrow-left"></i> Back to Calendar
                                </a>
                            </div>
                        </div>
                    </div>
                    <div class="card-body-modern">
                        <form method="POST" action="" id="reportForm">
                            <input type="hidden" name="generate_report" value="1">

                            <div class="row mb-4">
                                <div class="col-md-6">
                                    <label class="form-label">From Date <span class="text-danger">*</span></label>
                                    <input type="date" name="date_from" class="form-control" value="<?php echo date('Y-m-01'); ?>" required>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">To Date <span class="text-danger">*</span></label>
                                    <input type="date" name="date_to" class="form-control" value="<?php echo date('Y-m-t'); ?>" required>
                                </div>
                            </div>

                            <div class="mb-4">
                                <label class="form-label">Quick Date Ranges</label>
                                <div class="btn-group btn-group-sm w-100" role="group">
                                    <button type="button" class="btn btn-outline-primary" onclick="setDateRange('today')">Today</button>
                                    <button type="button" class="btn btn-outline-primary" onclick="setDateRange('yesterday')">Yesterday</button>
                                    <button type="button" class="btn btn-outline-primary" onclick="setDateRange('this_week')">This Week</button>
                                    <button type="button" class="btn btn-outline-primary" onclick="setDateRange('last_week')">Last Week</button>
                                    <button type="button" class="btn btn-outline-primary" onclick="setDateRange('this_month')">This Month</button>
                                    <button type="button" class="btn btn-outline-primary" onclick="setDateRange('last_month')">Last Month</button>
                                </div>
                            </div>

                            <div class="mb-4">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <label class="form-label mb-0">Select Employees</label>
                                    <div>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="selectAll()">Select All</button>
                                        <button type="button" class="btn btn-sm btn-outline-secondary" onclick="deselectAll()">Deselect All</button>
                                    </div>
                                </div>
                                <div class="border rounded p-3" style="max-height: 300px; overflow-y: auto;">
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="checkbox" id="all_workers" onchange="toggleAll(this)">
                                        <label class="form-check-label fw-bold" for="all_workers">
                                            All Employees (<?php echo count($all_workers); ?>)
                                        </label>
                                    </div>
                                    <hr>
                                    <?php foreach ($all_workers as $worker): ?>
                                    <div class="form-check mb-2">
                                        <input class="form-check-input worker-checkbox" type="checkbox" name="workers[]" value="<?php echo $worker['id']; ?>" id="worker_<?php echo $worker['id']; ?>">
                                        <label class="form-check-label" for="worker_<?php echo $worker['id']; ?>">
                                            <strong><?php echo htmlspecialchars($worker['employee_number']); ?></strong> -
                                            <?php echo htmlspecialchars($worker['full_name']); ?>
                                            <?php if ($worker['department']): ?>
                                                <span class="text-muted">(<?php echo htmlspecialchars($worker['department']); ?>)</span>
                                            <?php endif; ?>
                                        </label>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                                <small class="text-muted">Leave unchecked to include all employees</small>
                            </div>

                            <hr>

                            <div class="d-flex justify-content-between">
                                <a href="calendar.php" class="btn btn-secondary">Cancel</a>
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-file-earmark-bar-graph"></i> Generate Report
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
    function setDateRange(range) {
        const fromInput = document.querySelector('input[name="date_from"]');
        const toInput = document.querySelector('input[name="date_to"]');
        const today = new Date();

        let from, to;

        switch(range) {
            case 'today':
                from = to = new Date();
                break;
            case 'yesterday':
                from = to = new Date(today.setDate(today.getDate() - 1));
                break;
            case 'this_week':
                from = new Date(today.setDate(today.getDate() - today.getDay() + 1));
                to = new Date(today.setDate(today.getDate() - today.getDay() + 7));
                break;
            case 'last_week':
                from = new Date(today.setDate(today.getDate() - today.getDay() - 6));
                to = new Date(today.setDate(today.getDate() - today.getDay()));
                break;
            case 'this_month':
                from = new Date(today.getFullYear(), today.getMonth(), 1);
                to = new Date(today.getFullYear(), today.getMonth() + 1, 0);
                break;
            case 'last_month':
                from = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                to = new Date(today.getFullYear(), today.getMonth(), 0);
                break;
        }

        fromInput.value = from.toISOString().split('T')[0];
        toInput.value = to.toISOString().split('T')[0];
    }

    function toggleAll(checkbox) {
        const checkboxes = document.querySelectorAll('.worker-checkbox');
        checkboxes.forEach(cb => cb.checked = checkbox.checked);
    }

    function selectAll() {
        document.getElementById('all_workers').checked = true;
        toggleAll(document.getElementById('all_workers'));
    }

    function deselectAll() {
        document.getElementById('all_workers').checked = false;
        toggleAll(document.getElementById('all_workers'));
    }

    // Update "All Employees" checkbox based on individual checkboxes
    document.querySelectorAll('.worker-checkbox').forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const total = document.querySelectorAll('.worker-checkbox').length;
            const checked = document.querySelectorAll('.worker-checkbox:checked').length;
            document.getElementById('all_workers').checked = (total === checked);
        });
    });
    </script>

    <?php
    include '../../includes/footer.php';
    exit;
}

// Generate report HTML view
include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="modern-card">
        <div class="card-header-modern">
            <div class="d-flex justify-content-between align-items-center w-100">
                <h5 class="card-title-modern mb-0">
                    <i class="bi bi-file-earmark-bar-graph"></i> Attendance Report
                </h5>
                <div class="ms-auto d-flex gap-2">
                    <a href="report_monthly.php" class="btn btn-sm btn-outline-secondary">
                        <i class="bi bi-arrow-left"></i> New Report
                    </a>
                </div>
            </div>
        </div>
        <div class="card-body-modern">
            <!-- Report Header -->
            <div class="d-flex justify-content-between align-items-center mb-4 pb-3 border-bottom border-3">
                <div class="flex-shrink-0 text-center" style="width: 200px;">
                    <?php if ($company_logo && file_exists("../../" . $company_logo)): ?>
                        <img src="../../<?php echo htmlspecialchars($company_logo); ?>" alt="Logo" style="max-width: 180px; max-height: 120px;">
                    <?php endif; ?>
                </div>
                <div class="flex-grow-1 ps-4">
                    <h4 class="fw-semibold mb-2 text-uppercase"><?php echo htmlspecialchars($company_name); ?></h4>
                    <?php if ($company_reg): ?>
                        <p class="mb-2 text-muted small">Company Registration: <?php echo htmlspecialchars($company_reg); ?></p>
                    <?php endif; ?>
                    <p class="mb-0 text-muted small" style="line-height: 1.8;">
                        <?php if ($company_address): ?><?php echo htmlspecialchars($company_address); ?><br><?php endif; ?>
                        <?php if ($company_city || $company_postcode): ?>
                            <?php echo htmlspecialchars($company_city . ' ' . $company_postcode); ?><br>
                        <?php endif; ?>
                        <?php if ($company_phone): ?>Tel: <?php echo htmlspecialchars($company_phone); ?> <?php endif; ?>
                        <?php if ($company_email): ?>| Email: <?php echo htmlspecialchars($company_email); ?><?php endif; ?>
                    </p>
                </div>
            </div>

            <h5 class="mb-3">Attendance Report</h5>

            <!-- Report Info -->
            <div class="alert alert-light border-start border-primary border-4 mb-4">
                <div class="row small">
                    <div class="col-md-6">
                        <p class="mb-1"><strong>Report Period:</strong> <?php echo date('d F Y', strtotime($date_from)); ?> to <?php echo date('d F Y', strtotime($date_to)); ?></p>
                        <p class="mb-0"><strong>Total Days:</strong> <?php echo count($all_dates); ?> days</p>
                    </div>
                    <div class="col-md-6">
                        <p class="mb-1"><strong>Employees:</strong> <?php echo count($workers); ?></p>
                        <p class="mb-0"><strong>Generated:</strong> <?php echo date('d/m/Y H:i'); ?> by <?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username'] ?? 'System'); ?></p>
                    </div>
                </div>
            </div>

            <!-- Statistics Summary -->
            <div class="row mb-4 g-3">
                <?php
                $total_present = 0;
                $total_absent = 0;
                $avg_attendance = 0;
                foreach ($worker_stats as $stats) {
                    $total_present += $stats['P'];
                    $total_absent += $stats['A'];
                    $avg_attendance += $stats['attendance_rate'];
                }
                $avg_attendance = count($worker_stats) > 0 ? round($avg_attendance / count($worker_stats), 1) : 0;
                ?>
                <div class="col-md-4">
                    <div class="border rounded p-3 text-center">
                        <h3 class="mb-1 text-success"><?php echo $total_present; ?></h3>
                        <small class="text-muted">Total Present Days</small>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="border rounded p-3 text-center">
                        <h3 class="mb-1 text-danger"><?php echo $total_absent; ?></h3>
                        <small class="text-muted">Total Absent Days</small>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="border rounded p-3 text-center">
                        <h3 class="mb-1 text-primary"><?php echo $avg_attendance; ?>%</h3>
                        <small class="text-muted">Average Attendance</small>
                    </div>
                </div>
            </div>

            <!-- Detailed Table -->
            <div class="table-responsive">
                <table class="table table-bordered table-hover">
                    <thead class="table-light">
                        <tr>
                            <th>Emp #</th>
                            <th>Name</th>
                            <th>Department</th>
                            <th class="text-center">Present</th>
                            <th class="text-center">Absent</th>
                            <th class="text-center">Paid Leave</th>
                            <th class="text-center">Unpaid Leave</th>
                            <th class="text-center">Full Off</th>
                            <th class="text-center">Half Off</th>
                            <th class="text-center">Maternity</th>
                            <th class="text-center">Weekend</th>
                            <th class="text-center">Attendance %</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($workers as $worker):
                            $stats = $worker_stats[$worker['id']];
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($worker['employee_number']); ?></td>
                            <td><strong><?php echo htmlspecialchars($worker['full_name']); ?></strong></td>
                            <td><?php echo htmlspecialchars($worker['department'] ?? '-'); ?></td>
                            <td class="text-center"><?php echo $stats['P']; ?></td>
                            <td class="text-center"><?php echo $stats['A']; ?></td>
                            <td class="text-center"><?php echo $stats['PL']; ?></td>
                            <td class="text-center"><?php echo $stats['UL']; ?></td>
                            <td class="text-center"><?php echo $stats['FO']; ?></td>
                            <td class="text-center"><?php echo $stats['HO']; ?></td>
                            <td class="text-center"><?php echo $stats['ML']; ?></td>
                            <td class="text-center"><?php echo $stats['WE']; ?></td>
                            <td class="text-center">
                                <strong class="<?php echo $stats['attendance_rate'] < 80 ? 'text-danger' : 'text-success'; ?>">
                                    <?php echo $stats['attendance_rate']; ?>%
                                </strong>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

            <div class="text-center text-muted mt-4 pt-3 border-top">
                <small>
                    Status Codes: P = Present | A = Absent | PL = Paid Leave | UL = Unpaid Leave | FO = Full Day Off | HO = Half Day Off | ML = Maternity Leave | WE = Weekend
                </small>
            </div>
        </div>
    </div>
</div>

<style>
@media print {
    .card-header-modern,
    .btn,
    nav,
    footer,
    .sidebar {
        display: none !important;
    }

    body {
        font-size: 9pt;
    }

    .table {
        font-size: 8pt;
    }

    .table thead {
        background: #34495e !important;
        color: white !important;
    }
}
</style>

<?php include '../../includes/footer.php'; ?>
