<?php
$page_title = 'Mark Attendance';
require_once '../../includes/header.php';
require_permission(60);

$selected_date = $_GET['date'] ?? date('Y-m-d');
$success = '';
$errors = [];

// Get all active workers
$stmt = $pdo->prepare("SELECT sw.*, 
       CONCAT(sw.first_name, ' ', sw.last_name) as full_name,
       ar.id as attendance_id, ar.status, ar.notes
       FROM sponsored_workers sw
       LEFT JOIN attendance_records ar ON sw.id = ar.worker_id AND ar.attendance_date = ?
       WHERE sw.is_active = 1 AND sw.business_profile_id = ?
       ORDER BY sw.department, sw.last_name, sw.first_name");
$stmt->execute([$selected_date, $_SESSION['business_profile_id']]);
$workers = $stmt->fetchAll();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        foreach ($_POST['attendance'] as $worker_id => $data) {
            if (empty($data['status'])) continue;
            
            // Check if record exists
            $stmt = $pdo->prepare("SELECT id FROM attendance_records 
                                  WHERE worker_id = ? AND attendance_date = ?");
            $stmt->execute([$worker_id, $selected_date]);
            $existing = $stmt->fetch();
            
            if ($existing) {
                // Update
                $stmt = $pdo->prepare("UPDATE attendance_records SET 
                                      status = ?, notes = ?, marked_by = ?
                                      WHERE id = ?");
                $stmt->execute([
                    $data['status'],
                    $data['notes'] ?? null,
                    $_SESSION['user_id'],
                    $existing['id']
                ]);
            } else {
                // Insert
                $stmt = $pdo->prepare("INSERT INTO attendance_records 
                                      (business_profile_id, worker_id, attendance_date, 
                                      status, notes, marked_by) 
                                      VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $_SESSION['business_profile_id'],
                    $worker_id,
                    $selected_date,
                    $data['status'],
                    $data['notes'] ?? null,
                    $_SESSION['user_id']
                ]);
            }
        }
        
        $pdo->commit();
        log_action($pdo, 'Attendance Marked', 'attendance_records', null);
        header('Location: dashboard.php?date=' . $selected_date . '&success=Attendance marked successfully');
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $errors[] = "Error: " . $e->getMessage();
    }
}
?>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger">
        <?php foreach ($errors as $error): ?>
            <div><?= htmlspecialchars($error) ?></div>
        <?php endforeach; ?>
    </div>
<?php endif; ?>

<div class="stat-card mb-3">
    <div class="d-flex justify-content-between align-items-center">
        <div>
            <h5><i class="bi bi-calendar-check"></i> Mark Attendance: <?= format_date($selected_date) ?></h5>
            <p class="text-muted mb-0">Use status codes: P (Present), PL (Paid Leave), UL (Unpaid), FO (Full Off), HO (Half Off)</p>
        </div>
        <div>
            <input type="date" class="form-control" value="<?= $selected_date ?>" 
                   onchange="window.location='?date='+this.value">
        </div>
    </div>
</div>

<!-- Quick Fill Buttons -->
<div class="stat-card mb-3">
    <h6>Quick Fill</h6>
    <div class="btn-group">
        <button type="button" class="btn btn-sm btn-success" onclick="fillAll('P')">
            All Present (P)
        </button>
        <button type="button" class="btn btn-sm btn-info" onclick="fillAll('PL')">
            All Paid Leave (PL)
        </button>
        <button type="button" class="btn btn-sm btn-secondary" onclick="fillAll('FO')">
            All Full Off (FO)
        </button>
        <button type="button" class="btn btn-sm btn-danger" onclick="clearAll()">
            Clear All
        </button>
    </div>
</div>

<div class="stat-card">
    <form method="POST">
        <div class="table-responsive">
            <table class="table table-sm">
                <thead>
                    <tr>
                        <th width="5%">#</th>
                        <th width="20%">Worker Name</th>
                        <th width="15%">Department</th>
                        <th width="15%">Status</th>
                        <th width="45%">Notes (Optional)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php $count = 1; ?>
                    <?php foreach ($workers as $worker): ?>
                    <tr>
                        <td><?= $count++ ?></td>
                        <td>
                            <strong><?= htmlspecialchars($worker['full_name']) ?></strong><br>
                            <small class="text-muted"><?= htmlspecialchars($worker['employee_number']) ?></small>
                        </td>
                        <td><?= htmlspecialchars($worker['department']) ?></td>
                        <td>
                            <select name="attendance[<?= $worker['id'] ?>][status]" 
                                    class="form-select form-select-sm status-select">
                                <option value="">-</option>
                                <option value="P" <?= $worker['status'] === 'P' ? 'selected' : '' ?>>P - Present</option>
                                <option value="PL" <?= $worker['status'] === 'PL' ? 'selected' : '' ?>>PL - Paid Leave</option>
                                <option value="UL" <?= $worker['status'] === 'UL' ? 'selected' : '' ?>>UL - Unpaid Leave</option>
                                <option value="FO" <?= $worker['status'] === 'FO' ? 'selected' : '' ?>>FO - Full Day Off</option>
                                <option value="HO" <?= $worker['status'] === 'HO' ? 'selected' : '' ?>>HO - Half Day Off</option>
                            </select>
                        </td>
                        <td>
                            <input type="text" name="attendance[<?= $worker['id'] ?>][notes]" 
                                   class="form-control form-control-sm" 
                                   placeholder="Optional notes..."
                                   value="<?= htmlspecialchars($worker['notes'] ?? '') ?>">
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <hr>
        
        <div class="d-flex justify-content-between">
            <a href="dashboard.php?date=<?= $selected_date ?>" class="btn btn-secondary">
                <i class="bi bi-x-circle"></i> Cancel
            </a>
            <button type="submit" class="btn btn-success btn-lg">
                <i class="bi bi-save"></i> Save Attendance
            </button>
        </div>
    </form>
</div>

<script>
function fillAll(status) {
    document.querySelectorAll('.status-select').forEach(select => {
        select.value = status;
    });
}

function clearAll() {
    if (confirm('Clear all attendance selections?')) {
        document.querySelectorAll('.status-select').forEach(select => {
            select.value = '';
        });
    }
}
</script>

<?php require_once '../../includes/footer.php'; ?>
