<?php
/**
 * Attendance Module - Delete Attendance Record
 * Role-based deletion:
 * - Permission 70-89: Can delete attendance records
 * - Permission 90-99: Can delete attendance records
 * - Permission 100+: Can PERMANENT delete (requires PIN)
 *
 * Note: Attendance records don't have soft delete - they are either present or deleted
 * For audit trail, deletion is logged before removing
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }
}

// Check minimum permission - delete requires level 70
if ($current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete attendance records.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
    exit;
}

$record_id = (int)($_POST['id'] ?? 0);
$pin = $_POST['pin'] ?? '';

if ($record_id <= 0) {
    $_SESSION['error'] = 'Invalid attendance record ID.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
    exit;
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if record exists
    $stmt = $pdo->prepare("SELECT ar.*, sw.full_name, sw.employee_number
                          FROM " . DB_PREFIX . "attendance_records ar
                          LEFT JOIN " . DB_PREFIX . "sponsored_workers sw ON ar.worker_id = sw.id
                          WHERE ar.id = ?");
    $stmt->execute([$record_id]);
    $record = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$record) {
        $_SESSION['error'] = 'Attendance record not found.';
        header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
        exit;
    }

    $permission_level = $current_user['permission_level'];

    // For permission 70-99: Allow deletion with audit log
    if ($permission_level >= 70 && $permission_level < 100) {
        // Log the deletion for audit trail
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'DELETE', 'attendance_records', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'worker_name' => $record['full_name'],
            'employee_number' => $record['employee_number'],
            'date' => $record['date'],
            'status' => $record['status'],
            'hours' => $record['hours'],
            'reason' => 'User deleted attendance record'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $record_id, $details, $ip]);

        // Delete the record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "attendance_records WHERE id = ?");
        $stmt->execute([$record_id]);

        $_SESSION['success'] = 'Attendance record for ' . htmlspecialchars($record['full_name']) . ' on ' . date('d/m/Y', strtotime($record['date'])) . ' has been deleted.';
    }

    // For permission 100+: Require PIN for deletion
    elseif ($permission_level >= 100) {
        // REQUIRE PIN for admin deletion
        if (empty($pin)) {
            $_SESSION['error'] = 'PIN is required for deletion.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
            exit;
        }

        // Verify PIN
        if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
            $_SESSION['error'] = 'Invalid PIN. Deletion cancelled.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
            exit;
        }

        // Log the deletion for audit trail
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'ADMIN_DELETE', 'attendance_records', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'worker_name' => $record['full_name'],
            'employee_number' => $record['employee_number'],
            'date' => $record['date'],
            'status' => $record['status'],
            'hours' => $record['hours'],
            'reason' => 'Admin deletion with PIN verification'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $record_id, $details, $ip]);

        // Delete the record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "attendance_records WHERE id = ?");
        $stmt->execute([$record_id]);

        $_SESSION['success'] = 'Attendance record for ' . htmlspecialchars($record['full_name']) . ' on ' . date('d/m/Y', strtotime($record['date'])) . ' has been DELETED.';
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting attendance record: ' . $e->getMessage();
}

// Redirect back to list
header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'));
exit;
?>
