<?php
/**
 * Cleanup Script - Remove attendance records for deleted workers
 * This ensures statistics only count active workers
 */
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

// Check permission - admin only
if (($_SESSION['permission_level'] ?? 0) < 90) {
    die('Access denied. Admin permission required.');
}

$page_title = 'Cleanup Deleted Workers Attendance';
include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern mb-0">
                        <i class="bi bi-trash"></i> Cleanup Attendance Records
                    </h5>
                </div>
                <div class="card-body-modern">
                    <?php
                    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm'])) {
                        // Get all deleted workers with attendance records
                        $stmt = $pdo->query("
                            SELECT w.id, w.employee_number, w.full_name, w.status,
                                   COUNT(a.id) as attendance_count
                            FROM " . DB_PREFIX . "sponsored_workers w
                            LEFT JOIN " . DB_PREFIX . "attendance_records a ON w.id = a.worker_id
                            WHERE w.status = 'deleted'
                            GROUP BY w.id
                            HAVING attendance_count > 0
                        ");

                        $deleted_workers = $stmt->fetchAll(PDO::FETCH_ASSOC);

                        if (empty($deleted_workers)) {
                            echo '<div class="alert alert-info">No attendance records found for deleted workers.</div>';
                        } else {
                            $total_deleted = 0;

                            echo '<div class="alert alert-warning"><strong>Deleting attendance records for deleted workers...</strong></div>';
                            echo '<table class="table table-sm">';
                            echo '<thead><tr><th>Employee #</th><th>Name</th><th>Records Deleted</th></tr></thead>';
                            echo '<tbody>';

                            foreach ($deleted_workers as $worker) {
                                $delete_stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ?");
                                $delete_stmt->execute([$worker['id']]);
                                $deleted_count = $delete_stmt->rowCount();
                                $total_deleted += $deleted_count;

                                echo '<tr>';
                                echo '<td>' . htmlspecialchars($worker['employee_number']) . '</td>';
                                echo '<td>' . htmlspecialchars($worker['full_name']) . '</td>';
                                echo '<td><span class="badge bg-danger">' . $deleted_count . ' records deleted</span></td>';
                                echo '</tr>';
                            }

                            echo '</tbody></table>';
                            echo '<div class="alert alert-success mt-3">';
                            echo "<strong>✓ Cleanup Complete!</strong><br>";
                            echo "Total attendance records deleted: <strong>$total_deleted</strong><br>";
                            echo "Deleted workers cleaned: <strong>" . count($deleted_workers) . "</strong>";
                            echo '</div>';
                        }
                    } else {
                        // Show preview - what will be deleted
                        $stmt = $pdo->query("
                            SELECT w.id, w.employee_number, w.full_name, w.status, w.department,
                                   COUNT(a.id) as attendance_count
                            FROM " . DB_PREFIX . "sponsored_workers w
                            LEFT JOIN " . DB_PREFIX . "attendance_records a ON w.id = a.worker_id
                            WHERE w.status = 'deleted'
                            GROUP BY w.id
                            HAVING attendance_count > 0
                        ");

                        $deleted_workers = $stmt->fetchAll(PDO::FETCH_ASSOC);

                        if (empty($deleted_workers)) {
                            echo '<div class="alert alert-success">';
                            echo '<i class="bi bi-check-circle"></i> <strong>All Clean!</strong><br>';
                            echo 'No attendance records found for deleted workers.';
                            echo '</div>';
                        } else {
                            echo '<div class="alert alert-warning">';
                            echo '<i class="bi bi-exclamation-triangle"></i> <strong>Warning!</strong><br>';
                            echo 'Found ' . count($deleted_workers) . ' deleted worker(s) with attendance records.<br>';
                            echo 'These old records may be causing incorrect statistics.';
                            echo '</div>';

                            echo '<h6>Preview - Records to be deleted:</h6>';
                            echo '<table class="table table-bordered">';
                            echo '<thead class="table-light">';
                            echo '<tr><th>Employee #</th><th>Name</th><th>Department</th><th>Status</th><th>Attendance Records</th></tr>';
                            echo '</thead><tbody>';

                            $total_records = 0;
                            foreach ($deleted_workers as $worker) {
                                $total_records += $worker['attendance_count'];
                                echo '<tr>';
                                echo '<td>' . htmlspecialchars($worker['employee_number']) . '</td>';
                                echo '<td>' . htmlspecialchars($worker['full_name']) . '</td>';
                                echo '<td>' . htmlspecialchars($worker['department']) . '</td>';
                                echo '<td><span class="badge bg-danger">' . $worker['status'] . '</span></td>';
                                echo '<td><span class="badge bg-warning">' . $worker['attendance_count'] . ' records</span></td>';
                                echo '</tr>';
                            }

                            echo '</tbody></table>';

                            echo '<div class="alert alert-info mt-3">';
                            echo "<strong>Total attendance records to be deleted: $total_records</strong>";
                            echo '</div>';

                            echo '<form method="POST">';
                            echo '<input type="hidden" name="confirm" value="1">';
                            echo '<div class="d-flex justify-content-between align-items-center">';
                            echo '<a href="../attendance/calendar.php" class="btn btn-secondary">Cancel</a>';
                            echo '<button type="submit" class="btn btn-danger" onclick="return confirm(\'Are you sure you want to delete ' . $total_records . ' attendance records for deleted workers? This cannot be undone.\');">';
                            echo '<i class="bi bi-trash"></i> Delete All Attendance Records for Deleted Workers';
                            echo '</button>';
                            echo '</div>';
                            echo '</form>';
                        }
                    }
                    ?>

                    <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm'])): ?>
                    <hr>
                    <div class="d-flex justify-content-center mt-3">
                        <a href="../attendance/calendar.php" class="btn btn-success">
                            <i class="bi bi-calendar"></i> Back to Attendance Calendar
                        </a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include '../../includes/footer.php'; ?>
