<?php
/**
 * Attendance Calendar - By Location View
 * Shows attendance organised by location/site with shifts
 */
// Temporary debug
error_reporting(E_ALL);
ini_set('display_errors', 1);

if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }

    $page_title = 'Attendance by Location';
    include '../../includes/header.php';
}

// Check permission
if (($_SESSION['permission_level'] ?? 0) < 30) {
    die('Access denied.');
}

$pdo = get_database_connection();

// Debug: Output locations count
// echo "<!-- DEBUG: Starting calendar_by_location.php -->";

// Get view type from URL (day, week, month) - default to week
$view_type = isset($_GET['view']) ? $_GET['view'] : 'week';
if (!in_array($view_type, ['day', 'week', 'month'])) {
    $view_type = 'week'; // Fallback to weekly view
}

// Get date parameter
$current_date = isset($_GET['date']) ? $_GET['date'] : date('Y-m-d');

// Calculate date ranges based on view type
if ($view_type === 'day') {
    // Daily view - show single day
    $week_start = $current_date;
    $week_end = $current_date;
    $prev_week = date('Y-m-d', strtotime($current_date . ' -1 day'));
    $next_week = date('Y-m-d', strtotime($current_date . ' +1 day'));
    $this_week = date('Y-m-d');
    $date_range_label = date('D, d M Y', strtotime($current_date));
} elseif ($view_type === 'month') {
    // Monthly view - show full month
    $week_start = date('Y-m-01', strtotime($current_date)); // First day of month
    $week_end = date('Y-m-t', strtotime($current_date)); // Last day of month
    $prev_week = date('Y-m-01', strtotime($week_start . ' -1 month'));
    $next_week = date('Y-m-01', strtotime($week_start . ' +1 month'));
    $this_week = date('Y-m-01');
    $date_range_label = date('F Y', strtotime($current_date));
} else {
    // Weekly view (default) - show 7 days
    $week_start = date('Y-m-d', strtotime('monday this week', strtotime($current_date)));
    $week_end = date('Y-m-d', strtotime($week_start . ' +6 days'));
    $prev_week = date('Y-m-d', strtotime($week_start . ' -7 days'));
    $next_week = date('Y-m-d', strtotime($week_start . ' +7 days'));
    $this_week = date('Y-m-d', strtotime('monday this week'));
    $date_range_label = date('d M', strtotime($week_start)) . ' - ' . date('d M Y', strtotime($week_end));
}

// Get locations filter - support multiple locations
$selected_locations = [];
if (isset($_GET['locations']) && !empty($_GET['locations'])) {
    $selected_locations = explode(',', $_GET['locations']);
    $selected_locations = array_map('intval', $selected_locations); // Sanitize
}

// Get all locations for filter dropdown
$all_locations = $pdo->query("SELECT id, name FROM " . DB_PREFIX . "work_locations WHERE status = 'active' ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

// Debug output
// echo "<!-- DEBUG: Found " . count($all_locations) . " locations -->";

// Fetch locations to display
$locations_sql = "SELECT id, name, type, address, city FROM " . DB_PREFIX . "work_locations WHERE status = 'active'";
if (!empty($selected_locations)) {
    $placeholders = implode(',', array_fill(0, count($selected_locations), '?'));
    $locations_sql .= " AND id IN ($placeholders)";
}
$locations_sql .= " ORDER BY name";

if (!empty($selected_locations)) {
    $stmt = $pdo->prepare($locations_sql);
    $stmt->execute($selected_locations);
    $locations = $stmt->fetchAll(PDO::FETCH_ASSOC);
} else {
    $locations = $pdo->query($locations_sql)->fetchAll(PDO::FETCH_ASSOC);
}

// Get days based on view type
$days = [];
$start_timestamp = strtotime($week_start);
$end_timestamp = strtotime($week_end);

// Calculate number of days in the range
$num_days = ($end_timestamp - $start_timestamp) / (60 * 60 * 24) + 1;

for ($i = 0; $i < $num_days; $i++) {
    $date = date('Y-m-d', strtotime($week_start . " +$i days"));
    $days[] = [
        'date' => $date,
        'day_name' => date('D', strtotime($date)),
        'day_num' => date('j', strtotime($date)),
        'is_today' => $date === date('Y-m-d')
    ];
}

// Fetch worker assignments grouped by location (one column per worker)
// Each worker can have multiple shift assignments, but only ONE shift applies per day
$assignments_data = [];
$worker_shift_map = []; // Maps worker_id => location_id => day_abbr => shift details

foreach ($locations as $location) {
    // Get all worker assignments with their shifts for this location
    $stmt = $pdo->prepare("SELECT
        wla.id as assignment_id,
        wla.days_of_week,
        w.id as worker_id,
        w.full_name,
        w.employee_number,
        w.job_title,
        sp.id as shift_pattern_id,
        sp.shift_name,
        sp.shift_type,
        sp.start_time,
        sp.end_time
        FROM " . DB_PREFIX . "worker_location_assignments wla
        INNER JOIN " . DB_PREFIX . "sponsored_workers w ON wla.worker_id = w.id
        LEFT JOIN " . DB_PREFIX . "location_shift_patterns sp ON wla.shift_pattern_id = sp.id
        WHERE wla.location_id = ?
            AND w.status = 'active'
            AND (wla.end_date IS NULL OR wla.end_date >= ?)
        ORDER BY w.full_name, sp.start_time");
    $stmt->execute([$location['id'], $week_start]);
    $all_assignments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Build worker list (unique workers only) and shift map
    $workers_list = [];
    $seen_workers = [];

    foreach ($all_assignments as $assignment) {
        $worker_id = $assignment['worker_id'];

        // Add worker to list only once
        if (!isset($seen_workers[$worker_id])) {
            $workers_list[] = [
                'worker_id' => $worker_id,
                'full_name' => $assignment['full_name'],
                'employee_number' => $assignment['employee_number'],
                'job_title' => $assignment['job_title']
            ];
            $seen_workers[$worker_id] = true;
        }

        // Map shift to specific days
        $days_of_week = array_map('trim', explode(',', $assignment['days_of_week'] ?? ''));
        foreach ($days_of_week as $day) {
            // Store shift details for this worker, location, and specific day
            $worker_shift_map[$worker_id][$location['id']][$day] = [
                'shift_name' => $assignment['shift_name'],
                'shift_type' => $assignment['shift_type'],
                'start_time' => $assignment['start_time'],
                'end_time' => $assignment['end_time'],
                'days_of_week' => $assignment['days_of_week']
            ];
        }
    }

    $assignments_data[$location['id']] = $workers_list;
}

// Fetch attendance for the week
$attendance_stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "attendance_records WHERE attendance_date >= ? AND attendance_date <= ?");
$attendance_stmt->execute([$week_start, $week_end]);
$attendance_data = [];
foreach ($attendance_stmt->fetchAll(PDO::FETCH_ASSOC) as $record) {
    $key = $record['worker_id'] . '_' . $record['attendance_date'];
    $attendance_data[$key] = $record;
}

// Fetch roster overrides for the date range
// Roster overrides take precedence over regular shift assignments
$roster_overrides = [];
$roster_stmt = $pdo->prepare("SELECT ro.*, u.username as authorised_by_name
    FROM " . DB_PREFIX . "roster_overrides ro
    LEFT JOIN " . DB_PREFIX . "users u ON ro.authorized_by = u.id
    WHERE ro.roster_date >= ? AND ro.roster_date <= ?");
$roster_stmt->execute([$week_start, $week_end]);
foreach ($roster_stmt->fetchAll(PDO::FETCH_ASSOC) as $override) {
    $key = $override['worker_id'] . '_' . $override['location_id'] . '_' . $override['roster_date'];
    $roster_overrides[$key] = $override;
}

// Fetch absence authorisations for the date range
$absence_authorisations = [];
$auth_stmt = $pdo->prepare("SELECT aa.*, u.username as authorised_by_name
    FROM " . DB_PREFIX . "absence_authorizations aa
    LEFT JOIN " . DB_PREFIX . "users u ON aa.authorized_by = u.id
    WHERE aa.attendance_date >= ? AND aa.attendance_date <= ?");
$auth_stmt->execute([$week_start, $week_end]);
foreach ($auth_stmt->fetchAll(PDO::FETCH_ASSOC) as $auth) {
    $key = $auth['worker_id'] . '_' . $auth['attendance_date'];
    $absence_authorisations[$key] = $auth;
}
?>

<div class="d-flex justify-content-between align-items-center mb-3">
    <div>
        <h4><i class="bi bi-geo-alt-fill"></i> Attendance by Location</h4>
        <small class="text-muted"><?php echo $date_range_label; ?></small>
    </div>
    <div class="btn-group btn-group-sm">
        <?php
        $base_path = defined('MODULE_EMBEDDED') ? '/sys/scms/modules/attendance/' : '';
        $month = date('m');
        $year = date('Y');
        ?>
        <a href="<?php echo $base_path; ?>bulk_mark.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-success" title="Bulk mark attendance">
            <i class="bi bi-check-all"></i> Bulk
        </a>
        <a href="<?php echo $base_path; ?>roster.php" class="btn btn-primary" title="Manage shift roster overrides">
            <i class="bi bi-calendar-check"></i> Roster
        </a>
        <a href="<?php echo $base_path; ?>report_monthly.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-info" title="View reports">
            <i class="bi bi-file-earmark-text"></i> Reports
        </a>
        <a href="<?php echo $base_path; ?>settings.php" class="btn btn-secondary" title="Module settings">
            <i class="bi bi-gear"></i> Settings
        </a>
    </div>
</div>

<!-- Filters and Navigation -->
<div class="filter-section">
    <div class="row align-items-center">
        <div class="col-md-4">
            <label class="form-label mb-2 fw-bold" style="font-size: 13px;">
                <i class="bi bi-funnel"></i> Filter Locations:
            </label>
            <div class="dropdown location-dropdown" id="locationDropdown">
                    <button class="btn btn-outline-primary dropdown-toggle location-selector-btn w-100 text-start" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                        <span id="selectedLocationsText">
                            <?php
                            if (empty($selected_locations)) {
                                echo 'All Locations';
                            } else {
                                $selected_names = array_filter($all_locations, function($loc) use ($selected_locations) {
                                    return in_array($loc['id'], $selected_locations);
                                });
                                echo count($selected_names) . ' selected';
                            }
                            ?>
                        </span>
                    </button>
                    <ul class="dropdown-menu p-2" style="max-height: 300px; overflow-y: auto; min-width: 250px;">
                        <li class="px-2 mb-2">
                            <button type="button" class="btn btn-sm btn-outline-primary w-100" onclick="selectAllLocations()">Select All</button>
                        </li>
                        <li class="px-2 mb-2">
                            <button type="button" class="btn btn-sm btn-outline-secondary w-100" onclick="clearAllLocations()">Clear All</button>
                        </li>
                        <li><hr class="dropdown-divider"></li>
                        <?php foreach ($all_locations as $loc): ?>
                        <li class="px-2">
                            <div class="form-check">
                                <input class="form-check-input location-checkbox" type="checkbox" value="<?php echo $loc['id']; ?>"
                                    id="loc_<?php echo $loc['id']; ?>"
                                    <?php echo in_array($loc['id'], $selected_locations) ? 'checked' : ''; ?>
                                    onchange="updateLocationFilter()">
                                <label class="form-check-label" for="loc_<?php echo $loc['id']; ?>">
                                    <?php echo htmlspecialchars($loc['name']); ?>
                                </label>
                            </div>
                        </li>
                        <?php endforeach; ?>
                        <li><hr class="dropdown-divider"></li>
                        <li class="px-2">
                            <button type="button" class="btn btn-sm btn-primary w-100" onclick="applyLocationFilter()">
                                <i class="bi bi-check-circle"></i> Apply
                            </button>
                        </li>
                    </ul>
                </div>
            </div>
            <div class="col-md-5 text-center">
                <div class="d-flex align-items-center justify-content-center gap-2">
                    <div class="btn-group btn-group-sm">
                        <a href="?view=<?php echo $view_type; ?>&date=<?php echo $prev_week; ?><?php echo !empty($selected_locations) ? '&locations='.implode(',', $selected_locations) : ''; ?>" class="btn btn-outline-secondary">
                            <i class="bi bi-chevron-left"></i> <?php
                                echo $view_type === 'day' ? 'Prev Day' : ($view_type === 'month' ? 'Prev Month' : 'Prev Week');
                            ?>
                        </a>
                        <a href="?view=<?php echo $view_type; ?>&date=<?php echo $this_week; ?><?php echo !empty($selected_locations) ? '&locations='.implode(',', $selected_locations) : ''; ?>" class="btn btn-outline-primary">
                            <?php
                                echo $view_type === 'day' ? 'Today' : ($view_type === 'month' ? 'This Month' : 'This Week');
                            ?>
                        </a>
                        <a href="?view=<?php echo $view_type; ?>&date=<?php echo $next_week; ?><?php echo !empty($selected_locations) ? '&locations='.implode(',', $selected_locations) : ''; ?>" class="btn btn-outline-secondary">
                            <?php
                                echo $view_type === 'day' ? 'Next Day' : ($view_type === 'month' ? 'Next Month' : 'Next Week');
                            ?> <i class="bi bi-chevron-right"></i>
                        </a>
                    </div>

                    <!-- View Selector Dropdown -->
                    <div class="dropdown">
                        <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" id="viewSelector" data-bs-toggle="dropdown" aria-expanded="false" title="Change view">
                            <i class="bi bi-calendar3"></i>
                        </button>
                        <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="viewSelector">
                            <li><h6 class="dropdown-header"><i class="bi bi-calendar3"></i> Select View</h6></li>
                            <li><a class="dropdown-item <?php echo $view_type === 'day' ? 'active' : ''; ?>" href="#" onclick="changeView('day'); return false;">
                                <i class="bi bi-calendar-day"></i> Daily View
                            </a></li>
                            <li><a class="dropdown-item <?php echo $view_type === 'week' ? 'active' : ''; ?>" href="#" onclick="changeView('week'); return false;">
                                <i class="bi bi-calendar-week"></i> Weekly View (Default)
                            </a></li>
                            <li><a class="dropdown-item <?php echo $view_type === 'month' ? 'active' : ''; ?>" href="#" onclick="changeView('month'); return false;">
                                <i class="bi bi-calendar-month"></i> Monthly View
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item text-primary" href="#" onclick="resetToDefault(); return false;">
                                <i class="bi bi-arrow-clockwise"></i> Reset to Default (Weekly)
                            </a></li>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="col-md-3 text-end">
                <!-- Space for future actions -->
            </div>
        </div>
    </div>
</div>

<!-- Legend -->
<div class="legend-card">
        <div class="d-flex justify-content-center gap-3 flex-wrap">
            <span><span class="badge bg-success">P</span> Present</span>
            <span><span class="badge bg-danger">A</span> Absent</span>
            <span><span class="badge bg-info">PL</span> Paid Leave</span>
            <span><span class="badge bg-warning">UL</span> Unpaid Leave</span>
            <span><span class="badge bg-primary">FO</span> Full Day Off</span>
            <span><span class="badge bg-secondary">HO</span> Half Day Off</span>
            <span><span class="badge bg-pink">ML</span> Maternity Leave</span>
            <span><span class="badge bg-secondary">-</span> Not Marked</span>
        </div>
</div>

<!-- Location-based View -->
<?php if (empty($locations)): ?>
    <div class="card location-card">
        <div class="card-body empty-state">
            <i class="bi bi-geo-alt"></i>
            <h5 class="mt-3">No Locations Found</h5>
            <p>Add work locations to start tracking location-based attendance.</p>
        </div>
    </div>
<?php else: ?>
    <?php foreach ($locations as $location): ?>
    <div class="card location-card">
        <div class="location-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h5 class="mb-0">
                        <i class="bi bi-geo-alt-fill"></i> <?php echo htmlspecialchars($location['name']); ?>
                    </h5>
                    <small><?php echo htmlspecialchars($location['address'] . ', ' . $location['city']); ?></small>
                </div>
                <?php if ($location['type']): ?>
                <span class="badge bg-light text-dark"><?php echo ucfirst($location['type']); ?></span>
                <?php endif; ?>
            </div>
        </div>
        <div class="card-body p-0">
            <?php
            $workers = $assignments_data[$location['id']] ?? [];
            if (empty($workers)):
            ?>
                <div class="p-4 text-center text-muted">
                    <i class="bi bi-people"></i> No workers assigned to this location
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table attendance-table table-bordered">
                        <thead>
                            <tr>
                                <th class="day-cell-header resizable-column">
                                    DAY
                                    <div class="column-resizer"></div>
                                </th>
                                <?php foreach ($workers as $worker): ?>
                                <?php
                                // Determine job title display
                                $job_title = $worker['job_title'] ?? 'Staff';
                                ?>
                                <th class="worker-column-header resizable-column" data-worker-id="<?php echo $worker['worker_id']; ?>">
                                    <div class="position-title"><?php echo htmlspecialchars($job_title); ?></div>
                                    <div class="worker-name-header"><?php echo htmlspecialchars($worker['full_name']); ?></div>
                                    <div class="column-resizer"></div>
                                </th>
                                <?php endforeach; ?>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($days as $day): ?>
                            <tr>
                                <td class="day-cell <?php echo $day['is_today'] ? 'today-highlight' : ''; ?>">
                                    <div class="day-name"><?php echo $day['day_name']; ?></div>
                                    <div class="day-date"><?php echo date('d M', strtotime($day['date'])); ?></div>
                                </td>
                                <?php foreach ($workers as $worker): ?>
                                <?php
                                // Get shift for this specific worker, location, and day
                                $day_abbr = $day['day_name'];
                                $worker_id = $worker['worker_id'];
                                $location_id = $location['id'];

                                // Check for roster override first (takes precedence)
                                $override_key = $worker_id . '_' . $location_id . '_' . $day['date'];
                                $roster_override = $roster_overrides[$override_key] ?? null;

                                if ($roster_override) {
                                    // Roster override exists at THIS location
                                    if ($roster_override['override_type'] === 'remove') {
                                        // Worker is removed from this shift - don't show
                                        $works_this_day = false;
                                        $shift_data = null;
                                    } else {
                                        // Worker is added to this shift
                                        $works_this_day = true;
                                        $shift_data = [
                                            'shift_name' => $roster_override['shift_name'] ?? 'Roster Override',
                                            'shift_type' => null,
                                            'start_time' => $roster_override['start_time'],
                                            'end_time' => $roster_override['end_time'],
                                            'is_override' => true
                                        ];
                                    }
                                } else {
                                    // No override at this location - check if worker is overridden to ANOTHER location
                                    $is_at_other_location = false;
                                    foreach ($roster_overrides as $other_override) {
                                        if ($other_override['worker_id'] == $worker_id &&
                                            $other_override['roster_date'] == $day['date'] &&
                                            $other_override['location_id'] != $location_id &&
                                            $other_override['override_type'] === 'add') {
                                            // Worker is overridden to a different location - don't show here
                                            $is_at_other_location = true;
                                            break;
                                        }
                                    }

                                    if ($is_at_other_location) {
                                        // Worker is working at another location via override
                                        $works_this_day = false;
                                        $shift_data = null;
                                    } else {
                                        // No override anywhere - check regular shift assignment
                                        $shift_data = $worker_shift_map[$worker_id][$location_id][$day_abbr] ?? null;
                                        $works_this_day = ($shift_data !== null);
                                        if ($shift_data) {
                                            $shift_data['is_override'] = false;
                                        }
                                    }
                                }

                                // Get attendance record
                                $key = $worker_id . '_' . $day['date'];
                                $record = $attendance_data[$key] ?? null;
                                $status = $record ? $record['status'] : '-';

                                $badge_colors = [
                                    'P' => 'success',
                                    'A' => 'danger',
                                    'PL' => 'info',
                                    'UL' => 'warning',
                                    'FO' => 'primary',
                                    'HO' => 'secondary',
                                    'ML' => 'pink'
                                ];
                                $badge_class = $badge_colors[$status] ?? 'light text-dark border';

                                // Get shift info for this specific day
                                if ($shift_data) {
                                    $shift_name = $shift_data['shift_name'] ?? '';
                                    $shift_type = $shift_data['shift_type'] ?? '';
                                    $start_time = $shift_data['start_time'] ?? '';
                                    $end_time = $shift_data['end_time'] ?? '';
                                    $is_override = $shift_data['is_override'] ?? false;
                                } else {
                                    $shift_name = '';
                                    $shift_type = '';
                                    $start_time = '';
                                    $end_time = '';
                                    $is_override = false;
                                }
                                ?>
                                <td class="attendance-cell <?php echo $day['is_today'] ? 'today-highlight' : ''; ?> <?php echo !$works_this_day ? 'not-scheduled' : ''; ?>"
                                    <?php if ($works_this_day): ?>
                                    data-worker-id="<?php echo $worker['worker_id']; ?>"
                                    data-worker-name="<?php echo htmlspecialchars($worker['full_name']); ?>"
                                    data-date="<?php echo $day['date']; ?>"
                                    data-status="<?php echo $status; ?>"
                                    data-start-time="<?php echo $start_time ? date('H:i', strtotime($start_time)) : ''; ?>"
                                    data-end-time="<?php echo $end_time ? date('H:i', strtotime($end_time)) : ''; ?>"
                                    onclick="quickMark(this)"
                                    title="Click to mark attendance"
                                    <?php else: ?>
                                    title="Not scheduled"
                                    <?php endif; ?>
                                >
                                    <?php if ($works_this_day): ?>
                                        <!-- Badge (Roster OR Authorisation - mutually exclusive) -->
                                        <div class="mb-1">
                                            <?php if ($is_override): ?>
                                                <!-- ROSTER badge - anything coming from roster page -->
                                                <span class="badge bg-warning text-dark"
                                                      style="font-size: 9px; cursor: pointer;"
                                                      onclick="event.stopPropagation(); showRosterDetails(<?php echo $worker_id; ?>, <?php echo $location_id; ?>, '<?php echo $day['date']; ?>')"
                                                      title="Click to view roster change details">
                                                    <i class="bi bi-calendar-check"></i> ROSTER
                                                </span>
                                            <?php else:
                                                // AUTH badge - only for direct calendar actions (NOT roster)
                                                $auth_key = $worker_id . '_' . $day['date'];
                                                $has_auth = isset($absence_authorisations[$auth_key]);
                                                $auth_required_statuses = ['PL', 'FO', 'HO', 'UL', 'ML'];

                                                if ($has_auth && in_array($status, $auth_required_statuses)): ?>
                                                    <span class="badge text-success"
                                                          style="font-size: 9px; cursor: pointer; background-colour: #d4edda !important; colour: #155724 !important;"
                                                          onclick="event.stopPropagation(); showAuthDetails(<?php echo $worker_id; ?>, '<?php echo $day['date']; ?>')"
                                                          title="Click to view authorisation details">
                                                        <i class="bi bi-shield-check"></i> AUTH
                                                    </span>
                                                <?php endif;
                                            endif; ?>
                                        </div>

                                        <!-- Shift Name & Time -->
                                        <?php if ($shift_name): ?>
                                            <div class="shift-header"><?php echo htmlspecialchars($shift_name); ?></div>
                                        <?php endif; ?>
                                        <?php if ($start_time && $end_time): ?>
                                            <div class="shift-time">(<?php echo date('H:i A', strtotime($start_time)); ?> – <?php echo date('h:i A', strtotime($end_time)); ?>)</div>
                                        <?php endif; ?>

                                        <!-- Attendance Badge -->
                                        <div class="attendance-status mt-2">
                                            <span class="badge bg-<?php echo $badge_class; ?>"><?php echo $status; ?></span>
                                        </div>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <?php endforeach; ?>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
    <?php endforeach; ?>
<?php endif; ?>

<!-- Quick Mark Modal (same as calendar.php) -->
<div class="modal fade" id="quickMarkModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Mark Attendance</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="quickMarkForm">
                <div class="modal-body">
                    <input type="hidden" id="mark_worker_id" name="worker_id">
                    <input type="hidden" id="mark_date" name="date">

                    <p><strong>Worker:</strong> <span id="worker_name"></span></p>
                    <p><strong>Date:</strong> <span id="mark_date_display"></span></p>

                    <div class="mb-3">
                        <label class="form-label">Status <span class="text-danger">*</span></label>
                        <select name="status" id="mark_status" class="form-select" required onchange="handleStatusChange()">
                            <option value="" disabled selected>-- Select Attendance Status --</option>
                            <option value="P">P - Present</option>
                            <option value="A">A - Absent</option>
                            <option value="PL">PL - Paid Leave (Requires Authorisation)</option>
                            <option value="UL">UL - Unpaid Leave (Requires Authorisation)</option>
                            <option value="FO">FO - Full Day Off (Requires Authorisation)</option>
                            <option value="HO">HO - Half Day Off (Requires Authorisation)</option>
                            <option value="ML">ML - Maternity Leave (Requires Authorisation)</option>
                            <option value="DELETE" style="colour: red; display: none;">Un-mark</option>
                        </select>
                    </div>

                    <!-- Half Day Time Selection (for HO only) -->
                    <div id="halfDayTimeFields" style="display: none;" class="mb-3">
                        <div class="alert alert-info py-2 mb-2">
                            <i class="bi bi-info-circle"></i> <strong>Half Day Off - Select Time Period</strong>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Half Day Period <span class="text-danger">*</span></label>
                            <select name="half_day_period" id="half_day_period" class="form-select" onchange="handleHalfDayPeriod()">
                                <option value="">-- Select Period --</option>
                                <option value="first_half">First Half (Morning)</option>
                                <option value="second_half">Second Half (Afternoon)</option>
                                <option value="custom">Custom Time Range</option>
                            </select>
                        </div>
                        <div id="customTimeRange" style="display: none;">
                            <div class="row">
                                <div class="col-6 mb-3">
                                    <label class="form-label">From Time</label>
                                    <input type="time" name="time_from" id="time_from" class="form-control">
                                </div>
                                <div class="col-6 mb-3">
                                    <label class="form-label">To Time</label>
                                    <input type="time" name="time_to" id="time_to" class="form-control">
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Authorisation Fields (for PL, FO, HO, UL, ML) -->
                    <div id="authorisationFields" style="display: none;" class="mb-3">
                        <div class="alert alert-warning py-2 mb-2">
                            <i class="bi bi-shield-exclamation"></i> <strong>Management Authorisation Required</strong>
                        </div>

                        <?php if ($_SESSION['permission_level'] >= 90): ?>
                            <!-- Admin can authorise as another user -->
                            <div class="mb-3">
                                <label class="form-label">Authorised By <span class="text-danger">*</span></label>
                                <select name="authorized_by" id="authorized_by" class="form-select">
                                    <option value="<?php echo $_SESSION['user_id']; ?>">
                                        Myself (<?php echo htmlspecialchars($_SESSION['username']); ?>)
                                    </option>
                                    <?php
                                    $users_stmt = $pdo->query("SELECT id, username, full_name FROM " . DB_PREFIX . "users WHERE permission_level >= 50 AND id != " . $_SESSION['user_id'] . " ORDER BY username");
                                    foreach ($users_stmt->fetchAll(PDO::FETCH_ASSOC) as $user):
                                    ?>
                                        <option value="<?php echo $user['id']; ?>">
                                            <?php echo htmlspecialchars($user['username']); ?>
                                            <?php echo $user['full_name'] ? ' (' . htmlspecialchars($user['full_name']) . ')' : ''; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        <?php else: ?>
                            <!-- Regular manager - only authorise as themselves -->
                            <input type="hidden" name="authorized_by" id="authorized_by" value="<?php echo $_SESSION['user_id']; ?>">
                            <p class="mb-2"><strong>Authorised By:</strong> <?php echo htmlspecialchars($_SESSION['username']); ?></p>
                        <?php endif; ?>

                        <div class="mb-3">
                            <label class="form-label">Reason / Notes <span class="text-danger">*</span></label>
                            <textarea name="absence_notes" id="absence_notes" class="form-control" rows="2" placeholder="Enter reason for absence..." required></textarea>
                        </div>
                    </div>

                    <!-- Regular Time Fields (for Present) -->
                    <div id="regularTimeFields" class="row">
                        <div class="col-6 mb-3">
                            <label class="form-label">Check-in</label>
                            <input type="time" name="check_in" id="mark_check_in" class="form-control">
                        </div>
                        <div class="col-6 mb-3">
                            <label class="form-label">Check-out</label>
                            <input type="time" name="check_out" id="mark_check_out" class="form-control">
                        </div>
                        <div class="col-12 mb-3">
                            <label class="form-label">Notes (Optional)</label>
                            <textarea name="notes" id="mark_notes" class="form-control" rows="2" placeholder="Additional notes..."></textarea>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
let quickMarkModal;

document.addEventListener('DOMContentLoaded', function() {
    quickMarkModal = new bootstrap.Modal(document.getElementById('quickMarkModal'));

    document.getElementById('quickMarkForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const formData = new FormData(this);

        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Saving...';

        try {
            const ajaxUrl = window.location.pathname.includes('/m/')
                ? '/sys/scms/modules/attendance/ajax/mark_attendance_simple.php'
                : 'ajax/mark_attendance_simple.php';

            const response = await fetch(ajaxUrl, {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                throw new Error('Server returned ' + response.status);
            }

            const result = await response.json();

            if (result.success) {
                quickMarkModal.hide();
                location.reload();
            } else {
                alert('Error: ' + (result.message || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error:', error);
            alert('Error marking attendance: ' + error.message);
        } finally {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalText;
        }
    });

    // Prevent dropdown from closing when clicking inside
    document.querySelectorAll('#locationDropdown .dropdown-menu').forEach(function(menu) {
        menu.addEventListener('click', function(e) {
            e.stopPropagation();
        });
    });

    // Initialize column resizers after a short delay to ensure tables are rendered
    setTimeout(initColumnResizers, 100);

    // Initialize mobile touch support
    setTimeout(initMobileTouchSupport, 200);
});

// Column resizer functionality
function initColumnResizers() {
    const resizers = document.querySelectorAll('.column-resizer');

    if (resizers.length === 0) {
        console.warn('No column resizers found');
        return;
    }

    resizers.forEach(resizer => {
        resizer.addEventListener('mousedown', function(e) {
            e.preventDefault();
            e.stopPropagation();

            const th = this.parentElement;
            const table = th.closest('table');
            const colIndex = Array.from(th.parentElement.children).indexOf(th);
            const startX = e.pageX;
            const startWidth = th.offsetWidth;

            // Add resizing class
            th.classList.add('resizing');
            document.body.style.cursor = 'col-resize';
            document.body.style.userSelect = 'none';

            function resize(e) {
                e.preventDefault();
                const width = startWidth + (e.pageX - startX);
                if (width >= 40) { // Minimum width of 40px
                    th.style.width = width + 'px';

                    // Update all cells in this column
                    const rows = table.querySelectorAll('tbody tr');
                    rows.forEach(row => {
                        const cell = row.children[colIndex];
                        if (cell) {
                            cell.style.width = width + 'px';
                        }
                    });
                }
            }

            function stopResize() {
                document.removeEventListener('mousemove', resize);
                document.removeEventListener('mouseup', stopResize);
                th.classList.remove('resizing');
                document.body.style.cursor = '';
                document.body.style.userSelect = '';
            }

            document.addEventListener('mousemove', resize);
            document.addEventListener('mouseup', stopResize);
        });
    });
}

// Handle status change to show/hide authorisation fields
function handleStatusChange() {
    const status = document.getElementById('mark_status').value;
    const authFields = document.getElementById('authorisationFields');
    const halfDayFields = document.getElementById('halfDayTimeFields');
    const regularTimeFields = document.getElementById('regularTimeFields');
    const absenceNotesField = document.getElementById('absence_notes');

    // Statuses that require authorisation
    const authRequired = ['PL', 'FO', 'HO', 'UL', 'ML'];

    if (authRequired.includes(status)) {
        authFields.style.display = 'block';
        if (absenceNotesField) absenceNotesField.required = true;

        // Show half-day fields only for HO
        if (status === 'HO') {
            halfDayFields.style.display = 'block';
            regularTimeFields.style.display = 'none';
        } else {
            // For all other leaves (PL, FO, UL, ML) - hide time fields
            halfDayFields.style.display = 'none';
            regularTimeFields.style.display = 'none';
        }
    } else {
        // For Present (P) or Absent (A) - show time fields, hide authorisation
        authFields.style.display = 'none';
        halfDayFields.style.display = 'none';
        regularTimeFields.style.display = 'block';
        if (absenceNotesField) absenceNotesField.required = false;
    }
}

// Handle half-day period selection
function handleHalfDayPeriod() {
    const period = document.getElementById('half_day_period').value;
    const customTimeRange = document.getElementById('customTimeRange');

    if (period === 'custom') {
        customTimeRange.style.display = 'block';
    } else {
        customTimeRange.style.display = 'none';
    }
}

function quickMark(cell) {
    const workerId = cell.dataset.workerId;
    const workerName = cell.dataset.workerName;
    const date = cell.dataset.date;
    const currentStatus = cell.dataset.status;
    const startTime = cell.dataset.startTime;
    const endTime = cell.dataset.endTime;

    document.getElementById('mark_worker_id').value = workerId;
    document.getElementById('mark_date').value = date;
    document.getElementById('worker_name').textContent = workerName;
    document.getElementById('mark_date_display').textContent = date;

    // Auto-populate shift timings if available
    if (startTime) {
        document.getElementById('mark_check_in').value = startTime;
    } else {
        document.getElementById('mark_check_in').value = '';
    }

    if (endTime) {
        document.getElementById('mark_check_out').value = endTime;
    } else {
        document.getElementById('mark_check_out').value = '';
    }

    const deleteOption = document.querySelector('#mark_status option[value="DELETE"]');
    if (currentStatus === '-') {
        // Not yet marked - don't auto-select any status, let user choose manually
        if (deleteOption) deleteOption.style.display = 'none';
        document.getElementById('mark_status').value = ''; // Empty - user must select
        // Reset fields
        document.getElementById('authorizationFields').style.display = 'none';
        document.getElementById('halfDayTimeFields').style.display = 'none';
        document.getElementById('regularTimeFields').style.display = 'block';
    } else {
        // Already marked - show current status and allow un-marking
        if (deleteOption) deleteOption.style.display = 'block';
        document.getElementById('mark_status').value = currentStatus;
        // Trigger status change to show appropriate fields
        handleStatusChange();
    }

    quickMarkModal.show();
}

// Mobile touch support for attendance cells
function initMobileTouchSupport() {
    const attendanceCells = document.querySelectorAll('.attendance-cell[data-worker-id]');

    attendanceCells.forEach(cell => {
        // Add touch event handlers
        cell.addEventListener('touchend', function(e) {
            e.preventDefault(); // Prevent ghost clicks
            e.stopPropagation();
            quickMark(this);
        }, { passive: false });

        // Improve tap target size for mobile
        cell.style.minHeight = '60px';
    });
}

// Location filter functions
function selectAllLocations() {
    document.querySelectorAll('.location-checkbox').forEach(function(checkbox) {
        checkbox.checked = true;
    });
    updateLocationFilter();
}

function clearAllLocations() {
    document.querySelectorAll('.location-checkbox').forEach(function(checkbox) {
        checkbox.checked = false;
    });
    updateLocationFilter();
}

function updateLocationFilter() {
    const checkboxes = document.querySelectorAll('.location-checkbox:checked');
    const textSpan = document.getElementById('selectedLocationsText');

    if (checkboxes.length === 0) {
        textSpan.textContent = 'All Locations';
    } else if (checkboxes.length === document.querySelectorAll('.location-checkbox').length) {
        textSpan.textContent = 'All Locations';
    } else {
        textSpan.textContent = checkboxes.length + ' location' + (checkboxes.length > 1 ? 's' : '') + ' selected';
    }
}

function applyLocationFilter() {
    const checkboxes = document.querySelectorAll('.location-checkbox:checked');
    const selectedIds = Array.from(checkboxes).map(cb => cb.value);

    const urlParams = new URLSearchParams(window.location.search);
    const currentView = urlParams.get('view') || 'week';
    const currentDate = urlParams.get('date') || '<?php echo $current_date; ?>';

    if (selectedIds.length === 0 || selectedIds.length === document.querySelectorAll('.location-checkbox').length) {
        // All locations or none - go to all locations view
        window.location = '?view=' + currentView + '&date=' + currentDate;
    } else {
        // Specific locations selected
        window.location = '?view=' + currentView + '&date=' + currentDate + '&locations=' + selectedIds.join(',');
    }
}

// View selector functions
function changeView(viewType) {
    // Get current URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    const currentDate = urlParams.get('date') || '<?php echo date('Y-m-d'); ?>';
    const locations = urlParams.get('locations') || '';

    // Build new URL with view parameter
    let newUrl = '?view=' + viewType + '&date=' + currentDate;
    if (locations) {
        newUrl += '&locations=' + locations;
    }

    // Redirect to new view
    window.location = newUrl;
}

function resetToDefault() {
    // Get current URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    const locations = urlParams.get('locations') || '';

    // Reset to weekly view with current week
    let newUrl = '?view=week&date=<?php echo date('Y-m-d'); ?>';
    if (locations) {
        newUrl += '&locations=' + locations;
    }

    window.location = newUrl;
}
</script>

<style>
/* Professional attendance table styling */
.attendance-table {
    font-size: 13px;
    margin-bottom: 0;
}

.attendance-table thead th {
    background: #f8f9fa;
    colour: #333;
    font-weight: 600;
    border: 1px solid #dee2e6;
    padding: 12px 8px;
    text-align: center;
    vertical-align: middle;
    position: sticky;
    top: 0;
    z-index: 10;
}

.attendance-table tbody td {
    padding: 10px 8px;
    vertical-align: middle;
    border: 1px solid #e0e0e0;
}

/* Day column in header */
.day-cell-header {
    width: 55px;
    min-width: 55px;
    max-width: 55px;
    background: #f8f9fa;
    font-weight: 600;
    text-align: left;
    vertical-align: middle;
    padding: 8px 2px !important;
    border-right: 3px solid #dee2e6 !important;
    position: relative;
    user-select: none;
}

/* Worker column in header */
.worker-column-header {
    min-width: 150px;
    width: 150px;
    max-width: 150px;
    text-align: center;
    vertical-align: middle;
    padding: 10px 6px;
    position: relative;
    user-select: none;
}

/* Column resizer handle */
.column-resizer {
    position: absolute;
    top: 0;
    right: 0;
    width: 8px;
    height: 100%;
    cursor: col-resize;
    background: transparent;
    border-right: 2px solid transparent;
    transition: all 0.2s ease;
    z-index: 100;
}

.column-resizer:hover {
    border-right: 2px solid #6366f1;
    background: rgba(99, 102, 241, 0.1);
}

.resizable-column.resizing {
    user-select: none;
    cursor: col-resize;
}

.resizable-column.resizing .column-resizer {
    border-right: 2px solid #6366f1;
    background: rgba(99, 102, 241, 0.15);
}

.position-title {
    font-size: 13px;
    font-weight: 600;
    colour: #333;
    margin-bottom: 4px;
}

.worker-name-header {
    font-size: 12px;
    colour: #666;
    font-weight: 400;
}

/* Day cell in body */
.day-cell {
    font-weight: 600;
    background: #f8f9fa;
    width: 55px;
    text-align: left;
    padding: 8px 2px;
    border-right: 3px solid #dee2e6 !important;
}

.day-name {
    font-size: 13px;
    font-weight: 600;
    colour: #333;
}

.day-date {
    font-size: 11px;
    colour: #666;
    margin-top: 2px;
}

.attendance-cell {
    cursor: pointer;
    transition: all 0.2s ease;
    text-align: center;
    padding: 10px 6px !important;
    min-width: 150px;
    width: 150px;
    max-width: 150px;
    vertical-align: middle;
}

.attendance-cell:hover {
    background-color: #e3f2fd !important;
}

.attendance-cell.not-scheduled {
    background-colour: #f5f5f5;
    cursor: not-allowed;
}

/* Shift header (Shift Name) */
.shift-header {
    font-size: 12px;
    font-weight: 600;
    colour: #333;
    margin-bottom: 4px;
}

/* Shift timing */
.shift-time {
    font-size: 11px;
    colour: #666;
    margin-bottom: 8px;
}

/* Attendance status badge */
.attendance-status {
    margin-top: 8px;
}

.badge {
    min-width: 35px;
    padding: 5px 8px;
    font-size: 11px;
    font-weight: 600;
    border-radius: 4px;
}

/* Pink badge for maternity leave */
.bg-pink {
    background-colour: #e91e63 !important;
    colour: white !important;
}

/* Location cards */
.location-card {
    border-radius: 12px;
    overflow: hidden;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    margin-bottom: 24px;
    border: none;
}

.location-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    colour: white;
    padding: 16px 20px;
    border: none;
}

.location-header h5 {
    margin: 0;
    font-size: 18px;
    font-weight: 600;
}

.location-header small {
    opacity: 0.9;
    font-size: 13px;
}

/* Multi-select dropdown */
.location-dropdown {
    position: relative;
}

/* Location selector button styling */
.location-selector-btn {
    padding: 10px 14px;
    padding-right: 35px;
    font-size: 14px;
    font-weight: 500;
    border-radius: 8px;
    border: 1.5px solid #6366f1;
    background: #ffffff;
    colour: #6366f1;
    box-shadow: 0 2px 4px rgba(99, 102, 241, 0.1);
    transition: all 0.2s ease;
    position: relative;
}

.location-selector-btn::after {
    position: absolute;
    right: 14px;
    top: 50%;
    transform: translateY(-50%);
}

.location-selector-btn:hover {
    background: #f5f6ff;
    border-color: #5558e3;
    color: #5558e3;
    box-shadow: 0 4px 8px rgba(99, 102, 241, 0.15);
    transform: translateY(-1px);
}

.location-selector-btn:focus,
.location-selector-btn:active {
    background: #f5f6ff;
    border-color: #5558e3;
    color: #5558e3;
    box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.1);
}

.location-dropdown .dropdown-menu {
    min-width: 300px;
    max-height: 400px;
    overflow-y: auto;
    border-radius: 10px;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    border: 1px solid #e0e0e0;
    padding: 10px;
    margin-top: 8px;
}

.location-dropdown .form-check {
    padding: 10px 12px;
    margin: 4px 0;
    border-radius: 8px;
    transition: all 0.2s ease;
    background: #ffffff;
}

.location-dropdown .form-check:hover {
    background-color: #f5f6ff;
}

.location-dropdown .form-check-input {
    width: 18px;
    height: 18px;
    margin-top: 3px;
    cursor: pointer;
    border: 2px solid #6366f1;
    border-radius: 4px;
}

.location-dropdown .form-check-input:checked {
    background-color: #6366f1;
    border-color: #6366f1;
}

.location-dropdown .form-check-label {
    cursor: pointer;
    margin-left: 10px;
    font-size: 14px;
    font-weight: 500;
    color: #333;
}

.location-dropdown .dropdown-divider {
    margin: 10px 0;
    opacity: 0.15;
}

/* Dropdown action buttons */
.location-dropdown .btn-sm {
    padding: 8px 12px;
    font-size: 13px;
    font-weight: 600;
    border-radius: 6px;
    transition: all 0.2s ease;
}

.location-dropdown .btn-outline-primary {
    border-color: #6366f1;
    color: #6366f1;
}

.location-dropdown .btn-outline-primary:hover {
    background: #6366f1;
    color: #ffffff;
}

.location-dropdown .btn-outline-secondary {
    border-color: #94a3b8;
    color: #64748b;
}

.location-dropdown .btn-outline-secondary:hover {
    background: #94a3b8;
    color: #ffffff;
}

.location-dropdown .btn-primary {
    background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
    border: none;
    box-shadow: 0 2px 6px rgba(99, 102, 241, 0.25);
}

.location-dropdown .btn-primary:hover {
    background: linear-gradient(135deg, #5558e3 0%, #7c3aed 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(99, 102, 241, 0.3);
}

/* Filter section */
.filter-section {
    background: white;
    border-radius: 12px;
    padding: 20px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.08);
    margin-bottom: 20px;
}

/* Legend */
.legend-card {
    background: white;
    border-radius: 12px;
    padding: 12px 20px;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.08);
    margin-bottom: 20px;
}

.legend-card .badge {
    margin: 0 8px;
}

/* Table responsive */
.table-responsive {
    overflow-x: auto;
    border-radius: 0 0 12px 12px;
}

.table-responsive::-webkit-scrollbar {
    height: 10px;
}

.table-responsive::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 5px;
}

.table-responsive::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 5px;
}

.table-responsive::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
}

/* Job title badge in header */
.job-title-badge {
    display: inline-block;
    background: #e7f3ff;
    color: #0066cc;
    padding: 4px 10px;
    border-radius: 20px;
    font-size: 11px;
    font-weight: 600;
    margin-bottom: 4px;
    border: 1px solid #b3d9ff;
}

/* Worker name in header */
.worker-name {
    font-size: 12px;
    font-weight: 500;
    color: #333;
}

/* Shift badge */
.shift-badge {
    font-size: 10px;
    padding: 2px 8px;
    margin-top: 4px;
}

/* Empty state */
.empty-state {
    text-align: center;
    padding: 60px 20px;
    color: #999;
}

.empty-state i {
    font-size: 48px;
    margin-bottom: 16px;
    opacity: 0.3;
}

/* Today highlight */
.today-highlight {
    background-color: #fff3cd !important;
    font-weight: 600;
}

/* Time display in cells */
.time-display {
    font-size: 10px;
    color: #666;
    margin-top: 2px;
}

/* Buttons */
.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
}

/* Mobile Responsiveness */
@media (max-width: 768px) {
    .attendance-cell {
        min-height: 80px !important;
        padding: 15px 10px !important;
        touch-action: manipulation;
        -webkit-tap-highlight-color: rgba(0, 0, 0, 0.1);
    }

    .shift-header {
        font-size: 13px !important;
    }

    .shift-time {
        font-size: 10px !important;
    }

    .badge {
        font-size: 10px !important;
        padding: 4px 6px !important;
        min-width: 30px !important;
    }

    .filter-section {
        padding: 15px !important;
    }

    .location-header h5 {
        font-size: 16px !important;
    }

    .btn-group {
        flex-wrap: wrap;
    }

    .table-responsive {
        -webkit-overflow-scrolling: touch;
    }
}
</style>

<!-- Tracking Modals -->
<?php include 'modals/tracking_modals.php'; ?>

<!-- Tracking JavaScript -->
<script src="js/attendance_tracking.js"></script>

<?php
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
