<?php
// Attendance Calendar - Updated 2025-10-18
// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    // Get current user
    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }

    $page_title = 'Attendance Calendar';

    // Include header BEFORE doing any database queries
    include '../../includes/header.php';
}

// Check permission level (for both embedded and standalone)
if (($_SESSION['permission_level'] ?? 0) < 30) {
    die('Access denied. You do not have permission to view attendance.');
}

// Get month/year from URL or use current
$month = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('n');
$year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');

// Validate month/year
$month = max(1, min(12, $month));
$year = max(2020, min(2030, $year));

// Calculate dates
$first_day = date('Y-m-01', strtotime("$year-$month-01"));
$last_day = date('Y-m-t', strtotime("$year-$month-01"));
$days_in_month = (int)date('t', strtotime("$year-$month-01"));

// Get department filter
$department_filter = $_GET['department'] ?? '';

// Get all active workers
$workers_sql = "SELECT id, full_name, employee_number, department FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'";
if ($department_filter) {
    $workers_sql .= " AND department = " . $pdo->quote($department_filter);
}
$workers_sql .= " ORDER BY full_name";
$workers = $pdo->query($workers_sql)->fetchAll(PDO::FETCH_ASSOC);

// Get all attendance for the month
$attendance_stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "attendance_records WHERE attendance_date >= ? AND attendance_date <= ?");
$attendance_stmt->execute([$first_day, $last_day]);
$attendance_data = [];
foreach ($attendance_stmt->fetchAll(PDO::FETCH_ASSOC) as $record) {
    $key = $record['worker_id'] . '_' . $record['attendance_date'];
    $attendance_data[$key] = $record;
}

// Get departments for filter
$departments = $pdo->query("SELECT DISTINCT department FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' AND department IS NOT NULL ORDER BY department")->fetchAll(PDO::FETCH_COLUMN);

// Calculate navigation
$prev_month = $month == 1 ? 12 : $month - 1;
$prev_year = $month == 1 ? $year - 1 : $year;
$next_month = $month == 12 ? 1 : $month + 1;
$next_year = $month == 12 ? $year + 1 : $year;

$month_name = date('F Y', strtotime("$year-$month-01"));

// Get stats for cards - ALWAYS filter by active workers only, optionally by department
$active_workers_filter = " AND worker_id IN (SELECT id FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'";
if ($department_filter) {
    $active_workers_filter .= " AND department = " . $pdo->quote($department_filter);
}
$active_workers_filter .= ")";

$total_attendance_today = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "attendance_records WHERE DATE(attendance_date) = CURDATE()" . $active_workers_filter)->fetchColumn();
$total_attendance_month = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "attendance_records WHERE MONTH(attendance_date) = MONTH(CURDATE()) AND YEAR(attendance_date) = YEAR(CURDATE())" . $active_workers_filter)->fetchColumn();
$present_today = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "attendance_records WHERE DATE(attendance_date) = CURDATE() AND status = 'P'" . $active_workers_filter)->fetchColumn();
$absent_today = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "attendance_records WHERE DATE(attendance_date) = CURDATE() AND status = 'A'" . $active_workers_filter)->fetchColumn();
?>

<!-- Quick Stats Grid -->
<div class="quick-actions-grid mb-4">
    <div class="quick-action-btn gradient-cyan">
        <div class="quick-action-icon">
            <span class="material-symbols-outlined" style="font-size: 1.5rem;">today</span>
        </div>
        <span class="quick-action-label"><?php echo number_format($total_attendance_today); ?> Today's Records</span>
    </div>

    <div class="quick-action-btn gradient-blue">
        <div class="quick-action-icon">
            <i class="fas fa-calendar-alt"></i>
        </div>
        <span class="quick-action-label"><?php echo number_format($total_attendance_month); ?> This Month</span>
    </div>

    <div class="quick-action-btn gradient-green">
        <div class="quick-action-icon">
            <i class="ri-user-follow-fill"></i>
        </div>
        <span class="quick-action-label"><?php echo number_format($present_today); ?> Present Today</span>
    </div>

    <div class="quick-action-btn gradient-red">
        <div class="quick-action-icon">
            <i class="ri-user-unfollow-fill"></i>
        </div>
        <span class="quick-action-label"><?php echo number_format($absent_today); ?> Absent Today</span>
    </div>
</div>

<!-- Header -->
    <div class="modern-card mb-3">
        <div class="card-body-modern">
            <div class="row align-items-center">
                <div class="col-md-2">
                    <h4><i class="bi bi-calendar3"></i> <?php echo $month_name; ?></h4>
                    <small class="text-muted"><?php echo count($workers); ?> workers</small>
                </div>
                <div class="col-md-2">
                    <select class="form-select form-select-sm" onchange="window.location='?month=<?php echo $month; ?>&year=<?php echo $year; ?>&department='+this.value">
                        <option value="">All Departments</option>
                        <?php foreach ($departments as $dept): ?>
                        <option value="<?php echo htmlspecialchars($dept); ?>" <?php echo $department_filter === $dept ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($dept); ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2 text-center">
                    <div class="btn-group btn-group-sm">
                        <a href="?month=<?php echo $prev_month; ?>&year=<?php echo $prev_year; ?><?php echo $department_filter ? '&department='.$department_filter : ''; ?>" class="btn btn-outline-secondary">
                            <i class="bi bi-chevron-left"></i>
                        </a>
                        <a href="?month=<?php echo date('n'); ?>&year=<?php echo date('Y'); ?><?php echo $department_filter ? '&department='.$department_filter : ''; ?>" class="btn btn-outline-primary">
                            Today
                        </a>
                        <a href="?month=<?php echo $next_month; ?>&year=<?php echo $next_year; ?><?php echo $department_filter ? '&department='.$department_filter : ''; ?>" class="btn btn-outline-secondary">
                            <i class="bi bi-chevron-right"></i>
                        </a>
                    </div>
                </div>
                <div class="col-md-6 text-end">
                    <?php
                    // Determine base path for links
                    $base_path = defined('MODULE_EMBEDDED') ? '/sys/scms/modules/attendance/' : '';
                    ?>
                    <a href="<?php echo $base_path; ?>calendar_by_location.php" class="btn btn-sm btn-primary me-2" title="View by Location">
                        <i class="bi bi-geo-alt"></i> By Location
                    </a>
                    <div class="btn-group btn-group-sm" role="group">
                        <a href="<?php echo $base_path; ?>bulk_mark.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-success" title="Bulk mark attendance">
                            <i class="bi bi-check-all"></i> Bulk
                        </a>
                        <a href="<?php echo $base_path; ?>report_monthly.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-info" title="View monthly report">
                            <i class="bi bi-file-earmark-text"></i> Report
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Legend -->
    <div class="modern-card mb-3">
        <div class="card-body-modern py-2">
            <div class="d-flex justify-content-center gap-3 flex-wrap">
                <span><span class="badge bg-success">P</span> Present</span>
                <span><span class="badge bg-danger">A</span> Absent (Unauthorized)</span>
                <span><span class="badge bg-info">PL</span> Paid Leave</span>
                <span><span class="badge bg-warning">UL</span> Unpaid Leave</span>
                <span><span class="badge bg-primary">FO</span> Full Day Off</span>
                <span><span class="badge bg-secondary">HO</span> Half Day Off</span>
                <span><span class="badge" style="background-color: #e83e8c;">ML</span> Maternity Leave</span>
                <span><span class="badge bg-light text-dark border">-</span> Not Marked</span>
            </div>
        </div>
    </div>

    <!-- Calendar Table -->
    <div class="modern-card">
        <div class="card-body-modern p-0">
            <?php if (empty($workers)): ?>
            <!-- Empty State -->
            <div class="text-center py-5">
                <i class="bi bi-people" style="font-size: 4rem; color: #ccc;"></i>
                <h5 class="mt-3 text-muted">No Active Workers Found</h5>
                <p class="text-muted">Add workers to the system to start tracking attendance.</p>
                <a href="/sys/scms/modules/workers/add.php" class="btn btn-primary mt-3">
                    <i class="bi bi-person-plus"></i> Add Worker
                </a>
            </div>
            <?php else: ?>
            <div class="table-responsive">
                <table class="table table-bordered table-hover mb-0 attendance-calendar">
                    <thead class="table-light" style="position: sticky; top: 0; z-index: 10;">
                        <tr>
                            <th style="min-width: 200px; position: sticky; left: 0; background: #f8f9fa; z-index: 11;">Worker</th>
                            <?php for ($day = 1; $day <= $days_in_month; $day++):
                                $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
                                $day_name = date('D', strtotime($date));
                                $is_weekend = in_array($day_name, ['Sat', 'Sun']);
                                $is_today = $date === date('Y-m-d');
                            ?>
                            <th class="text-center <?php echo $is_weekend ? 'bg-secondary bg-opacity-10' : ''; ?> <?php echo $is_today ? 'border-primary border-3' : ''; ?>" style="min-width: 45px; width: 45px; font-size: 11px; padding: 4px 2px;">
                                <div style="font-weight: 600;"><?php echo $day; ?></div>
                                <div style="font-size: 9px; color: #666; font-weight: normal;"><?php echo $day_name; ?></div>
                            </th>
                            <?php endfor; ?>
                            <th style="min-width: 60px; width: 60px; position: sticky; right: 0; background: #f8f9fa; z-index: 10; padding: 4px;">Stats</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($workers as $worker): ?>
                        <tr>
                            <td style="position: sticky; left: 0; background: white; z-index: 5; font-size: 13px;">
                                <div><strong><?php echo htmlspecialchars($worker['full_name']); ?></strong></div>
                                <div><small class="text-muted"><?php echo htmlspecialchars($worker['employee_number']); ?></small></div>
                            </td>
                            <?php
                            $present_count = 0;
                            $leave_count = 0;
                            $absent_count = 0;

                            for ($day = 1; $day <= $days_in_month; $day++):
                                $date = sprintf('%04d-%02d-%02d', $year, $month, $day);
                                $key = $worker['id'] . '_' . $date;
                                $record = $attendance_data[$key] ?? null;
                                $day_name = date('D', strtotime($date));
                                $is_weekend = in_array($day_name, ['Sat', 'Sun']);
                                $is_today = $date === date('Y-m-d');

                                if ($record) {
                                    $status = $record['status'];
                                    if ($status == 'P') $present_count++;
                                    if ($status == 'PL') $leave_count++;
                                    if ($status == 'A' || $status == 'UL') $absent_count++;

                                    $badge_colors = [
                                        'P' => 'success',
                                        'A' => 'danger',
                                        'PL' => 'info',
                                        'UL' => 'warning',
                                        'FO' => 'primary',
                                        'HO' => 'secondary',
                                        'ML' => 'pink'
                                    ];
                                    // Add custom color for ML (don't use badge_colors array)
                                    if ($status == 'ML') {
                                        $badge_class = '';  // Empty - use inline style only
                                        $custom_bg_color = 'background-color: #e83e8c !important; color: white;';
                                    } else {
                                        $badge_class = $badge_colors[$status] ?? 'secondary';
                                        $custom_bg_color = '';
                                    }
                                } else {
                                    $status = '-';
                                    $badge_class = 'light text-dark border';
                                    $custom_bg_color = '';
                                }
                            ?>
                            <td class="text-center p-1 attendance-cell <?php echo $is_weekend ? 'bg-secondary bg-opacity-10' : ''; ?> <?php echo $is_today ? 'border-primary border-2' : ''; ?>"
                                data-worker-id="<?php echo $worker['id']; ?>"
                                data-worker-name="<?php echo htmlspecialchars($worker['full_name']); ?>"
                                data-date="<?php echo $date; ?>"
                                data-status="<?php echo $status; ?>"
                                onclick="quickMark(this)"
                                style="cursor: pointer;"
                                title="<?php echo $date; ?><?php echo $record && $record['notes'] ? ' - ' . htmlspecialchars($record['notes']) : ''; ?>">
                                <span class="badge bg-<?php echo $badge_class; ?>" style="<?php echo $custom_bg_color; ?> font-size: 10px; padding: 3px 5px; min-width: 25px;">
                                    <?php echo $status; ?>
                                </span>
                            </td>
                            <?php endfor; ?>
                            <td style="position: sticky; right: 0; background: white; z-index: 5; font-size: 11px; padding: 4px; text-align: center;">
                                <div class="text-success" style="line-height: 1.3;"><strong><?php echo $present_count; ?>P</strong></div>
                                <div class="text-info" style="line-height: 1.3;"><?php echo $leave_count; ?>L</div>
                                <?php if ($absent_count > 0): ?>
                                <div class="text-danger" style="line-height: 1.3;"><?php echo $absent_count; ?>A</div>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
    </div>

<!-- Quick Mark Modal -->
<div class="modal fade" id="quickMarkModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Mark Attendance</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="quickMarkForm">
                <div class="modal-body">
                    <input type="hidden" id="mark_worker_id" name="worker_id">
                    <input type="hidden" id="mark_date" name="date">

                    <p><strong>Worker:</strong> <span id="worker_name"></span></p>
                    <p><strong>Date:</strong> <span id="mark_date_display"></span></p>

                    <div class="mb-3">
                        <label class="form-label">Status <span class="text-danger">*</span></label>
                        <select name="status" id="mark_status" class="form-select" required>
                            <option value="P">P - Present (at work)</option>
                            <option value="A">A - Absent (unauthorized absence)</option>
                            <option value="PL">PL - Paid Leave (authorized)</option>
                            <option value="UL">UL - Unpaid Leave (authorized)</option>
                            <option value="FO">FO - Full Day Off (authorized)</option>
                            <option value="HO">HO - Half Day Off (authorized)</option>
                            <option value="ML">ML - Maternity Leave</option>
                            <option value="DELETE" style="color: red;">Un-mark (Remove Attendance)</option>
                        </select>
                    </div>

                    <div class="row">
                        <div class="col-6 mb-3">
                            <label class="form-label">Check-in</label>
                            <input type="time" name="check_in" id="mark_check_in" class="form-control">
                        </div>
                        <div class="col-6 mb-3">
                            <label class="form-label">Check-out</label>
                            <input type="time" name="check_out" id="mark_check_out" class="form-control">
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Notes</label>
                        <textarea name="notes" id="mark_notes" class="form-control" rows="2"></textarea>
                    </div>

                    <div id="pinSection" style="display: none;">
                        <div class="alert alert-warning">
                            <i class="bi bi-shield-lock"></i> <strong>PIN Required</strong><br>
                            Editing past attendance requires PIN verification.
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Enter your 4-digit PIN</label>
                            <input type="password" name="pin" id="mark_pin" class="form-control" maxlength="4" pattern="[0-9]{4}">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Reason for editing</label>
                            <input type="text" name="edit_reason" id="edit_reason" class="form-control">
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
let quickMarkModal;

document.addEventListener('DOMContentLoaded', function() {
    quickMarkModal = new bootstrap.Modal(document.getElementById('quickMarkModal'));

    document.getElementById('quickMarkForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        const formData = new FormData(this);

        // Add loading indicator
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Saving...';

        try {
            // Use absolute path to handle both direct and router access
            const ajaxUrl = window.location.pathname.includes('/m/')
                ? '/sys/scms/modules/attendance/ajax/mark_attendance_simple.php'
                : 'ajax/mark_attendance_simple.php';

            const response = await fetch(ajaxUrl, {
                method: 'POST',
                body: formData
            });

            // Check if response is ok
            if (!response.ok) {
                throw new Error('Server returned ' + response.status);
            }

            const text = await response.text();
            console.log('Response text:', text);

            let result;
            try {
                result = JSON.parse(text);
            } catch (e) {
                console.error('JSON parse error:', e);
                console.error('Response was:', text);
                throw new Error('Invalid JSON response from server');
            }

            if (result.success) {
                quickMarkModal.hide();
                location.reload();
            } else {
                alert('Error: ' + (result.message || 'Unknown error'));
            }
        } catch (error) {
            console.error('Fetch error:', error);
            alert('Error marking attendance: ' + error.message);
        } finally {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalText;
        }
    });
});

function quickMark(cell) {
    const workerId = cell.dataset.workerId;
    const workerName = cell.dataset.workerName;
    const date = cell.dataset.date;
    const currentStatus = cell.dataset.status;

    document.getElementById('mark_worker_id').value = workerId;
    document.getElementById('mark_date').value = date;
    document.getElementById('worker_name').textContent = workerName;
    document.getElementById('mark_date_display').textContent = date;

    // Show/hide "Un-mark" option based on whether there's an existing record
    const deleteOption = document.querySelector('#mark_status option[value="DELETE"]');
    if (currentStatus === '-') {
        // No existing record - hide the "Un-mark" option
        if (deleteOption) deleteOption.style.display = 'none';
        document.getElementById('mark_status').value = 'P';
    } else {
        // Existing record - show the "Un-mark" option
        if (deleteOption) deleteOption.style.display = 'block';
        document.getElementById('mark_status').value = currentStatus;
    }

    // Check if PIN required (editing past dates)
    const today = new Date().toISOString().split('T')[0];
    const targetDate = new Date(date);
    const todayDate = new Date(today);
    const daysDiff = Math.floor((todayDate - targetDate) / (1000 * 60 * 60 * 24));

    const pinSection = document.getElementById('pinSection');
    if (daysDiff > 2) { // PIN required after 2 days (configurable)
        pinSection.style.display = 'block';
        document.getElementById('mark_pin').required = true;
        document.getElementById('edit_reason').required = true;
    } else {
        pinSection.style.display = 'none';
        document.getElementById('mark_pin').required = false;
        document.getElementById('edit_reason').required = false;
    }

    quickMarkModal.show();
}

// Keyboard shortcuts
document.addEventListener('keydown', function(e) {
    if (e.ctrlKey && e.key === 'b') {
        e.preventDefault();
        window.location.href = 'bulk_mark.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>';
    }
});
</script>

<style>
.table-responsive {
    overflow-x: auto;
    overflow-y: visible;
    -webkit-overflow-scrolling: touch;
}

/* Custom scrollbar styling for horizontal scroll */
.table-responsive::-webkit-scrollbar {
    height: 8px;
}

.table-responsive::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 4px;
}

.table-responsive::-webkit-scrollbar-thumb {
    background: #888;
    border-radius: 4px;
}

.table-responsive::-webkit-scrollbar-thumb:hover {
    background: #555;
}

.attendance-calendar {
    font-size: 12px;
    margin-bottom: 0;
    white-space: nowrap;
}

.attendance-calendar td,
.attendance-calendar th {
    border: 1px solid #dee2e6;
    padding: 6px 4px;
    vertical-align: middle;
}

.attendance-calendar thead th {
    padding: 8px 4px;
    font-weight: 600;
}

.attendance-cell {
    transition: background-color 0.2s ease;
}

.attendance-cell:hover {
    background-color: rgba(13, 110, 253, 0.1) !important;
    cursor: pointer;
}

.attendance-calendar .badge {
    display: inline-block;
    font-weight: 500;
    min-width: 28px;
    font-size: 10px;
    padding: 4px 6px;
}

/* Sticky columns shadow effect */
.attendance-calendar th:first-child,
.attendance-calendar td:first-child {
    box-shadow: 2px 0 5px rgba(0,0,0,0.05);
}

.attendance-calendar th:last-child,
.attendance-calendar td:last-child {
    box-shadow: -2px 0 5px rgba(0,0,0,0.05);
}

/* Make the table more compact */
.attendance-calendar tbody tr {
    height: 45px;
}

@media print {
    .modern-card:first-child,
    .modern-card:nth-child(2) {
        display: none;
    }

    .attendance-calendar {
        font-size: 8px;
    }

    .table-responsive {
        overflow: visible !important;
    }
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .attendance-calendar {
        font-size: 10px;
    }

    .attendance-calendar th:first-child,
    .attendance-calendar td:first-child {
        min-width: 150px !important;
    }
}
</style>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
