<?php
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication('../../login.php');

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

// Check permission level - HR level (70+) required
if (($_SESSION['permission_level'] ?? 0) < 70) {
    die('Access denied. You do not have permission to bulk mark attendance. HR level or higher required.');
}

$page_title = 'Bulk Mark Attendance';
$errors = [];
$success = '';

// Get current user
$current_user = get_logged_in_user();

// Get month/year from URL
$month = isset($_GET['month']) ? (int)$_GET['month'] : (int)date('n');
$year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');

// Get all active workers
$workers = $pdo->query("SELECT id, full_name, employee_number, department FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' ORDER BY full_name")->fetchAll(PDO::FETCH_ASSOC);

// Get departments
$departments = $pdo->query("SELECT DISTINCT department FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' AND department IS NOT NULL ORDER BY department")->fetchAll(PDO::FETCH_COLUMN);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pin = $_POST['pin'] ?? '';
    $selected_workers = $_POST['workers'] ?? [];
    $date_from = $_POST['date_from'] ?? '';
    $date_to = $_POST['date_to'] ?? '';
    $status = $_POST['status'] ?? '';
    $overwrite = isset($_POST['overwrite']);

    // Validate PIN first
    if (empty($pin)) {
        $errors[] = 'PIN is required for bulk attendance marking';
    } elseif (!password_verify($pin, $current_user['pin_hash'])) {
        $errors[] = 'Invalid PIN. Bulk operation cancelled';
    }

    if (empty($selected_workers)) {
        $errors[] = 'Please select at least one worker';
    }
    if (empty($date_from) || empty($date_to)) {
        $errors[] = 'Please select date range';
    }
    if (empty($status)) {
        $errors[] = 'Please select status';
    }

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            $from_date = new DateTime($date_from);
            $to_date = new DateTime($date_to);
            $records_created = 0;
            $records_updated = 0;
            $records_skipped = 0;

            while ($from_date <= $to_date) {
                $current_date = $from_date->format('Y-m-d');

                foreach ($selected_workers as $worker_id) {
                    // Check if record exists
                    $check_stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ? AND attendance_date = ?");
                    $check_stmt->execute([$worker_id, $current_date]);
                    $existing = $check_stmt->fetch();

                    // Handle DELETE (un-mark)
                    if ($status === 'DELETE') {
                        if ($existing) {
                            $delete_stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "attendance_records WHERE id = ?");
                            $delete_stmt->execute([$existing['id']]);
                            $records_updated++; // Count deletions as updates
                        } else {
                            $records_skipped++;
                        }
                        continue;
                    }

                    if ($existing && !$overwrite) {
                        $records_skipped++;
                        continue;
                    }

                    if ($existing) {
                        // Update
                        $update_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "attendance_records
                            SET status = ?, updated_by = ?, updated_at = NOW()
                            WHERE id = ?");
                        $update_stmt->execute([$status, $_SESSION['user_id'], $existing['id']]);
                        $records_updated++;
                    } else {
                        // Insert
                        $insert_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "attendance_records
                            (worker_id, attendance_date, status, created_by, created_at)
                            VALUES (?, ?, ?, ?, NOW())");
                        $insert_stmt->execute([$worker_id, $current_date, $status, $_SESSION['user_id']]);
                        $records_created++;
                    }
                }

                $from_date->modify('+1 day');
            }

            $pdo->commit();

            $success = "Bulk operation completed! Created: $records_created, Updated: $records_updated, Skipped: $records_skipped";

        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = 'Error: ' . $e->getMessage();
        }
    }
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <div class="modern-card">
                <div class="card-header-modern">
                    <div class="d-flex justify-content-between align-items-center w-100">
                        <h5 class="card-title-modern mb-0">
                            <i class="bi bi-check-all"></i> Bulk Mark Attendance
                        </h5>
                        <div class="ms-auto">
                            <a href="calendar.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-sm btn-outline-secondary">
                                <i class="bi bi-arrow-left"></i> Back to Calendar
                            </a>
                        </div>
                    </div>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            <i class="bi bi-check-circle"></i> <?php echo htmlspecialchars($success); ?>
                            <div class="mt-2">
                                <a href="calendar.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-sm btn-primary">
                                    <i class="bi bi-calendar3"></i> View Calendar
                                </a>
                            </div>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Date From <span class="text-danger">*</span></label>
                                <input type="date" name="date_from" class="form-control" required value="<?php echo $_POST['date_from'] ?? date('Y-m-01'); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Date To <span class="text-danger">*</span></label>
                                <input type="date" name="date_to" class="form-control" required value="<?php echo $_POST['date_to'] ?? date('Y-m-t'); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Status <span class="text-danger">*</span></label>
                                <select name="status" class="form-select" required>
                                    <option value="">Select status...</option>
                                    <option value="P">P - Present</option>
                                    <option value="A">A - Absent</option>
                                    <option value="PL">PL - Paid Leave</option>
                                    <option value="UL">UL - Unpaid Leave</option>
                                    <option value="FO">FO - Full Day Off</option>
                                    <option value="HO">HO - Half Day Off</option>
                                    <option value="ML">ML - Maternity Leave</option>
                                    <option value="WE">WE - Weekend</option>
                                    <option value="DELETE" style="color: red;">Un-mark (Remove Attendance)</option>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Filter by Department</label>
                                <select id="deptFilter" class="form-select" onchange="filterWorkers()">
                                    <option value="">All Departments</option>
                                    <?php foreach ($departments as $dept): ?>
                                        <option value="<?php echo htmlspecialchars($dept); ?>"><?php echo htmlspecialchars($dept); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="overwrite" id="overwrite">
                                <label class="form-check-label" for="overwrite">
                                    Overwrite existing attendance records
                                </label>
                            </div>
                        </div>

                        <hr>

                        <div class="mb-3">
                            <div class="d-flex justify-content-between align-items-center">
                                <label class="form-label">Select Workers <span class="text-danger">*</span></label>
                                <div>
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="selectAll()">Select All</button>
                                    <button type="button" class="btn btn-sm btn-outline-secondary" onclick="deselectAll()">Deselect All</button>
                                </div>
                            </div>

                            <div class="border rounded p-3" style="max-height: 400px; overflow-y: auto;">
                                <div class="row" id="workersList">
                                    <?php foreach ($workers as $worker): ?>
                                    <div class="col-md-4 mb-2 worker-item" data-department="<?php echo htmlspecialchars($worker['department']); ?>">
                                        <div class="form-check">
                                            <input class="form-check-input worker-checkbox" type="checkbox" name="workers[]" value="<?php echo $worker['id']; ?>" id="worker_<?php echo $worker['id']; ?>">
                                            <label class="form-check-label" for="worker_<?php echo $worker['id']; ?>">
                                                <?php echo htmlspecialchars($worker['full_name']); ?>
                                                <br><small class="text-muted"><?php echo htmlspecialchars($worker['employee_number']); ?> - <?php echo htmlspecialchars($worker['department']); ?></small>
                                            </label>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>

                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> <strong>Note:</strong> This will mark attendance for all selected workers for the entire date range. Please verify your selection before proceeding.
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Security PIN <span class="text-danger">*</span></label>
                            <input type="password" name="pin" class="form-control" maxlength="4" required placeholder="Enter your 4-digit PIN">
                            <small class="text-muted">
                                <i class="bi bi-shield-lock"></i> PIN verification required for bulk attendance marking (HR/Admin function). Set your PIN in Profile.
                            </small>
                        </div>

                        <hr>
                        <div class="d-flex justify-content-between">
                            <a href="calendar.php?month=<?php echo $month; ?>&year=<?php echo $year; ?>" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-success">
                                <i class="bi bi-check-all"></i> Mark Attendance
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
function selectAll() {
    document.querySelectorAll('.worker-checkbox:not([style*="display: none"])').forEach(cb => {
        if (cb.closest('.worker-item').style.display !== 'none') {
            cb.checked = true;
        }
    });
}

function deselectAll() {
    document.querySelectorAll('.worker-checkbox').forEach(cb => cb.checked = false);
}

function filterWorkers() {
    const dept = document.getElementById('deptFilter').value;
    const workers = document.querySelectorAll('.worker-item');

    workers.forEach(worker => {
        if (dept === '' || worker.dataset.department === dept) {
            worker.style.display = 'block';
        } else {
            worker.style.display = 'none';
            worker.querySelector('.worker-checkbox').checked = false;
        }
    });
}
</script>

<?php include '../../includes/footer.php'; ?>
