/**
 * Attendance Tracking - Roster Override and Authorization Details
 * Shows detailed information about roster changes and absence authorizations
 */

// Modal instances
let rosterDetailsModal = null;
let authDetailsModal = null;

/**
 * Initialize modals when DOM is ready
 */
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Bootstrap modals
    const rosterModalEl = document.getElementById('rosterDetailsModal');
    const authModalEl = document.getElementById('authDetailsModal');

    if (rosterModalEl) {
        rosterDetailsModal = new bootstrap.Modal(rosterModalEl);
    }
    if (authModalEl) {
        authDetailsModal = new bootstrap.Modal(authModalEl);
    }
});

/**
 * Show roster override details
 * @param {number} workerId - Worker ID
 * @param {number} locationId - Location ID
 * @param {string} date - Date in YYYY-MM-DD format
 */
async function showRosterDetails(workerId, locationId, date) {
    try {
        const response = await fetch('ajax/get_roster_details.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                worker_id: workerId,
                location_id: locationId,
                date: date
            })
        });

        const result = await response.json();

        if (result.success && result.data) {
            const data = result.data;

            // Populate modal content
            document.getElementById('rosterWorkerName').textContent = data.worker_name || 'Unknown';
            document.getElementById('rosterDate').textContent = formatDate(date);
            document.getElementById('rosterLocation').textContent = data.location_name || 'Unknown';

            // Determine override type text and icon
            let typeText = '';
            let typeIcon = '';
            if (data.override_type === 'leave') {
                const leaveTypes = {
                    'PL': 'Paid Leave',
                    'FO': 'Day Off',
                    'HO': 'Half Day Off',
                    'UL': 'Unpaid Leave',
                    'ML': 'Maternity Leave'
                };
                typeText = 'Leave Authorized - ' + (leaveTypes[data.absence_type] || data.absence_type);
                typeIcon = 'bi bi-calendar-x text-warning';
            } else if (data.override_type === 'add') {
                typeText = 'Added to Shift';
                typeIcon = 'bi bi-plus-circle text-success';
            } else {
                typeText = 'Removed from Shift';
                typeIcon = 'bi bi-dash-circle text-danger';
            }

            document.getElementById('rosterType').textContent = typeText;
            document.getElementById('rosterTypeIcon').className = typeIcon;

            // Shift details (only for 'add' type)
            const shiftDetailsRow = document.getElementById('rosterShiftDetails');
            if (data.override_type === 'add' && data.shift_name) {
                document.getElementById('rosterShiftName').textContent = data.shift_name;
                document.getElementById('rosterShiftTime').textContent =
                    (data.start_time && data.end_time)
                    ? `${formatTime(data.start_time)} – ${formatTime(data.end_time)}`
                    : '-';
                shiftDetailsRow.style.display = '';
            } else {
                shiftDetailsRow.style.display = 'none';
            }

            // Leave details (only for 'leave' type)
            const leaveDetailsRow = document.getElementById('rosterLeaveDetails');
            if (data.override_type === 'leave') {
                let leaveInfo = '';
                if (data.absence_type === 'HO' && data.half_day_period) {
                    if (data.half_day_period === 'first_half') {
                        leaveInfo = 'First Half of Day';
                    } else if (data.half_day_period === 'second_half') {
                        leaveInfo = 'Second Half of Day';
                    } else if (data.half_day_period === 'custom' && data.time_from && data.time_to) {
                        leaveInfo = `Custom: ${formatTime(data.time_from)} – ${formatTime(data.time_to)}`;
                    }
                }
                document.getElementById('rosterLeaveInfo').textContent = leaveInfo || '-';
                leaveDetailsRow.style.display = '';
            } else {
                leaveDetailsRow.style.display = 'none';
            }

            // Authorization details
            document.getElementById('rosterAuthorizedBy').textContent = data.authorized_by_name || 'Unknown';
            document.getElementById('rosterAuthorizedAt').textContent = formatDateTime(data.created_at);

            // Notes
            document.getElementById('rosterNotes').textContent = data.notes || 'No notes provided';

            // Show modal
            if (rosterDetailsModal) {
                rosterDetailsModal.show();
            }
        } else {
            alert('Error: ' + (result.message || 'Failed to load roster details'));
        }
    } catch (error) {
        console.error('Error fetching roster details:', error);
        alert('Error loading roster details: ' + error.message);
    }
}

/**
 * Show authorization details for absence
 * @param {number} workerId - Worker ID
 * @param {string} date - Date in YYYY-MM-DD format
 */
async function showAuthDetails(workerId, date) {
    try {
        const response = await fetch('ajax/get_auth_details.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                worker_id: workerId,
                date: date
            })
        });

        const result = await response.json();

        if (result.success && result.data) {
            const data = result.data;

            // Populate modal content
            document.getElementById('authWorkerName').textContent = data.worker_name || 'Unknown';
            document.getElementById('authDate').textContent = formatDate(date);

            // Absence type with full name
            const absenceTypes = {
                'PL': 'Paid Leave',
                'FO': 'Day Off',
                'HO': 'Half Day Off',
                'UL': 'Unpaid Leave',
                'ML': 'Maternity Leave'
            };
            document.getElementById('authAbsenceType').textContent =
                absenceTypes[data.absence_type] || data.absence_type;
            document.getElementById('authAbsenceTypeBadge').textContent = data.absence_type;

            // Half-day details (only for HO)
            const halfDayRow = document.getElementById('authHalfDayDetails');
            if (data.absence_type === 'HO' && data.half_day_period) {
                let periodText = '';
                if (data.half_day_period === 'first_half') {
                    periodText = 'First Half of Day';
                } else if (data.half_day_period === 'second_half') {
                    periodText = 'Second Half of Day';
                } else if (data.half_day_period === 'custom' && data.time_from && data.time_to) {
                    periodText = `Custom: ${formatTime(data.time_from)} – ${formatTime(data.time_to)}`;
                }
                document.getElementById('authHalfDayPeriod').textContent = periodText;
                halfDayRow.style.display = '';
            } else {
                halfDayRow.style.display = 'none';
            }

            // Authorization details
            document.getElementById('authAuthorizedBy').textContent = data.authorized_by_name || 'Unknown';
            document.getElementById('authAuthorizedAt').textContent = formatDateTime(data.authorization_date);

            // Notes/Reason
            document.getElementById('authNotes').textContent = data.notes || 'No reason provided';

            // Show modal
            if (authDetailsModal) {
                authDetailsModal.show();
            }
        } else {
            alert('Error: ' + (result.message || 'Failed to load authorization details'));
        }
    } catch (error) {
        console.error('Error fetching authorization details:', error);
        alert('Error loading authorization details: ' + error.message);
    }
}

/**
 * Format date to DD/MM/YYYY
 */
function formatDate(dateStr) {
    const date = new Date(dateStr);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    return `${day}/${month}/${year}`;
}

/**
 * Format time to 12-hour format
 */
function formatTime(timeStr) {
    if (!timeStr) return '-';
    const [hours, minutes] = timeStr.split(':');
    const h = parseInt(hours);
    const ampm = h >= 12 ? 'PM' : 'AM';
    const displayHour = h % 12 || 12;
    return `${displayHour}:${minutes} ${ampm}`;
}

/**
 * Format datetime to DD/MM/YYYY HH:MM
 */
function formatDateTime(datetimeStr) {
    if (!datetimeStr) return '-';
    const date = new Date(datetimeStr);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    return `${day}/${month}/${year} ${hours}:${minutes}`;
}
