<?php
/**
 * Simple Attendance Marking AJAX Handler
 * Simplified version without complex dependencies
 */

header('Content-Type: application/json');
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors in JSON response
ini_set('log_errors', 1);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../../../config/database.php';
require_once '../../../includes/database.php';

try {
    // Get database connection
    $pdo = get_database_connection();

    // Debug: Log the request
    error_log('Attendance marking request: worker_id=' . ($_POST['worker_id'] ?? 'none') . ', date=' . ($_POST['date'] ?? 'none') . ', status=' . ($_POST['status'] ?? 'none'));

    // Get POST data
    $worker_id = $_POST['worker_id'] ?? null;
    $date = $_POST['date'] ?? null;
    $status = $_POST['status'] ?? null;
    $check_in = $_POST['check_in'] ?: null;
    $check_out = $_POST['check_out'] ?: null;
    $notes = $_POST['notes'] ?: null;

    // Absence authorization fields
    $authorized_by = $_POST['authorized_by'] ?? null;
    $absence_notes = $_POST['absence_notes'] ?? null;
    $half_day_period = $_POST['half_day_period'] ?? null;
    $time_from = $_POST['time_from'] ?: null;
    $time_to = $_POST['time_to'] ?: null;

    // Validate required fields
    if (!$worker_id || !$date || !$status) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        exit;
    }

    // Validate authorization for absence statuses
    $auth_required_statuses = ['PL', 'FO', 'HO', 'UL', 'ML'];
    if (in_array($status, $auth_required_statuses)) {
        if (!$authorized_by || !$absence_notes) {
            echo json_encode(['success' => false, 'message' => 'Authorization and reason required for absence']);
            exit;
        }
        // For HO, validate half-day period
        if ($status === 'HO' && !$half_day_period) {
            echo json_encode(['success' => false, 'message' => 'Half-day period required for Half Day Off']);
            exit;
        }
    }

    // Handle DELETE action (un-mark attendance)
    if ($status === 'DELETE') {
        $delete_stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ? AND attendance_date = ?");
        $result = $delete_stmt->execute([$worker_id, $date]);

        if ($result) {
            echo json_encode([
                'success' => true,
                'message' => 'Attendance unmarked successfully',
                'action' => 'deleted'
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to unmark attendance']);
        }
        exit;
    }

    // Validate status
    $valid_statuses = ['P', 'PL', 'UL', 'WE', 'HO', 'A', 'FO', 'ML'];
    if (!in_array($status, $valid_statuses)) {
        echo json_encode(['success' => false, 'message' => 'Invalid status code']);
        exit;
    }

    // Calculate hours if check in/out provided
    $hours = 0;
    if ($check_in && $check_out) {
        try {
            $check_in_time = new DateTime($date . ' ' . $check_in);
            $check_out_time = new DateTime($date . ' ' . $check_out);
            $interval = $check_in_time->diff($check_out_time);
            $hours = $interval->h + ($interval->i / 60);
        } catch (Exception $e) {
            // If time calculation fails, just set hours to 0
            $hours = 0;
        }
    }

    // Check if attendance already exists
    $check_stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ? AND attendance_date = ?");
    $check_stmt->execute([$worker_id, $date]);
    $existing = $check_stmt->fetch();

    $user_id = $_SESSION['user_id'] ?? 1; // Default to user 1 if session not set

    if ($existing) {
        // Update existing record
        $update_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "attendance_records
            SET status = ?, check_in = ?, check_out = ?, hours = ?, notes = ?, updated_at = NOW()
            WHERE id = ?");

        $result = $update_stmt->execute([$status, $check_in, $check_out, $hours, $notes, $existing['id']]);

        if ($result) {
            $attendance_id = $existing['id'];

            // Handle absence authorization if needed
            if (in_array($status, $auth_required_statuses)) {
                saveAbsenceAuthorization($pdo, $attendance_id, $worker_id, $date, $status, $authorized_by, $absence_notes, $half_day_period, $time_from, $time_to);
            }

            echo json_encode([
                'success' => true,
                'message' => 'Attendance updated successfully',
                'action' => 'updated'
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to update attendance record']);
        }
    } else {
        // Insert new record
        $insert_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "attendance_records
            (worker_id, attendance_date, status, check_in, check_out, hours, notes, created_by, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())");

        $result = $insert_stmt->execute([$worker_id, $date, $status, $check_in, $check_out, $hours, $notes, $user_id]);

        if ($result) {
            $attendance_id = $pdo->lastInsertId();

            // Handle absence authorization if needed
            if (in_array($status, $auth_required_statuses)) {
                saveAbsenceAuthorization($pdo, $attendance_id, $worker_id, $date, $status, $authorized_by, $absence_notes, $half_day_period, $time_from, $time_to);
            }

            echo json_encode([
                'success' => true,
                'message' => 'Attendance created successfully',
                'action' => 'created',
                'attendance_id' => $attendance_id
            ]);
        } else {
            $error_info = $insert_stmt->errorInfo();
            echo json_encode(['success' => false, 'message' => 'Failed to create attendance record: ' . ($error_info[2] ?? 'Unknown error')]);
        }
    }

} catch (PDOException $e) {
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}

/**
 * Save absence authorization record
 */
function saveAbsenceAuthorization($pdo, $attendance_id, $worker_id, $date, $status, $authorized_by, $notes, $half_day_period, $time_from, $time_to) {
    try {
        // Check if authorization already exists
        $check = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "absence_authorizations WHERE attendance_record_id = ?");
        $check->execute([$attendance_id]);
        $existing_auth = $check->fetch();

        if ($existing_auth) {
            // Update existing authorization
            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "absence_authorizations
                SET authorized_by = ?, half_day_period = ?, time_from = ?, time_to = ?, notes = ?, updated_at = NOW()
                WHERE attendance_record_id = ?");
            $stmt->execute([$authorized_by, $half_day_period, $time_from, $time_to, $notes, $attendance_id]);
        } else {
            // Insert new authorization
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "absence_authorizations
                (attendance_record_id, worker_id, attendance_date, absence_type, authorized_by, half_day_period, time_from, time_to, notes)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$attendance_id, $worker_id, $date, $status, $authorized_by, $half_day_period, $time_from, $time_to, $notes]);
        }

        return true;
    } catch (Exception $e) {
        error_log('Error saving absence authorization: ' . $e->getMessage());
        return false;
    }
}
?>
