<?php
session_start();
require_once '../../../config/database.php';
require_once '../../../includes/database.php';
require_once '../../../includes/session_manager.php';
require_once '../../../includes/module_helper.php';

try {
    require_authentication();
    require_permission(50);

    $worker_id = $_POST['worker_id'] ?? null;
    $date = $_POST['date'] ?? null;
    $status = $_POST['status'] ?? null;
    $check_in = $_POST['check_in'] ?? null;
    $check_out = $_POST['check_out'] ?? null;
    $notes = $_POST['notes'] ?? null;
    $pin = $_POST['pin'] ?? null;
    $edit_reason = $_POST['edit_reason'] ?? null;

    // Validate required fields
    $validation = validate_required([
        'worker_id' => 'Worker',
        'date' => 'Date',
        'status' => 'Status'
    ], 'attendance');

    if (!$validation['valid']) {
        ajax_response(false, null, 'Missing required fields');
    }

    // Handle DELETE action (un-mark attendance)
    if ($status === 'DELETE') {
        $check_stmt = safe_query($pdo,
            "SELECT id FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ? AND attendance_date = ?",
            [$worker_id, $date],
            'attendance',
            'check existing record'
        );
        $existing = $check_stmt ? $check_stmt->fetch() : null;

        if ($existing) {
            $delete_stmt = safe_query($pdo,
                "DELETE FROM " . DB_PREFIX . "attendance_records WHERE id = ?",
                [$existing['id']],
                'attendance',
                'delete attendance record'
            );

            if ($delete_stmt) {
                log_activity("Unmarked attendance for worker {$worker_id} on {$date}", 'attendance', ['worker_id' => $worker_id, 'date' => $date]);
                ajax_response(true, ['action' => 'deleted', 'message' => 'Attendance unmarked successfully']);
            } else {
                ajax_response(false, null, 'Failed to unmark attendance');
            }
        } else {
            ajax_response(false, null, 'No attendance record found to unmark');
        }
        exit;
    }

    // Validate status
    $valid_statuses = ['P', 'PL', 'UL', 'WE', 'HO', 'A', 'FO', 'ML'];
    if (!in_array($status, $valid_statuses)) {
        throw new Exception('Invalid status code');
    }

    // Check if editing past date (requires PIN)
    $today = date('Y-m-d');
    $target_date = new DateTime($date);
    $today_date = new DateTime($today);
    $days_diff = $today_date->diff($target_date)->days;
    $is_past = $target_date < $today_date;

    $pin_required_days = 2; // Configurable

    if ($is_past && $days_diff > $pin_required_days) {
        // PIN verification required
        if (!$pin) {
            throw new Exception('PIN required for editing past attendance');
        }

        // Verify PIN
        $user_stmt = $pdo->prepare("SELECT pin FROM " . DB_PREFIX . "users WHERE id = ?");
        $user_stmt->execute([$_SESSION['user_id']]);
        $user_pin = $user_stmt->fetchColumn();

        if (!$user_pin || $user_pin !== $pin) {
            throw new Exception('Invalid PIN');
        }

        if (!$edit_reason) {
            throw new Exception('Edit reason required for past attendance');
        }
    }

    // Calculate hours if check in/out provided
    $hours = 0;
    if ($check_in && $check_out) {
        $check_in_time = new DateTime($date . ' ' . $check_in);
        $check_out_time = new DateTime($date . ' ' . $check_out);
        $interval = $check_in_time->diff($check_out_time);
        $hours = $interval->h + ($interval->i / 60);
    }

    // Check if attendance already exists
    $check_stmt = safe_query($pdo,
        "SELECT id FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ? AND attendance_date = ?",
        [$worker_id, $date],
        'attendance',
        'check existing record'
    );

    if (!$check_stmt) {
        ajax_response(false, null, 'Database error checking attendance record');
    }

    $existing = $check_stmt->fetch();

    if ($existing) {
        // Update existing record
        $update_stmt = safe_query($pdo,
            "UPDATE " . DB_PREFIX . "attendance_records
            SET status = ?, check_in = ?, check_out = ?, hours = ?, notes = ?, updated_by = ?, updated_at = NOW(), edit_reason = ?
            WHERE id = ?",
            [$status, $check_in, $check_out, $hours, $notes, $_SESSION['user_id'], $edit_reason, $existing['id']],
            'attendance',
            'update attendance record'
        );

        if (!$update_stmt) {
            ajax_response(false, null, 'Failed to update attendance record');
        }

        $attendance_id = $existing['id'];
        $action = 'updated';
        log_activity("Updated attendance for worker {$worker_id} on {$date}", 'attendance', ['worker_id' => $worker_id, 'date' => $date, 'status' => $status]);
    } else {
        // Insert new record
        $insert_stmt = safe_query($pdo,
            "INSERT INTO " . DB_PREFIX . "attendance_records
            (worker_id, attendance_date, status, check_in, check_out, hours, notes, created_by, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())",
            [$worker_id, $date, $status, $check_in, $check_out, $hours, $notes, $_SESSION['user_id']],
            'attendance',
            'create attendance record'
        );

        if (!$insert_stmt) {
            ajax_response(false, null, 'Failed to create attendance record');
        }

        $attendance_id = $pdo->lastInsertId();
        $action = 'created';
        log_activity("Created attendance for worker {$worker_id} on {$date}", 'attendance', ['worker_id' => $worker_id, 'date' => $date, 'status' => $status]);
    }

    // Log edit if it was a past date edit
    if ($is_past && $days_diff > $pin_required_days && $edit_reason) {
        $log_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "attendance_edit_log
            (attendance_id, worker_id, attendance_date, field_changed, old_value, new_value, edit_reason, pin_verified, edited_by, edited_at)
            VALUES (?, ?, ?, 'status', ?, ?, ?, 1, ?, NOW())");
        $log_stmt->execute([$attendance_id, $worker_id, $date, $existing ? 'previous' : 'none', $status, $edit_reason, $_SESSION['user_id']]);
    }

    ajax_response(true, [
        'attendance_id' => $attendance_id,
        'action' => $action,
        'message' => 'Attendance ' . $action . ' successfully'
    ]);

} catch (Exception $e) {
    log_error('ERROR', 'Attendance marking failed', 'attendance', [
        'worker_id' => $worker_id ?? null,
        'date' => $date ?? null,
        'status' => $status ?? null
    ], $e);

    ajax_response(false, null, $e->getMessage());
}
