<?php
/**
 * Get Roster Override Details
 * Fetch detailed information about a roster override
 */
session_start();
require_once '../../../config/database.php';
require_once '../../../includes/database.php';
require_once '../../../includes/session_manager.php';

header('Content-Type: application/json');

// Check authentication
if (!is_authenticated()) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

try {
    $pdo = get_database_connection();

    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input) {
        echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
        exit;
    }

    $worker_id = intval($input['worker_id'] ?? 0);
    $location_id = intval($input['location_id'] ?? 0);
    $date = $input['date'] ?? '';

    // Validate
    if (!$worker_id || !$location_id || !$date) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        exit;
    }

    // Fetch roster override with joined data
    $stmt = $pdo->prepare("SELECT
        ro.*,
        sw.full_name as worker_name,
        wl.name as location_name,
        u.username as authorized_by_name,
        u.full_name as authorized_by_fullname
        FROM " . DB_PREFIX . "roster_overrides ro
        LEFT JOIN " . DB_PREFIX . "sponsored_workers sw ON ro.worker_id = sw.id
        LEFT JOIN " . DB_PREFIX . "work_locations wl ON ro.location_id = wl.id
        LEFT JOIN " . DB_PREFIX . "users u ON ro.authorized_by = u.id
        WHERE ro.worker_id = ?
        AND ro.location_id = ?
        AND ro.roster_date = ?
        LIMIT 1");

    $stmt->execute([$worker_id, $location_id, $date]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($result) {
        // Use full name if available, otherwise username
        $result['authorized_by_name'] = $result['authorized_by_fullname'] ?: $result['authorized_by_name'];

        echo json_encode([
            'success' => true,
            'data' => $result
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Roster override not found'
        ]);
    }

} catch (Exception $e) {
    error_log('Get roster details error: ' . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
?>
