<?php
/**
 * Bulk Roster Override Handler
 * Process multiple roster overrides at once
 */
session_start();
require_once '../../../config/database.php';
require_once '../../../includes/database.php';
require_once '../../../includes/session_manager.php';

header('Content-Type: application/json');

// Check authentication
if (!is_authenticated()) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

// Check permission
if (($_SESSION['permission_level'] ?? 0) < 30) {
    echo json_encode(['success' => false, 'message' => 'Insufficient permissions']);
    exit;
}

try {
    $pdo = get_database_connection();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input) {
        echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
        exit;
    }

    $location_id = intval($input['location_id'] ?? 0);
    $action = $input['action'] ?? '';
    $overrides = $input['overrides'] ?? [];
    $shift_pattern_id = !empty($input['shift_pattern_id']) ? intval($input['shift_pattern_id']) : null;
    $shift_name = $input['shift_name'] ?? null;
    $start_time = $input['start_time'] ?? null;
    $end_time = $input['end_time'] ?? null;
    $absence_type = $input['absence_type'] ?? null;
    $half_day_period = $input['half_day_period'] ?? null;
    $time_from = $input['time_from'] ?? null;
    $time_to = $input['time_to'] ?? null;
    $notes = $input['notes'] ?? null;
    $authorized_by = intval($input['authorized_by'] ?? $_SESSION['user_id']);
    $resolve_conflicts = $input['resolve_conflicts'] ?? false; // User confirmed to resolve conflicts

    // Validate
    if (!$location_id || !$action || empty($overrides)) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        exit;
    }

    if (!in_array($action, ['add', 'remove', 'leave', 'delete'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid action']);
        exit;
    }

    if ($action === 'add' && (!$shift_pattern_id || !$shift_name)) {
        echo json_encode(['success' => false, 'message' => 'Shift details required for add action']);
        exit;
    }

    if ($action === 'leave') {
        if (!$absence_type || !in_array($absence_type, ['PL', 'FO', 'HO', 'UL', 'ML'])) {
            echo json_encode(['success' => false, 'message' => 'Valid leave type required']);
            exit;
        }
        if ($absence_type === 'HO' && !$half_day_period) {
            echo json_encode(['success' => false, 'message' => 'Half-day period required for Half Day Off']);
            exit;
        }
        if (!$notes) {
            echo json_encode(['success' => false, 'message' => 'Reason required for leave authorization']);
            exit;
        }
    }

    // Check for conflicts if adding to shift (unless user already confirmed to resolve)
    $conflicts = [];
    if ($action === 'add' && !$resolve_conflicts) {
        // Get shift type to determine if we need to check next day
        $is_night_shift = false;
        if ($shift_pattern_id) {
            $shift_check = $pdo->prepare("SELECT shift_type, shift_name FROM " . DB_PREFIX . "location_shift_patterns WHERE id = ?");
            $shift_check->execute([$shift_pattern_id]);
            $shift_info = $shift_check->fetch();
            $is_night_shift = ($shift_info && $shift_info['shift_type'] === 'night');

            // Debug logging
            error_log("CONFLICT CHECK: Adding shift_pattern_id=$shift_pattern_id, shift_name={$shift_info['shift_name']}, is_night_shift=" . ($is_night_shift ? 'YES' : 'NO'));
        }

        foreach ($overrides as $override) {
            $worker_id = intval($override['worker_id']);
            $date = $override['date'];

            if (!$worker_id || !$date) continue;

            // Check if worker has shift at different location on this date
            // Check both regular assignments and overrides
            $day_abbr = date('D', strtotime($date));

            // If this is a night shift, also check next day for conflicts
            $dates_to_check = [$date];
            if ($is_night_shift) {
                $next_day = date('Y-m-d', strtotime($date . ' +1 day'));
                $dates_to_check[] = $next_day;
                error_log("NIGHT SHIFT DETECTED: Checking conflicts for worker $worker_id on dates: " . implode(', ', $dates_to_check));
            }

            // Check all dates (current date and next day for night shifts)
            foreach ($dates_to_check as $check_date) {
                $check_day_abbr = date('D', strtotime($check_date));
                $is_next_day = ($check_date !== $date);

                // Check regular assignments at ALL locations (worker can't be at multiple locations same time)
                $stmt = $pdo->prepare("SELECT
                    wla.location_id,
                    wl.name as location_name,
                    sp.shift_name,
                    sp.shift_type,
                    sp.start_time,
                    sp.end_time,
                    wla.days_of_week
                    FROM " . DB_PREFIX . "worker_location_assignments wla
                    INNER JOIN " . DB_PREFIX . "work_locations wl ON wla.location_id = wl.id
                    LEFT JOIN " . DB_PREFIX . "location_shift_patterns sp ON wla.shift_pattern_id = sp.id
                    WHERE wla.worker_id = ?
                    AND (wla.end_date IS NULL OR wla.end_date >= ?)");
                $stmt->execute([$worker_id, $check_date]);

                foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $assignment) {
                    $days = array_map('trim', explode(',', $assignment['days_of_week'] ?? ''));
                    error_log("Checking regular assignment: location_id={$assignment['location_id']}, days=" . implode(',', $days) . ", check_day_abbr=$check_day_abbr");

                    if (in_array($check_day_abbr, $days)) {
                        $conflict_msg = $is_next_day
                            ? $assignment['shift_name'] . ' (next day after night shift)'
                            : $assignment['shift_name'];

                        error_log("CONFLICT FOUND: {$conflict_msg} at {$assignment['location_name']} on {$check_date}");

                        $conflicts[] = [
                            'worker_id' => $worker_id,
                            'worker_name' => $override['worker_name'],
                            'date' => $check_date,
                            'conflict_location_id' => $assignment['location_id'],
                            'conflict_location_name' => $assignment['location_name'],
                            'conflict_shift_name' => $conflict_msg,
                            'conflict_type' => 'regular',
                            'is_next_day_conflict' => $is_next_day
                        ];
                    }
                }

                // Check roster overrides at ALL locations (worker can't be at multiple locations same time)
                $stmt = $pdo->prepare("SELECT
                    ro.location_id,
                    wl.name as location_name,
                    ro.shift_name
                    FROM " . DB_PREFIX . "roster_overrides ro
                    INNER JOIN " . DB_PREFIX . "work_locations wl ON ro.location_id = wl.id
                    WHERE ro.worker_id = ?
                    AND ro.roster_date = ?
                    AND ro.override_type = 'add'");
                $stmt->execute([$worker_id, $check_date]);

                $override_conflict = $stmt->fetch();
                if ($override_conflict) {
                    $conflict_msg = $is_next_day
                        ? $override_conflict['shift_name'] . ' (next day after night shift)'
                        : $override_conflict['shift_name'];

                    $conflicts[] = [
                        'worker_id' => $worker_id,
                        'worker_name' => $override['worker_name'],
                        'date' => $check_date,
                        'conflict_location_id' => $override_conflict['location_id'],
                        'conflict_location_name' => $override_conflict['location_name'],
                        'conflict_shift_name' => $conflict_msg,
                        'conflict_type' => 'override',
                        'is_next_day_conflict' => $is_next_day
                    ];
                }
            }
        }

        // If conflicts found, return them for user confirmation
        if (!empty($conflicts)) {
            $conflict_debug = count($conflicts) . " conflict(s) found";
            if ($is_night_shift) {
                $conflict_debug .= " [NIGHT SHIFT - checked next day]";
            }

            echo json_encode([
                'success' => false,
                'has_conflicts' => true,
                'conflicts' => $conflicts,
                'message' => 'Worker(s) already scheduled at other location(s). ' . $conflict_debug
            ]);
            exit;
        }
    }

    // If resolving conflicts, remove worker from other locations first
    if ($action === 'add' && $resolve_conflicts) {
        // Re-check if this is a night shift
        $is_night_shift = false;
        if ($shift_pattern_id) {
            $shift_check = $pdo->prepare("SELECT shift_type FROM " . DB_PREFIX . "location_shift_patterns WHERE id = ?");
            $shift_check->execute([$shift_pattern_id]);
            $shift_info = $shift_check->fetch();
            $is_night_shift = ($shift_info && $shift_info['shift_type'] === 'night');
        }

        foreach ($overrides as $override) {
            $worker_id = intval($override['worker_id']);
            $date = $override['date'];

            if (!$worker_id || !$date) continue;

            // If night shift, also handle next day
            $dates_to_clear = [$date];
            if ($is_night_shift) {
                $next_day = date('Y-m-d', strtotime($date . ' +1 day'));
                $dates_to_clear[] = $next_day;
            }

            // Find and clear conflicts for all relevant dates
            foreach ($dates_to_clear as $clear_date) {
                $day_abbr = date('D', strtotime($clear_date));
                $is_next_day = ($clear_date !== $date);

                // Find all locations where worker is scheduled on this date
                $conflict_locations = [];

                // Check roster overrides at ALL locations (worker can't be at multiple places same time)
                $stmt = $pdo->prepare("SELECT DISTINCT location_id
                    FROM " . DB_PREFIX . "roster_overrides
                    WHERE worker_id = ?
                    AND roster_date = ?
                    AND override_type = 'add'");
                $stmt->execute([$worker_id, $clear_date]);
                foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
                    $conflict_locations[] = $row['location_id'];
                }

                // Check regular assignments at ALL locations (worker can't be at multiple places same time)
                $stmt = $pdo->prepare("SELECT DISTINCT wla.location_id, wla.days_of_week
                    FROM " . DB_PREFIX . "worker_location_assignments wla
                    WHERE wla.worker_id = ?
                    AND (wla.end_date IS NULL OR wla.end_date >= ?)");
                $stmt->execute([$worker_id, $clear_date]);
                foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $assignment) {
                    $days = array_map('trim', explode(',', $assignment['days_of_week'] ?? ''));
                    if (in_array($day_abbr, $days)) {
                        $conflict_locations[] = $assignment['location_id'];
                    }
                }

                // Create removal overrides for all conflict locations (except target location for same day)
                foreach (array_unique($conflict_locations) as $conflict_location_id) {
                    // For same day: don't remove from target location (we're adding there)
                    // For next day: remove from ALL locations (including target, if scheduled)
                    if (!$is_next_day && $conflict_location_id == $location_id) {
                        continue; // Skip - we're adding to this location on same day
                    }

                    $note = $is_next_day
                        ? 'Removed due to night shift on previous day'
                        : 'Moved to another location via roster management';

                    $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "roster_overrides
                        (worker_id, location_id, roster_date, override_type, notes, authorized_by)
                        VALUES (?, ?, ?, 'remove', ?, ?)
                        ON DUPLICATE KEY UPDATE override_type = 'remove', notes = ?, authorized_by = ?");
                    $stmt->execute([
                        $worker_id,
                        $conflict_location_id,
                        $clear_date,
                        $note,
                        $authorized_by,
                        $note,
                        $authorized_by
                    ]);
                }
            }
        }
    }

    // Process each override
    $success_count = 0;
    $error_count = 0;
    $last_error = null;

    foreach ($overrides as $override) {
        $worker_id = intval($override['worker_id']);
        $date = $override['date'];

        if (!$worker_id || !$date) {
            $error_count++;
            continue;
        }

        try {
            // Check if override exists
            $check = $pdo->prepare("SELECT id, shift_pattern_id, override_type FROM " . DB_PREFIX . "roster_overrides
                WHERE worker_id = ? AND location_id = ? AND roster_date = ?");
            $check->execute([$worker_id, $location_id, $date]);
            $existing = $check->fetch();

            // Handle delete action - remove the override entirely
            if ($action === 'delete') {
                if ($existing) {
                    $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "roster_overrides WHERE id = ?");
                    $stmt->execute([$existing['id']]);
                    $success_count++;
                } else {
                    // No override to delete, but don't count as error
                    $success_count++;
                }
                continue;
            }

            if ($existing) {
                // Check if we're MODIFYING to a night shift - need to check next-day conflicts
                if ($action === 'add' && $shift_pattern_id && $shift_pattern_id != $existing['shift_pattern_id'] && !$resolve_conflicts) {
                    $shift_check = $pdo->prepare("SELECT shift_type FROM " . DB_PREFIX . "location_shift_patterns WHERE id = ?");
                    $shift_check->execute([$shift_pattern_id]);
                    $new_shift_info = $shift_check->fetch();

                    if ($new_shift_info && $new_shift_info['shift_type'] === 'night') {
                        // Check next day for conflicts
                        $next_day = date('Y-m-d', strtotime($date . ' +1 day'));
                        $next_day_abbr = date('D', strtotime($next_day));
                        $modification_conflicts = [];

                        // Check regular assignments
                        $stmt = $pdo->prepare("SELECT wla.location_id, wl.name as location_name, sp.shift_name, wla.days_of_week
                            FROM " . DB_PREFIX . "worker_location_assignments wla
                            INNER JOIN " . DB_PREFIX . "work_locations wl ON wla.location_id = wl.id
                            LEFT JOIN " . DB_PREFIX . "location_shift_patterns sp ON wla.shift_pattern_id = sp.id
                            WHERE wla.worker_id = ?
                            AND (wla.end_date IS NULL OR wla.end_date >= ?)");
                        $stmt->execute([$worker_id, $next_day]);

                        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $assignment) {
                            $days = array_map('trim', explode(',', $assignment['days_of_week'] ?? ''));
                            if (in_array($next_day_abbr, $days)) {
                                $modification_conflicts[] = [
                                    'worker_id' => $worker_id,
                                    'worker_name' => $override['worker_name'],
                                    'date' => $next_day,
                                    'conflict_location_id' => $assignment['location_id'],
                                    'conflict_location_name' => $assignment['location_name'],
                                    'conflict_shift_name' => $assignment['shift_name'] . ' (next day after night shift)',
                                    'conflict_type' => 'regular',
                                    'is_next_day_conflict' => true
                                ];
                            }
                        }

                        // Check roster overrides for next day
                        $stmt = $pdo->prepare("SELECT ro.location_id, wl.name as location_name, ro.shift_name
                            FROM " . DB_PREFIX . "roster_overrides ro
                            INNER JOIN " . DB_PREFIX . "work_locations wl ON ro.location_id = wl.id
                            WHERE ro.worker_id = ? AND ro.roster_date = ? AND ro.override_type = 'add'");
                        $stmt->execute([$worker_id, $next_day]);

                        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $next_override) {
                            $modification_conflicts[] = [
                                'worker_id' => $worker_id,
                                'worker_name' => $override['worker_name'],
                                'date' => $next_day,
                                'conflict_location_id' => $next_override['location_id'],
                                'conflict_location_name' => $next_override['location_name'],
                                'conflict_shift_name' => $next_override['shift_name'] . ' (next day after night shift)',
                                'conflict_type' => 'override',
                                'is_next_day_conflict' => true
                            ];
                        }

                        // If conflicts found, return them
                        if (!empty($modification_conflicts)) {
                            echo json_encode([
                                'success' => false,
                                'has_conflicts' => true,
                                'conflicts' => $modification_conflicts,
                                'message' => 'Worker has shifts scheduled next day after night shift'
                            ]);
                            exit;
                        }
                    }
                }

                // Check if we're changing from leave to shift - need to remove attendance marking
                $check_type = $pdo->prepare("SELECT override_type FROM " . DB_PREFIX . "roster_overrides WHERE id = ?");
                $check_type->execute([$existing['id']]);
                $old_type = $check_type->fetchColumn();

                // Update existing - different fields for leave vs add/remove
                if ($action === 'leave') {
                    // For leave: clear shift fields, set absence fields
                    $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "roster_overrides
                        SET override_type = 'leave',
                            shift_pattern_id = NULL, shift_name = NULL,
                            start_time = NULL, end_time = NULL,
                            absence_type = ?, half_day_period = ?, time_from = ?, time_to = ?,
                            notes = ?, authorized_by = ?,
                            updated_at = CURRENT_TIMESTAMP
                        WHERE id = ?");
                    $stmt->execute([
                        $absence_type,
                        $half_day_period,
                        $time_from,
                        $time_to,
                        $notes,
                        $authorized_by,
                        $existing['id']
                    ]);
                } else {
                    // For add/remove: set shift fields, clear absence fields
                    $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "roster_overrides
                        SET override_type = ?, shift_pattern_id = ?, shift_name = ?,
                            start_time = ?, end_time = ?,
                            absence_type = NULL, half_day_period = NULL, time_from = NULL, time_to = NULL,
                            notes = ?, authorized_by = ?,
                            updated_at = CURRENT_TIMESTAMP
                        WHERE id = ?");
                    $stmt->execute([
                        $action,
                        $shift_pattern_id,
                        $shift_name,
                        $start_time,
                        $end_time,
                        $notes,
                        $authorized_by,
                        $existing['id']
                    ]);

                    // If changing from leave to shift/remove, delete attendance records
                    if ($old_type === 'leave') {
                        // Delete attendance record (unmarking)
                        $del_att = $pdo->prepare("DELETE FROM " . DB_PREFIX . "attendance_records
                            WHERE worker_id = ? AND attendance_date = ?");
                        $del_att->execute([$worker_id, $date]);

                        // Delete absence authorization
                        $del_auth = $pdo->prepare("DELETE FROM " . DB_PREFIX . "absence_authorizations
                            WHERE worker_id = ? AND attendance_date = ?");
                        $del_auth->execute([$worker_id, $date]);
                    }
                }
            } else {
                // Insert new
                $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "roster_overrides
                    (worker_id, location_id, roster_date, override_type, shift_pattern_id,
                     shift_name, start_time, end_time,
                     absence_type, half_day_period, time_from, time_to,
                     notes, authorized_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $worker_id,
                    $location_id,
                    $date,
                    $action,
                    $shift_pattern_id,
                    $shift_name,
                    $start_time,
                    $end_time,
                    $absence_type,
                    $half_day_period,
                    $time_from,
                    $time_to,
                    $notes,
                    $authorized_by
                ]);
            }

            // If this is a leave authorization from roster, automatically create attendance record
            if ($action === 'leave') {
                // Check if attendance record already exists
                $att_check = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "attendance_records
                    WHERE worker_id = ? AND attendance_date = ?");
                $att_check->execute([$worker_id, $date]);
                $existing_att = $att_check->fetch();

                // For attendance records, only include check_in/check_out if they have values
                if ($existing_att) {
                    // Update existing attendance record
                    if ($time_from && $time_to) {
                        $att_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "attendance_records
                            SET status = ?, check_in = ?, check_out = ?, notes = ?,
                                updated_by = ?, updated_at = CURRENT_TIMESTAMP
                            WHERE id = ?");
                        $att_stmt->execute([
                            $absence_type,
                            $time_from,
                            $time_to,
                            $notes,
                            $authorized_by,
                            $existing_att['id']
                        ]);
                    } else {
                        $att_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "attendance_records
                            SET status = ?, notes = ?,
                                updated_by = ?, updated_at = CURRENT_TIMESTAMP
                            WHERE id = ?");
                        $att_stmt->execute([
                            $absence_type,
                            $notes,
                            $authorized_by,
                            $existing_att['id']
                        ]);
                    }
                } else {
                    // Create new attendance record (auto-marked from roster)
                    if ($time_from && $time_to) {
                        $att_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "attendance_records
                            (worker_id, attendance_date, status, check_in, check_out,
                             notes, created_by, created_at)
                            VALUES (?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)");
                        $att_stmt->execute([
                            $worker_id,
                            $date,
                            $absence_type,
                            $time_from,
                            $time_to,
                            $notes,
                            $authorized_by
                        ]);
                    } else {
                        $att_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "attendance_records
                            (worker_id, attendance_date, status, notes, created_by, created_at)
                            VALUES (?, ?, ?, ?, ?, CURRENT_TIMESTAMP)");
                        $att_stmt->execute([
                            $worker_id,
                            $date,
                            $absence_type,
                            $notes,
                            $authorized_by
                        ]);
                    }
                }

                // Also ensure absence authorization exists
                $auth_check = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "absence_authorizations
                    WHERE worker_id = ? AND attendance_date = ?");
                $auth_check->execute([$worker_id, $date]);
                $existing_auth = $auth_check->fetch();

                if (!$existing_auth) {
                    // Create absence authorization
                    $auth_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "absence_authorizations
                        (worker_id, attendance_date, absence_type, half_day_period,
                         time_from, time_to, reason, authorized_by, authorized_at)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)");
                    $auth_stmt->execute([
                        $worker_id,
                        $date,
                        $absence_type,
                        $half_day_period,
                        $time_from,
                        $time_to,
                        $notes,
                        $authorized_by
                    ]);
                }
            }

            $success_count++;
        } catch (Exception $e) {
            $error_msg = 'Error processing override for worker ' . $worker_id . ' on ' . $date . ': ' . $e->getMessage();
            error_log($error_msg);
            error_log('Stack trace: ' . $e->getTraceAsString());
            $error_count++;
            $last_error = $e->getMessage(); // Store last error for response
        }
    }

    if ($success_count > 0) {
        // Build debug message showing what happened
        $debug_info = [];
        if (isset($is_night_shift)) {
            if ($is_night_shift) {
                $debug_info[] = "✓ NIGHT SHIFT detected";
                $debug_info[] = "✓ Next day checked";
            } else {
                $debug_info[] = "ℹ NOT night shift";
            }
        }
        if ($action === 'add') {
            $debug_info[] = "Action: ADD";
        }
        if (!empty($conflicts) && $resolve_conflicts) {
            $debug_info[] = "⚠ Conflicts resolved";
        }

        $debug_msg = !empty($debug_info) ? " [DEBUG: " . implode(", ", $debug_info) . "]" : "";

        echo json_encode([
            'success' => true,
            'message' => "$success_count roster override(s) saved successfully" . ($error_count > 0 ? ", $error_count failed" : "") . $debug_msg,
            'success_count' => $success_count,
            'error_count' => $error_count
        ]);
    } else {
        $error_message = 'Failed to save roster overrides';
        if ($last_error) {
            $error_message .= ': ' . $last_error;
        }
        echo json_encode([
            'success' => false,
            'message' => $error_message,
            'error_count' => $error_count
        ]);
    }

} catch (Exception $e) {
    error_log('Bulk roster override error: ' . $e->getMessage());
    error_log('Stack trace: ' . $e->getTraceAsString());
    echo json_encode(['success' => false, 'message' => 'Server error: ' . $e->getMessage()]);
}
?>
