<?php
/**
 * AJAX Handler - Add Roster Override
 */
session_start();
require_once '../../../config/database.php';
require_once '../../../includes/database.php';
require_once '../../../includes/session_manager.php';

header('Content-Type: application/json');

// Check authentication
if (!is_authenticated()) {
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

// Check permission
if (($_SESSION['permission_level'] ?? 0) < 30) {
    echo json_encode(['success' => false, 'message' => 'Insufficient permissions']);
    exit;
}

try {
    $pdo = get_database_connection();

    // Validate required fields
    $required = ['location_id', 'roster_date', 'worker_id', 'override_type'];
    foreach ($required as $field) {
        if (empty($_POST[$field])) {
            echo json_encode(['success' => false, 'message' => 'Missing required field: ' . $field]);
            exit;
        }
    }

    $location_id = intval($_POST['location_id']);
    $roster_date = $_POST['roster_date'];
    $worker_id = intval($_POST['worker_id']);
    $override_type = $_POST['override_type'];
    $shift_pattern_id = !empty($_POST['shift_pattern_id']) ? intval($_POST['shift_pattern_id']) : null;
    $shift_name = $_POST['shift_name'] ?? null;
    $start_time = $_POST['start_time'] ?? null;
    $end_time = $_POST['end_time'] ?? null;
    $notes = $_POST['notes'] ?? null;

    // Determine who authorized this (admin can authorize as another user)
    $authorized_by = $_SESSION['user_id'];
    if ($_SESSION['permission_level'] >= 90 && !empty($_POST['authorize_as'])) {
        $authorized_by = intval($_POST['authorize_as']);
    }

    // Validate override type
    if (!in_array($override_type, ['add', 'remove'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid override type']);
        exit;
    }

    // If adding a shift, require shift details
    if ($override_type === 'add') {
        if (empty($shift_name) && empty($shift_pattern_id)) {
            echo json_encode(['success' => false, 'message' => 'Shift details required for ADD override']);
            exit;
        }

        // Check if this is a night shift - if so, check next day for conflicts
        if ($shift_pattern_id) {
            $shift_check = $pdo->prepare("SELECT shift_type FROM " . DB_PREFIX . "location_shift_patterns WHERE id = ?");
            $shift_check->execute([$shift_pattern_id]);
            $shift_info = $shift_check->fetch();

            if ($shift_info && $shift_info['shift_type'] === 'night') {
                // Check next day for conflicts
                $next_day = date('Y-m-d', strtotime($roster_date . ' +1 day'));
                $next_day_abbr = date('D', strtotime($next_day));
                $conflicts = [];

                // Check regular assignments at ALL locations (including this one)
                $stmt = $pdo->prepare("SELECT wla.location_id, wl.name as location_name, sp.shift_name, wla.days_of_week
                    FROM " . DB_PREFIX . "worker_location_assignments wla
                    INNER JOIN " . DB_PREFIX . "work_locations wl ON wla.location_id = wl.id
                    LEFT JOIN " . DB_PREFIX . "location_shift_patterns sp ON wla.shift_pattern_id = sp.id
                    WHERE wla.worker_id = ?
                    AND (wla.end_date IS NULL OR wla.end_date >= ?)");
                $stmt->execute([$worker_id, $next_day]);

                foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $assignment) {
                    $days = array_map('trim', explode(',', $assignment['days_of_week'] ?? ''));
                    if (in_array($next_day_abbr, $days)) {
                        $conflicts[] = "Worker has {$assignment['shift_name']} at {$assignment['location_name']} on next day ({$next_day})";
                    }
                }

                // Check roster overrides for next day at ALL locations
                $stmt = $pdo->prepare("SELECT ro.location_id, wl.name as location_name, ro.shift_name
                    FROM " . DB_PREFIX . "roster_overrides ro
                    INNER JOIN " . DB_PREFIX . "work_locations wl ON ro.location_id = wl.id
                    WHERE ro.worker_id = ? AND ro.roster_date = ? AND ro.override_type = 'add'");
                $stmt->execute([$worker_id, $next_day]);

                foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $override) {
                    $conflicts[] = "Worker has {$override['shift_name']} at {$override['location_name']} on next day ({$next_day})";
                }

                // If conflicts found, reject the operation
                if (!empty($conflicts)) {
                    echo json_encode([
                        'success' => false,
                        'message' => 'Cannot add night shift - worker has shifts on next day. ' . implode('; ', $conflicts)
                    ]);
                    exit;
                }
            }
        }
    }

    // Check if override already exists
    $check_stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "roster_overrides
                                  WHERE worker_id = ? AND location_id = ? AND roster_date = ?");
    $check_stmt->execute([$worker_id, $location_id, $roster_date]);

    if ($check_stmt->fetch()) {
        // Update existing override
        $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "roster_overrides
                               SET override_type = ?, shift_pattern_id = ?, shift_name = ?,
                                   start_time = ?, end_time = ?, notes = ?, authorized_by = ?,
                                   updated_at = CURRENT_TIMESTAMP
                               WHERE worker_id = ? AND location_id = ? AND roster_date = ?");
        $stmt->execute([
            $override_type,
            $shift_pattern_id,
            $shift_name,
            $start_time,
            $end_time,
            $notes,
            $authorized_by,
            $worker_id,
            $location_id,
            $roster_date
        ]);
    } else {
        // Insert new override
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "roster_overrides
                               (worker_id, location_id, roster_date, override_type, shift_pattern_id,
                                shift_name, start_time, end_time, notes, authorized_by)
                               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $worker_id,
            $location_id,
            $roster_date,
            $override_type,
            $shift_pattern_id,
            $shift_name,
            $start_time,
            $end_time,
            $notes,
            $authorized_by
        ]);
    }

    echo json_encode(['success' => true, 'message' => 'Roster override saved successfully']);

} catch (Exception $e) {
    error_log('Error adding roster override: ' . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
