<?php
/**
 * Attendance Module - Add Attendance Record
 * Allows manual entry of attendance for a worker on a specific date
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
    require_permission(70);
}

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

$page_title = 'Add Attendance Record';
$current_page = 'attendance';
$errors = [];

// Get database connection
$pdo = get_database_connection();

// Get all active workers
$workers = [];
try {
    $stmt = $pdo->query("SELECT id, full_name, employee_number, job_title FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' ORDER BY full_name");
    $workers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $errors[] = "Error loading workers: " . $e->getMessage();
}

// Get all active locations
$locations = [];
try {
    $stmt = $pdo->query("SELECT id, name, type FROM " . DB_PREFIX . "work_locations WHERE status = 'active' ORDER BY name");
    $locations = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Locations table may not exist yet
}

// Status codes: P=Present, PL=Paid Leave, UL=Unpaid Leave, FO=Furlough, HO=Holiday, WE=Weekend, A=Absent
$attendance_statuses = [
    'P' => 'Present',
    'PL' => 'Paid Leave',
    'UL' => 'Unpaid Leave',
    'FO' => 'Furlough',
    'HO' => 'Holiday',
    'WE' => 'Weekend',
    'A' => 'Absent'
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $worker_id = (int)($_POST['worker_id'] ?? 0);
    $date = $_POST['date'] ?? '';
    $status = $_POST['status'] ?? 'P';
    $check_in = $_POST['check_in'] ?? null;
    $check_out = $_POST['check_out'] ?? null;
    $location_id = $_POST['location_id'] ?: null;
    $notes = trim($_POST['notes'] ?? '');

    // Validation
    if ($worker_id <= 0) {
        $errors[] = "Please select a worker";
    }
    if (empty($date)) {
        $errors[] = "Date is required";
    }
    if (empty($status)) {
        $errors[] = "Status is required";
    }

    // Calculate hours and flags
    $hours = 0;
    $is_late = 0;
    $is_early_leave = 0;
    $overtime_hours = 0;

    if ($status === 'P' && $check_in && $check_out) {
        // Calculate hours worked
        $check_in_time = strtotime($check_in);
        $check_out_time = strtotime($check_out);

        if ($check_out_time > $check_in_time) {
            $hours = round(($check_out_time - $check_in_time) / 3600, 2);

            // Check if late (after 9:00 AM)
            $nine_am = strtotime(date('Y-m-d', strtotime($date)) . ' 09:00:00');
            if ($check_in_time > $nine_am) {
                $is_late = 1;
            }

            // Check if early leave (before 5:00 PM)
            $five_pm = strtotime(date('Y-m-d', strtotime($date)) . ' 17:00:00');
            if ($check_out_time < $five_pm && $hours < 8) {
                $is_early_leave = 1;
            }

            // Calculate overtime (over 8 hours)
            if ($hours > 8) {
                $overtime_hours = round($hours - 8, 2);
            }
        }
    }

    if (empty($errors)) {
        try {
            // Check if attendance record already exists for this worker and date
            $stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "attendance_records WHERE worker_id = ? AND date = ?");
            $stmt->execute([$worker_id, $date]);
            $existing = $stmt->fetch();

            if ($existing) {
                $errors[] = "Attendance record already exists for this worker on this date. Please edit the existing record instead.";
            } else {
                // Insert new attendance record
                $stmt = $pdo->prepare("
                    INSERT INTO " . DB_PREFIX . "attendance_records
                    (worker_id, date, status, check_in, check_out, hours, is_late, is_early_leave,
                     overtime_hours, notes, location_id, created_by, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                ");

                $stmt->execute([
                    $worker_id, $date, $status, $check_in, $check_out, $hours,
                    $is_late, $is_early_leave, $overtime_hours, $notes, $location_id, $current_user['id']
                ]);

                $_SESSION['success'] = 'Attendance record added successfully';

                // Redirect based on how the page was accessed
                if (defined('MODULE_EMBEDDED')) {
                    header('Location: ../../m/attendance/list?success=1');
                } else {
                    header('Location: list.php?success=1');
                }
                exit;
            }
        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Only include header if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<style>
.form-section {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
}

.form-section-title {
    font-size: 16px;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 2px solid var(--primary-color);
    display: flex;
    align-items: center;
    gap: 8px;
}
</style>

<div class="row justify-content-center">
    <div class="col-12 col-lg-10">
        <div class="modern-card">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-calendar-check"></i> Add Attendance Record
                </h5>
            </div>
            <div class="card-body-modern">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <i class="bi bi-exclamation-triangle"></i>
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>

                <form method="POST" id="attendanceForm">
                    <!-- Basic Information -->
                    <div class="form-section">
                        <div class="form-section-title">
                            <i class="bi bi-info-circle"></i> Basic Information
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Worker <span class="text-danger">*</span></label>
                                <select name="worker_id" class="form-select" required>
                                    <option value="">-- Select Worker --</option>
                                    <?php foreach ($workers as $worker): ?>
                                        <option value="<?php echo $worker['id']; ?>"
                                                <?php echo ($_POST['worker_id'] ?? '') == $worker['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($worker['full_name']); ?>
                                            (<?php echo htmlspecialchars($worker['employee_number']); ?>)
                                            <?php if ($worker['job_title']): ?>
                                                - <?php echo htmlspecialchars($worker['job_title']); ?>
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3 mb-3">
                                <label class="form-label">Date <span class="text-danger">*</span></label>
                                <input type="date" name="date" class="form-control" required
                                       value="<?php echo htmlspecialchars($_POST['date'] ?? date('Y-m-d')); ?>">
                            </div>
                            <div class="col-md-3 mb-3">
                                <label class="form-label">Status <span class="text-danger">*</span></label>
                                <select name="status" class="form-select" id="statusSelect" required>
                                    <?php foreach ($attendance_statuses as $code => $label): ?>
                                        <option value="<?php echo $code; ?>"
                                                <?php echo ($_POST['status'] ?? 'P') === $code ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Time Details (shown only for Present status) -->
                    <div class="form-section" id="timeSection">
                        <div class="form-section-title">
                            <i class="bi bi-clock"></i> Time Details
                        </div>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Check In Time</label>
                                <input type="time" name="check_in" class="form-control"
                                       value="<?php echo htmlspecialchars($_POST['check_in'] ?? '09:00'); ?>">
                                <small class="text-muted">Leave empty if not applicable</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Check Out Time</label>
                                <input type="time" name="check_out" class="form-control"
                                       value="<?php echo htmlspecialchars($_POST['check_out'] ?? '17:00'); ?>">
                                <small class="text-muted">Leave empty if not applicable</small>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Location</label>
                                <select name="location_id" class="form-select">
                                    <option value="">-- Not Specified --</option>
                                    <?php foreach ($locations as $location): ?>
                                        <option value="<?php echo $location['id']; ?>"
                                                <?php echo ($_POST['location_id'] ?? '') == $location['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($location['name']); ?>
                                            <?php if ($location['type']): ?>
                                                (<?php echo ucfirst($location['type']); ?>)
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Notes -->
                    <div class="form-section">
                        <div class="form-section-title">
                            <i class="bi bi-journal-text"></i> Notes
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Notes / Remarks</label>
                            <textarea name="notes" class="form-control" rows="3"
                                      placeholder="Additional comments or reasons for absence..."><?php echo htmlspecialchars($_POST['notes'] ?? ''); ?></textarea>
                        </div>
                    </div>

                    <hr class="my-4">
                    <div class="d-flex justify-content-between align-items-center">
                        <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/attendance/list' : 'list.php'; ?>"
                           class="btn btn-secondary">
                            <i class="bi bi-x-circle"></i> Cancel
                        </a>
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-circle"></i> Add Attendance Record
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
// Show/hide time section based on status
document.getElementById('statusSelect')?.addEventListener('change', function() {
    const timeSection = document.getElementById('timeSection');
    if (this.value === 'P') {
        timeSection.style.display = 'block';
    } else {
        timeSection.style.display = 'none';
    }
});

// Initial state
document.addEventListener('DOMContentLoaded', function() {
    const statusSelect = document.getElementById('statusSelect');
    const timeSection = document.getElementById('timeSection');
    if (statusSelect && timeSection) {
        if (statusSelect.value !== 'P') {
            timeSection.style.display = 'none';
        }
    }
});
</script>

<?php
// Only include footer if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
