<?php
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication('../../login.php');

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

// Check permission level
if (($_SESSION['permission_level'] ?? 0) < 50) {
    die('Access denied. You do not have permission to view absence alerts.');
}

$page_title = 'Absence Alerts - Home Office Reporting';

// Get all active absence alerts
$alerts_sql = "SELECT aa.*,
    CONCAT(sw.first_name, ' ', sw.last_name) as worker_name,
    sw.employee_number,
    sw.department,
    DATEDIFF(CURDATE(), aa.absence_start_date) as days_absent
    FROM " . DB_PREFIX . "attendance_absence_alerts aa
    JOIN " . DB_PREFIX . "sponsored_workers sw ON aa.worker_id = sw.id
    WHERE aa.status != 'resolved'
    ORDER BY aa.consecutive_days DESC, aa.absence_start_date ASC";

$alerts = $pdo->query($alerts_sql)->fetchAll(PDO::FETCH_ASSOC);

// Separate into critical (10+ days) and warnings (5-9 days)
$critical_alerts = [];
$warning_alerts = [];

foreach ($alerts as $alert) {
    if ($alert['consecutive_days'] >= 10) {
        $critical_alerts[] = $alert;
    } else if ($alert['consecutive_days'] >= 5) {
        $warning_alerts[] = $alert;
    }
}

// Handle mark as reported
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mark_reported'])) {
    $alert_id = $_POST['alert_id'];
    $ho_report_ref = $_POST['ho_report_ref'] ?? '';

    $update_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "attendance_absence_alerts
        SET ho_reported = 1, ho_report_date = CURDATE(), ho_report_ref = ?, status = 'reported'
        WHERE id = ?");
    $update_stmt->execute([$ho_report_ref, $alert_id]);

    $_SESSION['success'] = 'Absence marked as reported to Home Office';
    header('Location: absence_alerts.php');
    exit;
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <!-- Header -->
    <div class="modern-card mb-3">
        <div class="card-body-modern">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <h4><i class="bi bi-exclamation-triangle"></i> Absence Alerts</h4>
                    <p class="text-muted mb-0">Home Office Reporting - 10+ Day Absences</p>
                </div>
                <div class="col-md-6 text-end">
                    <a href="calendar.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Calendar
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistics -->
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="stat-card bg-danger text-white text-center">
                <h3><?php echo count($critical_alerts); ?></h3>
                <small>Critical Alerts (10+ Days)</small>
                <div class="mt-2"><small>Requires Home Office Reporting</small></div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="stat-card bg-warning text-white text-center">
                <h3><?php echo count($warning_alerts); ?></h3>
                <small>Warning (5-9 Days)</small>
                <div class="mt-2"><small>Approaching Threshold</small></div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="stat-card bg-info text-white text-center">
                <h3><?php echo count($alerts); ?></h3>
                <small>Total Active Alerts</small>
                <div class="mt-2"><small>All Absences Being Tracked</small></div>
            </div>
        </div>
    </div>

    <!-- Critical Alerts (10+ Days) -->
    <?php if (!empty($critical_alerts)): ?>
    <div class="modern-card mb-4">
        <div class="card-header-modern bg-danger text-white">
            <h5 class="card-title-modern mb-0">
                <i class="bi bi-exclamation-octagon"></i> Critical: Requires Home Office Reporting
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="alert alert-danger">
                <strong><i class="bi bi-info-circle"></i> Home Office Requirement:</strong>
                You must report absences of 10 or more consecutive days to the Home Office within 20 working days from the 10th day of absence.
            </div>

            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Absence Start</th>
                            <th>Days Absent</th>
                            <th>Deadline</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($critical_alerts as $alert):
                            // Calculate reporting deadline (20 working days from 10th day)
                            $tenth_day = new DateTime($alert['absence_start_date']);
                            $tenth_day->modify('+9 days');
                            $deadline = clone $tenth_day;
                            $deadline->modify('+20 weekdays');
                            $days_until_deadline = (new DateTime())->diff($deadline)->days;
                            $overdue = new DateTime() > $deadline;
                        ?>
                        <tr class="<?php echo $overdue ? 'table-danger' : ''; ?>">
                            <td>
                                <strong><?php echo htmlspecialchars($alert['worker_name']); ?></strong><br>
                                <small class="text-muted"><?php echo htmlspecialchars($alert['employee_number']); ?></small>
                            </td>
                            <td><?php echo date('d/m/Y', strtotime($alert['absence_start_date'])); ?></td>
                            <td>
                                <span class="badge bg-danger"><?php echo $alert['consecutive_days']; ?> days</span>
                            </td>
                            <td>
                                <?php echo $deadline->format('d/m/Y'); ?>
                                <?php if ($overdue): ?>
                                    <br><span class="badge bg-danger">OVERDUE</span>
                                <?php else: ?>
                                    <br><small class="text-warning"><?php echo $days_until_deadline; ?> days left</small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($alert['ho_reported']): ?>
                                    <span class="badge bg-success">Reported</span><br>
                                    <small><?php echo date('d/m/Y', strtotime($alert['ho_report_date'])); ?></small><br>
                                    <small class="text-muted"><?php echo htmlspecialchars($alert['ho_report_ref'] ?? '-'); ?></small>
                                <?php else: ?>
                                    <span class="badge bg-warning">Pending</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!$alert['ho_reported']): ?>
                                    <button type="button" class="btn btn-sm btn-success"
                                            data-bs-toggle="modal"
                                            data-bs-target="#reportModal<?php echo $alert['id']; ?>">
                                        <i class="bi bi-check-circle"></i> Mark as Reported
                                    </button>
                                <?php else: ?>
                                    <a href="../workers/view.php?id=<?php echo $alert['worker_id']; ?>" class="btn btn-sm btn-outline-primary">
                                        <i class="bi bi-eye"></i> View Worker
                                    </a>
                                <?php endif; ?>
                            </td>
                        </tr>

                        <!-- Report Modal -->
                        <div class="modal fade" id="reportModal<?php echo $alert['id']; ?>" tabindex="-1">
                            <div class="modal-dialog">
                                <form method="POST">
                                    <div class="modal-content">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Mark as Reported to Home Office</h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                        </div>
                                        <div class="modal-body">
                                            <input type="hidden" name="mark_reported" value="1">
                                            <input type="hidden" name="alert_id" value="<?php echo $alert['id']; ?>">

                                            <p><strong>Worker:</strong> <?php echo htmlspecialchars($alert['worker_name']); ?></p>
                                            <p><strong>Absence Period:</strong> <?php echo $alert['consecutive_days']; ?> days</p>

                                            <div class="mb-3">
                                                <label class="form-label">Home Office Reference Number (Optional)</label>
                                                <input type="text" name="ho_report_ref" class="form-control" placeholder="e.g., HO-2025-12345">
                                            </div>

                                            <div class="alert alert-info">
                                                <small>This will mark the absence as reported and remove it from the critical alerts list.</small>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                            <button type="submit" class="btn btn-success">
                                                <i class="bi bi-check-circle"></i> Confirm Reported
                                            </button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php else: ?>
    <div class="alert alert-success">
        <i class="bi bi-check-circle"></i> <strong>All Clear!</strong> No critical absence alerts requiring Home Office reporting at this time.
    </div>
    <?php endif; ?>

    <!-- Warning Alerts (5-9 Days) -->
    <?php if (!empty($warning_alerts)): ?>
    <div class="modern-card">
        <div class="card-header-modern bg-warning">
            <h5 class="card-title-modern mb-0">
                <i class="bi bi-exclamation-triangle"></i> Warning: Approaching Threshold
            </h5>
        </div>
        <div class="card-body-modern">
            <p class="text-muted">These workers are approaching the 10-day absence threshold. Monitor closely.</p>

            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Absence Start</th>
                            <th>Days Absent</th>
                            <th>Days Until Threshold</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($warning_alerts as $alert): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($alert['worker_name']); ?></strong><br>
                                <small class="text-muted"><?php echo htmlspecialchars($alert['employee_number']); ?></small>
                            </td>
                            <td><?php echo date('d/m/Y', strtotime($alert['absence_start_date'])); ?></td>
                            <td>
                                <span class="badge bg-warning"><?php echo $alert['consecutive_days']; ?> days</span>
                            </td>
                            <td>
                                <span class="badge bg-info"><?php echo (10 - $alert['consecutive_days']); ?> days</span>
                            </td>
                            <td>
                                <a href="../workers/view.php?id=<?php echo $alert['worker_id']; ?>" class="btn btn-sm btn-outline-primary">
                                    <i class="bi bi-eye"></i> View Worker
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<?php include '../../includes/footer.php'; ?>
