<?php
/**
 * System Settings - Admin Module
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

// Check admin permission (80+)
if (($_SESSION['permission_level'] ?? 0) < 80) {
    die('Access denied. Admin permission required.');
}

$page_title = 'System Settings';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

$success = '';
$errors = [];
$tab = $_GET['tab'] ?? 'company';

// Helper function to get setting
function get_setting($pdo, $key, $default = '') {
    $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = ?");
    $stmt->execute([$key]);
    $result = $stmt->fetchColumn();
    return $result !== false ? $result : $default;
}

// Helper function to update setting
function update_setting($pdo, $key, $value) {
    $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value, updated_at)
                          VALUES (?, ?, NOW())
                          ON DUPLICATE KEY UPDATE setting_value = ?, updated_at = NOW()");
    return $stmt->execute([$key, $value, $value]);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'update_company') {
        try {
            update_setting($pdo, 'company_name', $_POST['company_name'] ?? '');
            update_setting($pdo, 'company_reg', $_POST['company_reg'] ?? '');
            update_setting($pdo, 'address', $_POST['address'] ?? '');
            update_setting($pdo, 'city', $_POST['city'] ?? '');
            update_setting($pdo, 'postcode', $_POST['postcode'] ?? '');
            update_setting($pdo, 'phone', $_POST['phone'] ?? '');
            update_setting($pdo, 'email', $_POST['email'] ?? '');
            update_setting($pdo, 'industry', $_POST['industry'] ?? '');

            $success = 'Company information updated successfully';
        } catch (Exception $e) {
            $errors[] = 'Error updating settings: ' . $e->getMessage();
        }
    }

    elseif ($action === 'update_system') {
        try {
            update_setting($pdo, 'error_log_entries_per_page', $_POST['error_log_entries_per_page'] ?? '15');
            update_setting($pdo, 'error_log_auto_delete_days', $_POST['error_log_auto_delete_days'] ?? '30');
            update_setting($pdo, 'worker_limit', $_POST['worker_limit'] ?? '50');

            $success = 'System settings updated successfully';
        } catch (Exception $e) {
            $errors[] = 'Error updating settings: ' . $e->getMessage();
        }
    }
}

// Fetch current settings
$settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings");
while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $settings[$row['setting_key']] = $row['setting_value'];
}
?>

<?php if ($success): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <?= htmlspecialchars($success) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<?php if (!empty($errors)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <?php foreach ($errors as $error): ?>
            <?= htmlspecialchars($error) ?><br>
        <?php endforeach; ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
<?php endif; ?>

<!-- Tabs -->
<ul class="nav nav-tabs mb-3">
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'company' ? 'active' : '' ?>" href="?tab=company">
            <i class="bi bi-building"></i> Company Information
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'system' ? 'active' : '' ?>" href="?tab=system">
            <i class="bi bi-gear"></i> System Settings
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'license' ? 'active' : '' ?>" href="?tab=license">
            <i class="bi bi-shield-check"></i> License Information
        </a>
    </li>
    <li class="nav-item">
        <a class="nav-link <?= $tab === 'users' ? 'active' : '' ?>" href="?tab=users">
            <i class="bi bi-people"></i> System Users
        </a>
    </li>
    <?php if (($_SESSION['permission_level'] ?? 0) >= 100): ?>
    <li class="nav-item">
        <a class="nav-link" href="module_manager.php">
            <i class="bi bi-puzzle"></i> Module Manager
        </a>
    </li>
    <?php endif; ?>
</ul>

<?php if ($tab === 'company'): ?>
<!-- Company Information Tab -->
<div class="modern-card">
    <div class="card-body-modern">
        <h5><i class="bi bi-building"></i> Company Information</h5>
        <hr>

        <form method="POST">
            <input type="hidden" name="action" value="update_company">

            <div class="row">
                <div class="col-md-6">
                    <h6 class="text-primary mb-3">Company Details</h6>

                    <div class="mb-3">
                        <label class="form-label">Company Name <span class="text-danger">*</span></label>
                        <input type="text" name="company_name" class="form-control" required
                               value="<?= htmlspecialchars($settings['company_name'] ?? '') ?>">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Registration Number</label>
                        <input type="text" name="company_reg" class="form-control"
                               value="<?= htmlspecialchars($settings['company_reg'] ?? '') ?>">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Industry</label>
                        <input type="text" name="industry" class="form-control"
                               value="<?= htmlspecialchars($settings['industry'] ?? '') ?>"
                               placeholder="e.g., Healthcare, IT Services, Manufacturing">
                    </div>
                </div>

                <div class="col-md-6">
                    <h6 class="text-primary mb-3">Contact Information</h6>

                    <div class="mb-3">
                        <label class="form-label">Address</label>
                        <input type="text" name="address" class="form-control"
                               value="<?= htmlspecialchars($settings['address'] ?? '') ?>">
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">City</label>
                            <input type="text" name="city" class="form-control"
                                   value="<?= htmlspecialchars($settings['city'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Postcode</label>
                            <input type="text" name="postcode" class="form-control"
                                   value="<?= htmlspecialchars($settings['postcode'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Phone</label>
                        <input type="tel" name="phone" class="form-control"
                               value="<?= htmlspecialchars($settings['phone'] ?? '') ?>">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Email</label>
                        <input type="email" name="email" class="form-control"
                               value="<?= htmlspecialchars($settings['email'] ?? '') ?>">
                    </div>
                </div>
            </div>

            <hr>

            <button type="submit" class="btn btn-primary">
                <i class="bi bi-save"></i> Save Changes
            </button>
        </form>
    </div>
</div>

<?php elseif ($tab === 'system'): ?>
<!-- System Settings Tab -->
<div class="modern-card">
    <div class="card-body-modern">
        <h5><i class="bi bi-gear"></i> System Configuration</h5>
        <hr>

        <form method="POST">
            <input type="hidden" name="action" value="update_system">

            <h6 class="text-primary mb-3">Error Log Settings</h6>

            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">Entries Per Page</label>
                    <input type="number" name="error_log_entries_per_page" class="form-control"
                           value="<?= htmlspecialchars($settings['error_log_entries_per_page'] ?? '15') ?>"
                           min="5" max="100">
                    <small class="text-muted">Number of error log entries to display per page (5-100)</small>
                </div>

                <div class="col-md-6 mb-3">
                    <label class="form-label">Auto Delete After (Days)</label>
                    <input type="number" name="error_log_auto_delete_days" class="form-control"
                           value="<?= htmlspecialchars($settings['error_log_auto_delete_days'] ?? '30') ?>"
                           min="1" max="365">
                    <small class="text-muted">Automatically delete error logs older than this many days</small>
                </div>
            </div>

            <hr>

            <h6 class="text-primary mb-3">Worker Management</h6>

            <div class="row">
                <div class="col-md-6 mb-3">
                    <label class="form-label">Worker Limit</label>
                    <input type="number" name="worker_limit" class="form-control"
                           value="<?= htmlspecialchars($settings['worker_limit'] ?? '50') ?>"
                           min="1" max="10000" readonly>
                    <small class="text-muted">Maximum number of workers (set by license)</small>
                </div>
            </div>

            <hr>

            <button type="submit" class="btn btn-primary">
                <i class="bi bi-save"></i> Save Changes
            </button>
        </form>
    </div>
</div>

<?php elseif ($tab === 'license'): ?>
<!-- License Information Tab -->
<div class="modern-card">
    <div class="card-body-modern">
        <h5><i class="bi bi-shield-check"></i> License Information</h5>
        <hr>

        <div class="row">
            <div class="col-md-6">
                <table class="table">
                    <tr>
                        <th width="40%">License Key:</th>
                        <td><code><?= htmlspecialchars($settings['license_key'] ?? 'N/A') ?></code></td>
                    </tr>
                    <tr>
                        <th>License Type:</th>
                        <td><span class="badge bg-<?= ($settings['license_type'] ?? '') === 'free' ? 'info' : 'success' ?>">
                            <?= htmlspecialchars(strtoupper($settings['license_type'] ?? 'N/A')) ?>
                        </span></td>
                    </tr>
                    <tr>
                        <th>Customer:</th>
                        <td><?= htmlspecialchars($settings['license_customer'] ?? 'N/A') ?></td>
                    </tr>
                    <tr>
                        <th>Domain:</th>
                        <td><?= htmlspecialchars($settings['license_domain'] ?? 'N/A') ?></td>
                    </tr>
                </table>
            </div>

            <div class="col-md-6">
                <table class="table">
                    <tr>
                        <th width="40%">Expiry Date:</th>
                        <td><?= htmlspecialchars($settings['license_expiry'] ?? 'N/A') ?></td>
                    </tr>
                    <tr>
                        <th>Last Validation:</th>
                        <td><?= htmlspecialchars($settings['last_license_validation'] ?? 'Never') ?></td>
                    </tr>
                    <tr>
                        <th>Worker Limit:</th>
                        <td><?= htmlspecialchars($settings['worker_limit'] ?? 'N/A') ?></td>
                    </tr>
                    <tr>
                        <th>Validator URL:</th>
                        <td><small><?= htmlspecialchars($settings['validator_url'] ?? 'N/A') ?></small></td>
                    </tr>
                </table>
            </div>
        </div>

        <div class="alert alert-info mt-3">
            <i class="bi bi-info-circle"></i> <strong>Note:</strong> License is automatically validated on each login.
            Contact support to upgrade or modify your license.
        </div>
    </div>
</div>

<?php elseif ($tab === 'users'): ?>
<!-- Users Tab -->
<div class="modern-card">
    <div class="card-body-modern">
        <div class="d-flex justify-content-between align-items-center mb-3">
            <h5><i class="bi bi-people"></i> System Users</h5>
            <a href="add_user.php" class="btn btn-primary btn-sm">
                <i class="bi bi-person-plus"></i> Add User
            </a>
        </div>
        <hr>

        <?php
        $stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "users ORDER BY permission_level DESC, full_name");
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        ?>

        <div class="table-responsive">
            <table class="table table-hover">
                <thead class="table-light">
                    <tr>
                        <th>Name</th>
                        <th>Username</th>
                        <th>Email</th>
                        <th>Role</th>
                        <th>Permission</th>
                        <th>Status</th>
                        <th>Last Login</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $user): ?>
                    <tr>
                        <td><strong><?= htmlspecialchars($user['full_name'] ?? $user['first_name'] . ' ' . $user['last_name']) ?></strong></td>
                        <td><?= htmlspecialchars($user['username']) ?></td>
                        <td><?= htmlspecialchars($user['email']) ?></td>
                        <td><?= htmlspecialchars($user['user_role'] ?? 'User') ?></td>
                        <td>
                            <span class="badge bg-<?= $user['permission_level'] >= 90 ? 'danger' : ($user['permission_level'] >= 70 ? 'warning' : 'primary') ?>">
                                Level <?= $user['permission_level'] ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge <?= $user['is_active'] ? 'bg-success' : 'bg-secondary' ?>">
                                <?= $user['is_active'] ? 'Active' : 'Inactive' ?>
                            </span>
                        </td>
                        <td>
                            <?php if ($user['last_login']): ?>
                                <?= date('d M Y H:i', strtotime($user['last_login'])) ?>
                            <?php else: ?>
                                <span class="text-muted">Never</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($user['id'] != $_SESSION['user_id']): ?>
                            <a href="edit_user.php?id=<?= $user['id'] ?>" class="btn btn-sm btn-outline-warning" title="Edit">
                                <i class="bi bi-pencil"></i>
                            </a>
                            <a href="delete_user.php?id=<?= $user['id'] ?>" class="btn btn-sm btn-outline-danger" title="Delete"
                               onclick="return confirm('Are you sure you want to delete this user?')">
                                <i class="bi bi-trash"></i>
                            </a>
                            <?php else: ?>
                            <span class="badge bg-info">You</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <div class="alert alert-info mt-3">
            <strong><i class="bi bi-info-circle"></i> Permission Levels:</strong>
            <ul class="mb-0 small mt-2">
                <li><strong>100:</strong> System Administrator - Full access to everything</li>
                <li><strong>90:</strong> HR Director - All HR functions including user management</li>
                <li><strong>80:</strong> HR Manager - Worker and document management</li>
                <li><strong>70:</strong> Department Manager - Team management and approvals</li>
                <li><strong>60:</strong> Supervisor - Attendance marking and basic reporting</li>
                <li><strong>50:</strong> Staff - View own records only</li>
                <li><strong>30:</strong> Limited/Read-only access</li>
            </ul>
        </div>
    </div>
</div>
<?php endif; ?>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>
