/* ============================================================
   SCMS MODULE MANAGER SYSTEM
   File: modules/admin/module_manager.php
   
   Central repository for all system files with:
   - Version control
   - Automatic updates
   - Schema migrations
   - Package building
   - Git-like functionality
   ============================================================ */

<?php
require __DIR__ . '/../../config/database.php';
require __DIR__ . '/../../includes/functions.php';

require_login();
require_permission(100); // Admin only
$page_title = 'Module Manager';

// Initialize Module Manager
class ModuleManager {
    private $pdo;
    private $version_file = __DIR__ . '/../../config/versions.json';
    private $modules_file = __DIR__ . '/../../config/modules.json';
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->initializeVersionControl();
    }
    
    private function initializeVersionControl() {
        // Create versions file if not exists
        if (!file_exists($this->version_file)) {
            $initial = [
                'system_version' => '1.0.0',
                'last_update' => date('Y-m-d H:i:s'),
                'files' => []
            ];
            file_put_contents($this->version_file, json_encode($initial, JSON_PRETTY_PRINT));
        }
        
        // Create modules registry
        if (!file_exists($this->modules_file)) {
            $initial = [
                'modules' => [],
                'last_update' => date('Y-m-d H:i:s')
            ];
            file_put_contents($this->modules_file, json_encode($initial, JSON_PRETTY_PRINT));
        }
    }
    
    public function getVersionInfo() {
        return json_decode(file_get_contents($this->version_file), true);
    }
    
    public function getModules() {
        return json_decode(file_get_contents($this->modules_file), true);
    }
    
    public function getAllFiles() {
        $base = __DIR__ . '/../..';
        $files = [];
        
        // Scan directories
        $dirs = [
            'modules/workers' => 'Workers Module',
            'modules/holidays' => 'Holidays Module',
            'modules/attendance' => 'Attendance Module',
            'modules/documents' => 'Documents Module',
            'modules/reports' => 'Reports Module',
            'modules/cos' => 'CoS Module',
            'modules/admin' => 'Admin Module',
            'includes' => 'Core Includes',
            'config' => 'Configuration'
        ];
        
        foreach ($dirs as $dir => $name) {
            $full_path = $base . '/' . $dir;
            if (is_dir($full_path)) {
                $scan = scandir($full_path);
                foreach ($scan as $file) {
                    if ($file === '.' || $file === '..') continue;
                    if (pathinfo($file, PATHINFO_EXTENSION) !== 'php') continue;
                    
                    $filepath = $dir . '/' . $file;
                    $full = $base . '/' . $filepath;
                    
                    $files[] = [
                        'path' => $filepath,
                        'name' => $file,
                        'module' => $name,
                        'size' => filesize($full),
                        'modified' => filemtime($full),
                        'version' => $this->getFileVersion($filepath),
                        'exists' => true
                    ];
                }
            }
        }
        
        return $files;
    }
    
    private function getFileVersion($filepath) {
        $versions = $this->getVersionInfo();
        return $versions['files'][$filepath] ?? '1.0.0';
    }
    
    public function updateFileVersion($filepath, $new_version) {
        $versions = $this->getVersionInfo();
        $versions['files'][$filepath] = $new_version;
        $versions['last_update'] = date('Y-m-d H:i:s');
        file_put_contents($this->version_file, json_encode($versions, JSON_PRETTY_PRINT));
    }
    
    public function parseModulePackage($package_code) {
        // Parse the special format from Claude
        $result = [
            'success' => false,
            'module' => null,
            'files' => [],
            'schema' => null,
            'menu_items' => [],
            'version' => '1.0.0',
            'errors' => []
        ];
        
        // Extract module metadata
        if (preg_match('/@MODULE:\s*([^\n]+)/', $package_code, $matches)) {
            $result['module'] = trim($matches[1]);
        }
        
        // Extract version
        if (preg_match('/@VERSION:\s*([^\n]+)/', $package_code, $matches)) {
            $result['version'] = trim($matches[1]);
        }
        
        // Extract files
        preg_match_all('/@FILE:\s*([^\n]+)\n@CONTENT:\n(.*?)(?=@FILE:|@SCHEMA:|@MENU:|$)/s', 
                       $package_code, $file_matches, PREG_SET_ORDER);
        
        foreach ($file_matches as $match) {
            $filepath = trim($match[1]);
            $content = trim($match[2]);
            $result['files'][$filepath] = $content;
        }
        
        // Extract schema
        if (preg_match('/@SCHEMA:\n(.*?)(?=@MENU:|@FILE:|$)/s', $package_code, $matches)) {
            $result['schema'] = trim($matches[1]);
        }
        
        // Extract menu items
        if (preg_match('/@MENU:\n(.*?)(?=@FILE:|@SCHEMA:|$)/s', $package_code, $matches)) {
            $menu_data = trim($matches[1]);
            $result['menu_items'] = json_decode($menu_data, true) ?: [];
        }
        
        $result['success'] = !empty($result['files']);
        
        return $result;
    }
    
    public function installModulePackage($parsed) {
        $errors = [];
        $installed_files = [];
        
        try {
            $this->pdo->beginTransaction();
            
            // 1. Run schema migrations
            if (!empty($parsed['schema'])) {
                try {
                    $this->pdo->exec($parsed['schema']);
                } catch (PDOException $e) {
                    $errors[] = "Schema error: " . $e->getMessage();
                }
            }
            
            // 2. Install files
            foreach ($parsed['files'] as $filepath => $content) {
                $full_path = __DIR__ . '/../../' . $filepath;
                $dir = dirname($full_path);
                
                // Create directory if needed
                if (!is_dir($dir)) {
                    mkdir($dir, 0755, true);
                }
                
                // Write file
                if (file_put_contents($full_path, $content) !== false) {
                    $installed_files[] = $filepath;
                    $this->updateFileVersion($filepath, $parsed['version']);
                } else {
                    $errors[] = "Failed to write: $filepath";
                }
            }
            
            // 3. Update menu
            if (!empty($parsed['menu_items'])) {
                $this->updateMenuItems($parsed['menu_items']);
            }
            
            // 4. Register module
            $modules = $this->getModules();
            $modules['modules'][$parsed['module']] = [
                'version' => $parsed['version'],
                'installed' => date('Y-m-d H:i:s'),
                'files' => $installed_files
            ];
            $modules['last_update'] = date('Y-m-d H:i:s');
            file_put_contents($this->modules_file, json_encode($modules, JSON_PRETTY_PRINT));
            
            $this->pdo->commit();
            
            return [
                'success' => true,
                'installed_files' => $installed_files,
                'errors' => $errors
            ];
            
        } catch (Exception $e) {
            $this->pdo->rollBack();
            return [
                'success' => false,
                'installed_files' => [],
                'errors' => array_merge($errors, [$e->getMessage()])
            ];
        }
    }
    
    private function updateMenuItems($menu_items) {
        // Update header.php to add menu items
        $header_path = __DIR__ . '/../../includes/header.php';
        if (file_exists($header_path)) {
            $content = file_get_contents($header_path);
            
            // Find menu insertion point
            $marker = '<!-- MODULE_MENU_ITEMS -->';
            if (strpos($content, $marker) !== false) {
                $menu_html = $this->generateMenuHtml($menu_items);
                $content = str_replace($marker, $menu_html . "\n" . $marker, $content);
                file_put_contents($header_path, $content);
            }
        }
    }
    
    private function generateMenuHtml($items) {
        $html = '';
        foreach ($items as $item) {
            $html .= sprintf(
                '<li class="nav-item"><a class="nav-link" href="%s"><i class="bi bi-%s"></i> %s</a></li>' . "\n",
                $item['url'],
                $item['icon'],
                $item['label']
            );
        }
        return $html;
    }
    
    public function buildPackage($module_name) {
        $modules = $this->getModules();
        if (!isset($modules['modules'][$module_name])) {
            return ['success' => false, 'error' => 'Module not found'];
        }
        
        $module = $modules['modules'][$module_name];
        $package = [];
        
        $package[] = "@MODULE: $module_name";
        $package[] = "@VERSION: {$module['version']}";
        $package[] = "@GENERATED: " . date('Y-m-d H:i:s');
        $package[] = "";
        
        // Add files
        foreach ($module['files'] as $filepath) {
            $full_path = __DIR__ . '/../../' . $filepath;
            if (file_exists($full_path)) {
                $content = file_get_contents($full_path);
                $package[] = "@FILE: $filepath";
                $package[] = "@CONTENT:";
                $package[] = $content;
                $package[] = "";
            }
        }
        
        return [
            'success' => true,
            'package' => implode("\n", $package)
        ];
    }
    
    public function getSchemaInfo() {
        $tables = [];
        $stmt = $this->pdo->query("SHOW TABLES");
        foreach ($stmt->fetchAll(PDO::FETCH_COLUMN) as $table) {
            $columns = $this->pdo->query("DESCRIBE $table")->fetchAll();
            $tables[$table] = [
                'columns' => count($columns),
                'details' => $columns
            ];
        }
        return $tables;
    }
    
    public function migrateSchema($sql) {
        try {
            $this->pdo->beginTransaction();
            
            // Split multiple statements
            $statements = array_filter(
                array_map('trim', explode(';', $sql)),
                function($s) { return !empty($s); }
            );
            
            foreach ($statements as $stmt) {
                $this->pdo->exec($stmt);
            }
            
            $this->pdo->commit();
            
            // Log migration
            log_activity($this->pdo, 'schema_migration', 'system', 0, 
                        'Schema migration executed: ' . substr($sql, 0, 100));
            
            return ['success' => true];
        } catch (PDOException $e) {
            $this->pdo->rollBack();
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}

$manager = new ModuleManager($pdo);
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($action === 'install_package') {
        $package_code = $_POST['package_code'] ?? '';
        $parsed = $manager->parseModulePackage($package_code);
        
        if ($parsed['success']) {
            $result = $manager->installModulePackage($parsed);
            if ($result['success']) {
                $message = "Module installed successfully! Files: " . count($result['installed_files']);
                if (!empty($result['errors'])) {
                    $message .= " (with warnings)";
                }
            } else {
                $error = "Installation failed: " . implode(', ', $result['errors']);
            }
        } else {
            $error = "Failed to parse package format";
        }
    }
    
    if ($action === 'migrate_schema') {
        $sql = $_POST['schema_sql'] ?? '';
        $result = $manager->migrateSchema($sql);
        if ($result['success']) {
            $message = "Schema migration completed successfully";
        } else {
            $error = "Migration failed: " . $result['error'];
        }
    }
    
    if ($action === 'update_installer') {
        // Update installer with new schema
        $schema_sql = $_POST['schema_sql'] ?? '';
        $installer_path = __DIR__ . '/../../install/index.php';
        
        if (file_exists($installer_path)) {
            $content = file_get_contents($installer_path);
            
            // Find schema section and update
            $pattern = '/(\/\* EMBEDDED SQL SCHEMA \*\/.*?)(\/\* END SQL SCHEMA \*\/)/s';
            $replacement = "$1\n" . $schema_sql . "\n$2";
            $content = preg_replace($pattern, $replacement, $content);
            
            file_put_contents($installer_path, $content);
            $message = "Installer updated with new schema";
        } else {
            $error = "Installer file not found";
        }
    }
}

require __DIR__ . '/../../includes/header.php';
?>

<div class="row mb-4">
    <div class="col-md-8">
        <h2><i class="bi bi-gear-fill"></i> Module Manager</h2>
        <p class="text-muted">Central repository for all system files, versions, and modules</p>
    </div>
    <div class="col-md-4 text-end">
        <div class="btn-group">
            <a href="?action=list" class="btn btn-outline-primary">Files</a>
            <a href="?action=install" class="btn btn-outline-success">Install</a>
            <a href="?action=schema" class="btn btn-outline-info">Schema</a>
            <a href="?action=help" class="btn btn-outline-secondary">Help</a>
        </div>
    </div>
</div>

<?php if ($message): ?>
<div class="alert alert-success alert-dismissible fade show">
    <i class="bi bi-check-circle"></i> <?= htmlspecialchars($message) ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<?php if ($error): ?>
<div class="alert alert-danger alert-dismissible fade show">
    <i class="bi bi-exclamation-triangle"></i> <?= htmlspecialchars($error) ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<?php
// ============================================================
// ACTION: FILES LIST
// ============================================================
if ($action === 'list'):
    $files = $manager->getAllFiles();
    $versions = $manager->getVersionInfo();
?>

<div class="row mb-4">
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3><?= count($files) ?></h3>
            <small>Total Files</small>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3><?= $versions['system_version'] ?></h3>
            <small>System Version</small>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <h3><?= count($manager->getModules()['modules']) ?></h3>
            <small>Modules</small>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stat-card text-center">
            <small class="text-muted">Last Update</small><br>
            <small><?= date('d/m/Y H:i', strtotime($versions['last_update'])) ?></small>
        </div>
    </div>
</div>

<div class="stat-card">
    <h5 class="mb-3">All System Files</h5>
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>File</th>
                    <th>Module</th>
                    <th>Version</th>
                    <th>Size</th>
                    <th>Modified</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($files as $file): ?>
                <tr>
                    <td>
                        <i class="bi bi-file-code"></i>
                        <code><?= htmlspecialchars($file['path']) ?></code>
                    </td>
                    <td><?= htmlspecialchars($file['module']) ?></td>
                    <td><span class="badge bg-primary"><?= htmlspecialchars($file['version']) ?></span></td>
                    <td><?= number_format($file['size'] / 1024, 1) ?> KB</td>
                    <td><?= date('d/m/Y H:i', $file['modified']) ?></td>
                    <td>
                        <button class="btn btn-sm btn-outline-primary" onclick="viewFile('<?= htmlspecialchars($file['path']) ?>')">
                            <i class="bi bi-eye"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-info" onclick="downloadFile('<?= htmlspecialchars($file['path']) ?>')">
                            <i class="bi bi-download"></i>
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<?php
// ============================================================
// ACTION: INSTALL PACKAGE
// ============================================================
elseif ($action === 'install'):
?>

<div class="stat-card">
    <h4 class="mb-4"><i class="bi bi-box-arrow-in-down"></i> Install Module Package</h4>
    
    <div class="alert alert-info">
        <strong>How it works:</strong> Paste the module package code from Claude. It will automatically:
        <ul class="mb-0 mt-2">
            <li>Parse and extract all files</li>
            <li>Run database migrations</li>
            <li>Update menu navigation</li>
            <li>Register the module</li>
            <li>Set correct versions</li>
        </ul>
    </div>
    
    <form method="POST" action="?action=install_package">
        <div class="mb-3">
            <label class="form-label">Module Package Code</label>
            <textarea name="package_code" class="form-control" rows="20" 
                      placeholder="Paste module package from Claude here..."
                      style="font-family: monospace; font-size: 12px;" required></textarea>
            <small class="text-muted">
                Package format: @MODULE, @VERSION, @FILE, @CONTENT, @SCHEMA, @MENU
            </small>
        </div>
        
        <button type="submit" class="btn btn-success btn-lg">
            <i class="bi bi-box-arrow-in-down"></i> Install Package
        </button>
    </form>
</div>

<div class="stat-card mt-4">
    <h5>Package Format Example</h5>
    <pre class="bg-light p-3" style="font-size: 12px;">@MODULE: Holiday Management
@VERSION: 1.2.0

@FILE: modules/holidays/list.php
@CONTENT:
<?php
// File content here...
?>

@FILE: modules/holidays/request.php
@CONTENT:
<?php
// More content...
?>

@SCHEMA:
ALTER TABLE holiday_requests ADD COLUMN new_field VARCHAR(255);

@MENU:
[
  {"label": "Holidays", "url": "modules/holidays/list.php", "icon": "calendar"}
]</pre>
</div>

<?php
// ============================================================
// ACTION: SCHEMA MANAGEMENT
// ============================================================
elseif ($action === 'schema'):
    $tables = $manager->getSchemaInfo();
?>

<div class="row mb-4">
    <div class="col-md-6">
        <div class="stat-card">
            <h5><i class="bi bi-database"></i> Run Schema Migration</h5>
            <form method="POST" action="?action=migrate_schema">
                <div class="mb-3">
                    <label class="form-label">SQL Statements</label>
                    <textarea name="schema_sql" class="form-control" rows="10" 
                              placeholder="ALTER TABLE...&#10;CREATE TABLE...&#10;INSERT INTO..."
                              style="font-family: monospace;"></textarea>
                </div>
                <button type="submit" class="btn btn-primary">
                    <i class="bi bi-lightning"></i> Run Migration
                </button>
            </form>
        </div>
    </div>
    
    <div class="col-md-6">
        <div class="stat-card">
            <h5><i class="bi bi-box-seam"></i> Update Installer Schema</h5>
            <form method="POST" action="?action=update_installer">
                <div class="mb-3">
                    <label class="form-label">New Schema SQL</label>
                    <textarea name="schema_sql" class="form-control" rows="10" 
                              placeholder="This will be embedded into install/index.php"
                              style="font-family: monospace;"></textarea>
                </div>
                <button type="submit" class="btn btn-warning">
                    <i class="bi bi-box-seam"></i> Update Installer
                </button>
            </form>
        </div>
    </div>
</div>

<div class="stat-card">
    <h5 class="mb-3">Current Database Schema (<?= count($tables) ?> Tables)</h5>
    <div class="accordion" id="schemaAccordion">
        <?php $index = 0; foreach ($tables as $table_name => $table_info): ?>
        <div class="accordion-item">
            <h2 class="accordion-header">
                <button class="accordion-button collapsed" type="button" 
                        data-bs-toggle="collapse" data-bs-target="#table<?= $index ?>">
                    <strong><?= htmlspecialchars($table_name) ?></strong>
                    <span class="badge bg-secondary ms-2"><?= $table_info['columns'] ?> columns</span>
                </button>
            </h2>
            <div id="table<?= $index ?>" class="accordion-collapse collapse">
                <div class="accordion-body">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Field</th>
                                <th>Type</th>
                                <th>Null</th>
                                <th>Key</th>
                                <th>Default</th>
                                <th>Extra</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($table_info['details'] as $col): ?>
                            <tr>
                                <td><code><?= htmlspecialchars($col['Field']) ?></code></td>
                                <td><?= htmlspecialchars($col['Type']) ?></td>
                                <td><?= htmlspecialchars($col['Null']) ?></td>
                                <td><?= htmlspecialchars($col['Key']) ?></td>
                                <td><?= htmlspecialchars($col['Default'] ?? 'NULL') ?></td>
                                <td><?= htmlspecialchars($col['Extra']) ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <?php $index++; endforeach; ?>
    </div>
</div>

<?php
// ============================================================
// ACTION: HELP & PROMPTS
// ============================================================
elseif ($action === 'help'):
?>

<div class="stat-card">
    <h4 class="mb-4"><i class="bi bi-question-circle"></i> How to Use Module Manager</h4>
    
    <h5 class="mt-4">📋 Step-by-Step Workflow</h5>
    <ol>
        <li><strong>Request Feature from Claude:</strong> Use the prompt templates below</li>
        <li><strong>Receive Package:</strong> Claude sends code in special format</li>
        <li><strong>Install Package:</strong> Paste into "Install" tab, click install</li>
        <li><strong>Auto-Deploy:</strong> System extracts files, runs migrations, updates menus</li>
        <li><strong>Go Live:</strong> Feature is immediately available</li>
    </ol>
    
    <hr>
    
    <h5 class="mt-4">💬 Prompt Templates for Claude</h5>
    
    <div class="accordion" id="promptAccordion">
        
        <!-- Template 1 -->
        <div class="accordion-item">
            <h2 class="accordion-header">
                <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#prompt1">
                    <strong>🆕 New Module Request</strong>
                </button>
            </h2>
            <div id="prompt1" class="accordion-collapse collapse show">
                <div class="accordion-body">
                    <pre class="bg-light p-3" style="font-size: 13px;">Create a new [MODULE_NAME] module for SCMS using Module Manager format.

Requirements:
- [Feature 1]
- [Feature 2]
- [Feature 3]

Please provide in MODULE PACKAGE FORMAT with:
@MODULE: [Name]
@VERSION: 1.0.0
@FILE: [paths]
@CONTENT: [code]
@SCHEMA: [SQL migrations]
@MENU: [navigation items]

Ensure version compatibility with SCMS v1.0.0</pre>
                    <button class="btn btn-sm btn-outline-primary" onclick="copyPrompt('prompt1')">
                        <i class="bi bi-clipboard"></i> Copy Template
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Template 2 -->
        <div class="accordion-item">
            <h2 class="accordion-header">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#prompt2">
                    <strong>🔄 Update Existing Module</strong>
                </button>
            </h2>
            <div id="prompt2" class="accordion-collapse collapse">
                <div class="accordion-body">
                    <pre class="bg-light p-3" style="font-size: 13px;">Update the [MODULE_NAME] module (currently v[X.X.X]) with these changes:

Changes needed:
- [Change 1]
- [Change 2]

Provide update in MODULE PACKAGE FORMAT with:
- Only files that changed
- Version incremented to [X.X.X]
- Schema migrations (if any)
- Updated menu items (if changed)</pre>
                    <button class="btn btn-sm btn-outline-primary" onclick="copyPrompt('prompt2')">
                        <i class="bi bi-clipboard"></i> Copy Template
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Template 3 -->
        <div class="accordion-item">
            <h2 class="accordion-header">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#prompt3">
                    <strong>🔧 Fix Bug</strong>
                </button>
            </h2>
            <div id="prompt3" class="accordion-collapse collapse">
                <div class="accordion-body">
                    <pre class="bg-light p-3" style="font-size: 13px;">Fix bug in [MODULE_NAME] module:

Issue: [Describe the bug]

File affected: [filepath]
Current behavior: [what's wrong]
Expected behavior: [what should happen]

Provide fix in MODULE PACKAGE FORMAT with only the affected file(s).
Increment patch version.</pre>
                    <button class="btn btn-sm btn-outline-primary" onclick="copyPrompt('prompt3')">
                        <i class="bi bi-clipboard"></i> Copy Template
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Template 4 -->
        <div class="accordion-item">
            <h2 class="accordion-header">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#prompt4">
                    <strong>🗄️ Database Schema Update</strong>
                </button>
            </h2>
            <div id="prompt4" class="accordion-collapse collapse">
                <div class="accordion-body">
                    <pre class="bg-light p-3" style="font-size: 13px;">Update database schema for SCMS:

Changes needed:
- Add table: [table_name] with columns [...]
- Modify table: [table_name] - add column [column_name]
- Add index on [table.column]

Provide:
1. @SCHEMA section with SQL migration
2. Updated affected PHP files if needed
3. Rollback SQL (as comment)</pre>
                    <button class="btn btn-sm btn-outline-primary" onclick="copyPrompt('prompt4')">
                        <i class="bi bi-clipboard"></i> Copy Template
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Template 5 -->
        <div class="accordion-item">
            <h2 class="accordion-header">
                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#prompt5">
                    <strong>📦 Export Module</strong>
                </button>
            </h2>
            <div id="prompt5" class="accordion-collapse collapse">
                <div class="accordion-body">
                    <pre class="bg-light p-3" style="font-size: 13px;">Export the complete [MODULE_NAME] module in MODULE PACKAGE FORMAT.

Include:
- All module files
- Current schema (CREATE TABLE statements)
- Menu configuration
- Current version number
- Installation instructions

Make it ready for deployment on a fresh SCMS installation.</pre>
                    <button class="btn btn-sm btn-outline-primary" onclick="copyPrompt('prompt5')">
                        <i class="bi bi-clipboard"></i> Copy Template
                    </button>
                </div>
            </div>
        </div>
        
    </div>
    
    <hr class="my-4">
    
    <h5>🎯 Module Package Format Specification</h5>
    <pre class="bg-dark text-light p-3" style="font-size: 12px;">@MODULE: [Module Name]
@VERSION: [Semantic Version - X.Y.Z]
@DESCRIPTION: [Optional description]
@REQUIRES: [SCMS version required]
@AUTHOR: [Optional]

@FILE: [relative/path/to/file.php]
@CONTENT:
[Complete file content here]

@FILE: [another/file.php]
@CONTENT:
[Another file content]

@SCHEMA:
[SQL statements for database changes]
-- Use comments for documentation
-- Separate statements with semicolons

@MENU:
[
  {
    "label": "Menu Label",
    "url": "modules/path/file.php",
    "icon": "bootstrap-icon-name",
    "permission": 50
  }
]

@ROLLBACK:
[Optional SQL to rollback schema changes]</pre>

    <hr class="my-4">
    
    <h5>⚙️ Version Numbering</h5>
    <ul>
        <li><strong>Major (X.0.0):</strong> Breaking changes, major new features</li>
        <li><strong>Minor (x.Y.0):</strong> New features, backwards compatible</li>
        <li><strong>Patch (x.x.Z):</strong> Bug fixes, small improvements</li>
    </ul>
    
    <hr class="my-4">
    
    <h5>🔄 Typical Workflow</h5>
    <div class="row">
        <div class="col-md-6">
            <div class="card">
                <div class="card-body">
                    <h6>Developer → Claude</h6>
                    <ol class="small">
                        <li>Copy prompt template</li>
                        <li>Fill in requirements</li>
                        <li>Send to Claude</li>
                    </ol>
                </div>
            </div>
        </div>
        <div class="col-md-6">
            <div class="card">
                <div class="card-body">
                    <h6>Claude → System</h6>
                    <ol class="small">
                        <li>Claude generates package</li>
                        <li>Copy package code</li>
                        <li>Paste in "Install" tab</li>
                        <li>Click "Install Package"</li>
                        <li>Feature goes live!</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
</div>

<?php endif; ?>

<script>
function viewFile(filepath) {
    alert('View file: ' + filepath);
    // Implement file viewer
}

function downloadFile(filepath) {
    window.location.href = 'download_file.php?file=' + encodeURIComponent(filepath);
}

function copyPrompt(promptId) {
    const element = document.querySelector('#' + promptId + ' pre');
    const text = element.textContent;
    navigator.clipboard.writeText(text).then(() => {
        alert('Prompt template copied to clipboard!');
    });
}
</script>

<?php require __DIR__ . '/../../includes/footer.php'; ?>