<?php
/**
 * Edit User
 * Edit existing system user accounts with appropriate permissions
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_once '../../includes/error_logger.php';

require_authentication();

$page_title = 'Edit User';
$current_page = 'users';

// Check permission - minimum level 90 (HR Director)
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 90) {
    log_error('UNAUTHORIZED_ACCESS', 'User ' . $current_user['username'] . ' attempted to access edit user page', [
        'user_id' => $current_user['id'],
        'permission_level' => $current_user['permission_level']
    ]);
    die('Access denied. HR Director permission (Level 90) or higher required.');
}

$user_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$errors = [];
$success = '';

// Fetch user to edit
try {
    $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        $_SESSION['error'] = 'User not found';
        header('Location: ../../settings.php?tab=users');
        exit;
    }

    // Cannot edit users with higher or equal permissions
    if ($user['permission_level'] >= $current_user['permission_level'] && $user['id'] !== $current_user['id']) {
        $_SESSION['error'] = 'Cannot edit user with equal or higher permissions';
        log_error('PERMISSION_VIOLATION', 'User attempted to edit account with higher/equal permission', [
            'editor_id' => $current_user['id'],
            'editor_level' => $current_user['permission_level'],
            'target_user_id' => $user_id,
            'target_level' => $user['permission_level']
        ]);
        header('Location: ../../settings.php?tab=users');
        exit;
    }
} catch (PDOException $e) {
    log_error('DATABASE_ERROR', 'Failed to fetch user for editing', [
        'user_id' => $user_id,
        'error' => $e->getMessage()
    ]);
    $_SESSION['error'] = 'Database error occurred';
    header('Location: ../../settings.php?tab=users');
    exit;
}

// Permission level definitions
$permission_levels = [
    100 => 'System Administrator',
    90 => 'HR Director',
    80 => 'HR Manager',
    70 => 'Department Manager',
    60 => 'Supervisor',
    50 => 'Staff',
    40 => 'Read Only',
    30 => 'Limited'
];

// Fetch workers for optional linking
$workers = [];
try {
    $stmt = $pdo->query("SELECT id, employee_number, full_name, email
                         FROM " . DB_PREFIX . "sponsored_workers
                         WHERE status = 'active'
                         ORDER BY full_name ASC");
    $workers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    log_error('DATABASE_ERROR', 'Failed to fetch workers list', [
        'error' => $e->getMessage()
    ]);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate input
    $email = trim($_POST['email'] ?? '');
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $user_role = trim($_POST['user_role'] ?? '');
    $permission_level = (int)($_POST['permission_level'] ?? 0);
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $worker_id = !empty($_POST['worker_id']) ? (int)$_POST['worker_id'] : null;
    $status = $_POST['status'] ?? 'active';

    // Validation
    if (empty($email)) $errors[] = 'Email is required';
    if (empty($first_name)) $errors[] = 'First name is required';
    if (empty($last_name)) $errors[] = 'Last name is required';
    if (empty($user_role)) $errors[] = 'Role title is required';
    if (empty($permission_level)) $errors[] = 'Permission level is required';

    // Email validation
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email format';
    }

    // Password validation (if changing)
    if (!empty($new_password)) {
        if (strlen($new_password) < 8) {
            $errors[] = 'Password must be at least 8 characters';
        }
        if ($new_password !== $confirm_password) {
            $errors[] = 'Passwords do not match';
        }
    }

    // Permission level validation - can't assign higher permission than self
    if ($permission_level > $current_user['permission_level']) {
        $errors[] = 'You cannot assign higher permission level than your own';
        log_error('PERMISSION_VIOLATION', 'User attempted to assign higher permission', [
            'editor_id' => $current_user['id'],
            'editor_level' => $current_user['permission_level'],
            'requested_level' => $permission_level
        ]);
    }

    // Check for duplicate email (excluding current user)
    if (empty($errors) && !empty($email)) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "users WHERE email = ? AND id != ? LIMIT 1");
            $stmt->execute([$email, $user_id]);
            if ($stmt->fetch()) {
                $errors[] = 'Email already exists';
            }
        } catch (PDOException $e) {
            log_error('DATABASE_ERROR', 'Failed to check duplicate email', [
                'email' => $email,
                'error' => $e->getMessage()
            ]);
            $errors[] = 'Database error occurred';
        }
    }

    // Update user if no errors
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            $full_name = $first_name . ' ' . $last_name;

            // Build update query
            $sql = "UPDATE " . DB_PREFIX . "users SET
                    first_name = ?,
                    last_name = ?,
                    full_name = ?,
                    email = ?,
                    phone = ?,
                    user_role = ?,
                    permission_level = ?,
                    worker_id = ?,
                    status = ?,
                    updated_at = NOW()";

            $params = [
                $first_name,
                $last_name,
                $full_name,
                $email,
                $phone,
                $user_role,
                $permission_level,
                $worker_id,
                $status
            ];

            // Add password if provided
            if (!empty($new_password)) {
                $sql .= ", password_hash = ?";
                $params[] = password_hash($new_password, PASSWORD_DEFAULT);
            }

            $sql .= " WHERE id = ?";
            $params[] = $user_id;

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            // Log the update
            log_error('USER_UPDATED', 'User account updated', [
                'user_id' => $user_id,
                'updated_by' => $current_user['id'],
                'changes' => [
                    'email' => $email,
                    'permission_level' => $permission_level,
                    'status' => $status,
                    'password_changed' => !empty($new_password)
                ]
            ], 'info');

            $pdo->commit();

            $_SESSION['success'] = 'User updated successfully';
            header('Location: ../../settings.php?tab=users');
            exit;

        } catch (PDOException $e) {
            $pdo->rollBack();
            log_error('DATABASE_ERROR', 'Failed to update user', [
                'user_id' => $user_id,
                'error' => $e->getMessage()
            ]);
            $errors[] = 'Database error: ' . $e->getMessage();
        }
    }
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <div class="modern-card">
                <div class="card-header-modern">
                    <div class="d-flex justify-content-between align-items-center w-100">
                        <h5 class="card-title-modern mb-0">
                            <i class="bi bi-pencil-square"></i> Edit User
                        </h5>
                        <a href="../../settings.php?tab=users" class="btn btn-outline-secondary btn-sm">
                            <i class="bi bi-arrow-left"></i> Back to Users
                        </a>
                    </div>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger alert-dismissible fade show">
                            <i class="bi bi-exclamation-triangle"></i>
                            <strong>Please correct the following errors:</strong>
                            <ul class="mb-0 mt-2">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="row">
                            <!-- Left Column - Basic Information -->
                            <div class="col-md-6">
                                <h6 class="text-primary mb-3">
                                    <i class="bi bi-person"></i> Basic Information
                                </h6>

                                <div class="mb-3">
                                    <label class="form-label">Username</label>
                                    <input type="text" class="form-control" value="<?php echo htmlspecialchars($user['username']); ?>" disabled>
                                    <small class="text-muted">Username cannot be changed</small>
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">First Name <span class="text-danger">*</span></label>
                                    <input type="text" name="first_name" class="form-control" required
                                           value="<?php echo htmlspecialchars($user['first_name']); ?>">
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Last Name <span class="text-danger">*</span></label>
                                    <input type="text" name="last_name" class="form-control" required
                                           value="<?php echo htmlspecialchars($user['last_name']); ?>">
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Email <span class="text-danger">*</span></label>
                                    <input type="email" name="email" class="form-control" required
                                           value="<?php echo htmlspecialchars($user['email']); ?>">
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Phone</label>
                                    <input type="tel" name="phone" class="form-control"
                                           value="<?php echo htmlspecialchars($user['phone'] ?? ''); ?>">
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Link to Worker (Optional)</label>
                                    <select name="worker_id" class="form-select">
                                        <option value="">Not linked to any worker</option>
                                        <?php foreach ($workers as $worker): ?>
                                            <option value="<?php echo $worker['id']; ?>"
                                                    <?php echo ($user['worker_id'] == $worker['id']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($worker['full_name']) . ' (' . htmlspecialchars($worker['employee_number']) . ')'; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">Link this user account to a worker record</small>
                                </div>
                            </div>

                            <!-- Right Column - Permissions & Password -->
                            <div class="col-md-6">
                                <h6 class="text-primary mb-3">
                                    <i class="bi bi-shield-check"></i> Permissions & Role
                                </h6>

                                <div class="mb-3">
                                    <label class="form-label">Permission Level <span class="text-danger">*</span></label>
                                    <select name="permission_level" id="permission_level" class="form-select" required>
                                        <option value="">Select permission level...</option>
                                        <?php foreach ($permission_levels as $level => $name): ?>
                                            <?php if ($level < $current_user['permission_level'] || ($level == $current_user['permission_level'] && $user_id == $current_user['id'])): ?>
                                                <option value="<?php echo $level; ?>"
                                                        <?php echo ($user['permission_level'] == $level) ? 'selected' : ''; ?>>
                                                    Level <?php echo $level; ?> - <?php echo htmlspecialchars($name); ?>
                                                </option>
                                            <?php endif; ?>
                                        <?php endforeach; ?>
                                    </select>
                                    <small class="text-muted">Cannot assign higher than your own level (<?php echo $current_user['permission_level']; ?>)</small>
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Role Title <span class="text-danger">*</span></label>
                                    <input type="text" name="user_role" id="user_role" class="form-control" required
                                           value="<?php echo htmlspecialchars($user['user_role']); ?>">
                                    <small class="text-muted">Auto-fills based on permission level</small>
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Status</label>
                                    <select name="status" class="form-select">
                                        <option value="active" <?php echo ($user['status'] == 'active') ? 'selected' : ''; ?>>Active</option>
                                        <option value="inactive" <?php echo ($user['status'] == 'inactive') ? 'selected' : ''; ?>>Inactive</option>
                                    </select>
                                </div>

                                <hr>

                                <h6 class="text-primary mb-3">
                                    <i class="bi bi-key"></i> Change Password (Optional)
                                </h6>

                                <div class="mb-3">
                                    <label class="form-label">New Password</label>
                                    <input type="password" name="new_password" class="form-control" minlength="8">
                                    <small class="text-muted">Leave blank to keep current password (minimum 8 characters)</small>
                                </div>

                                <div class="mb-3">
                                    <label class="form-label">Confirm New Password</label>
                                    <input type="password" name="confirm_password" class="form-control" minlength="8">
                                </div>
                            </div>
                        </div>

                        <hr class="my-4">

                        <!-- Account Info -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="card bg-light">
                                    <div class="card-body">
                                        <h6><i class="bi bi-info-circle"></i> Account Information</h6>
                                        <table class="table table-sm table-borderless mb-0">
                                            <tr>
                                                <td class="text-muted">Created:</td>
                                                <td><strong><?php echo date('d M Y, H:i', strtotime($user['created_at'])); ?></strong></td>
                                            </tr>
                                            <tr>
                                                <td class="text-muted">Last Login:</td>
                                                <td><strong><?php echo $user['last_login'] ? date('d M Y, H:i', strtotime($user['last_login'])) : 'Never'; ?></strong></td>
                                            </tr>
                                            <tr>
                                                <td class="text-muted">Last IP:</td>
                                                <td><strong><?php echo htmlspecialchars($user['last_login_ip'] ?? 'N/A'); ?></strong></td>
                                            </tr>
                                            <tr>
                                                <td class="text-muted">Login Count:</td>
                                                <td><strong><?php echo $user['login_count'] ?? 0; ?></strong></td>
                                            </tr>
                                        </table>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="alert alert-info mb-0">
                                    <h6><i class="bi bi-shield-exclamation"></i> Editing Rules</h6>
                                    <ul class="small mb-0">
                                        <li>You cannot edit users with equal or higher permissions</li>
                                        <li>You cannot assign permissions higher than your own</li>
                                        <li>Username cannot be changed for security</li>
                                        <li>Setting status to "Inactive" will prevent login</li>
                                        <li>Password is only updated if you fill both password fields</li>
                                        <li>All changes are logged for audit purposes</li>
                                    </ul>
                                </div>
                            </div>
                        </div>

                        <hr class="my-4">

                        <div class="d-flex justify-content-between">
                            <a href="../../settings.php?tab=users" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle"></i> Save Changes
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Auto-fill role title based on permission level
const permissionLevels = <?php echo json_encode($permission_levels); ?>;

document.getElementById('permission_level').addEventListener('change', function() {
    const level = parseInt(this.value);
    const roleInput = document.getElementById('user_role');

    if (permissionLevels[level]) {
        roleInput.value = permissionLevels[level];
    }
});
</script>

<?php include '../../includes/footer.php'; ?>
