<?php
/**
 * Delete User
 * Deactivate or permanently delete user accounts
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_once '../../includes/error_logger.php';

require_authentication();

// Check permission - minimum level 90 (HR Director)
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 90) {
    log_error('UNAUTHORIZED_ACCESS', 'User attempted to access delete user page', [
        'user_id' => $current_user['id'],
        'permission_level' => $current_user['permission_level']
    ]);
    die('Access denied. HR Director permission (Level 90) or higher required.');
}

$user_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($user_id <= 0) {
    log_error('INVALID_REQUEST', 'Invalid user ID for deletion', [
        'user_id' => $user_id,
        'requested_by' => $current_user['id']
    ]);
    header('Location: ../../users.php?error=invalid_id');
    exit;
}

// Prevent self-deletion
if ($user_id === $current_user['id']) {
    log_error('SELF_DELETE_ATTEMPT', 'User attempted to delete their own account', [
        'user_id' => $current_user['id']
    ]);
    header('Location: ../../users.php?error=cannot_delete_self');
    exit;
}

// Fetch user to delete
try {
    $stmt = $pdo->prepare("SELECT id, username, full_name, email, permission_level, status
                           FROM " . DB_PREFIX . "users
                           WHERE id = ?
                           LIMIT 1");
    $stmt->execute([$user_id]);
    $user_to_delete = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user_to_delete) {
        log_error('USER_NOT_FOUND', 'User not found for deletion', [
            'user_id' => $user_id,
            'requested_by' => $current_user['id']
        ]);
        header('Location: ../../users.php?error=user_not_found');
        exit;
    }

    // Check permission - cannot delete user with higher or equal permission level
    if ($user_to_delete['permission_level'] >= $current_user['permission_level']) {
        log_error('PERMISSION_VIOLATION', 'Attempted to delete user with equal/higher permission', [
            'target_user' => $user_to_delete['username'],
            'target_level' => $user_to_delete['permission_level'],
            'deleter_level' => $current_user['permission_level']
        ]);
        header('Location: ../../users.php?error=insufficient_permission');
        exit;
    }

} catch (PDOException $e) {
    log_error('DATABASE_ERROR', 'Failed to fetch user for deletion', [
        'user_id' => $user_id,
        'error' => $e->getMessage()
    ]);
    header('Location: ../../users.php?error=database_error');
    exit;
}

// Handle deletion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['confirm_delete'])) {
    $delete_type = $_POST['delete_type'] ?? 'deactivate';

    try {
        if ($delete_type === 'permanent') {
            // Permanent deletion
            $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "users WHERE id = ?");
            $stmt->execute([$user_id]);

            log_error('USER_DELETED_PERMANENT', 'User account permanently deleted', [
                'deleted_user_id' => $user_id,
                'deleted_username' => $user_to_delete['username'],
                'deleted_by' => $current_user['username'],
                'deleted_by_id' => $current_user['id']
            ], 'warning');

            header('Location: ../../users.php?success=user_deleted');
            exit;

        } else {
            // Deactivate (soft delete)
            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "users SET status = 'inactive' WHERE id = ?");
            $stmt->execute([$user_id]);

            log_error('USER_DEACTIVATED', 'User account deactivated', [
                'deactivated_user_id' => $user_id,
                'deactivated_username' => $user_to_delete['username'],
                'deactivated_by' => $current_user['username'],
                'deactivated_by_id' => $current_user['id']
            ], 'info');

            header('Location: ../../users.php?success=user_deactivated');
            exit;
        }

    } catch (PDOException $e) {
        log_error('DATABASE_ERROR', 'Failed to delete/deactivate user', [
            'user_id' => $user_id,
            'delete_type' => $delete_type,
            'error' => $e->getMessage()
        ]);
        header('Location: ../../users.php?error=deletion_failed');
        exit;
    }
}

$page_title = 'Delete User';
$current_page = 'users';
include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-exclamation-triangle-fill text-danger"></i> Delete User
            </h1>
            <p class="welcome-subtitle">Remove user account from the system</p>
        </div>
        <div class="welcome-actions">
            <a href="../../users.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Users
            </a>
        </div>
    </div>

    <div class="modern-card">
        <div class="card-body-modern">
            <div class="alert alert-warning">
                <i class="bi bi-exclamation-triangle-fill"></i>
                <strong>Warning:</strong> You are about to delete the following user account.
            </div>

            <!-- User Details -->
            <div class="row mb-4">
                <div class="col-md-6">
                    <h5 class="mb-3">User Information</h5>
                    <table class="table table-bordered">
                        <tr>
                            <th width="40%">Username</th>
                            <td><?php echo htmlspecialchars($user_to_delete['username']); ?></td>
                        </tr>
                        <tr>
                            <th>Full Name</th>
                            <td><?php echo htmlspecialchars($user_to_delete['full_name']); ?></td>
                        </tr>
                        <tr>
                            <th>Email</th>
                            <td><?php echo htmlspecialchars($user_to_delete['email']); ?></td>
                        </tr>
                        <tr>
                            <th>Permission Level</th>
                            <td>Level <?php echo $user_to_delete['permission_level']; ?></td>
                        </tr>
                        <tr>
                            <th>Current Status</th>
                            <td>
                                <span class="badge bg-<?php echo $user_to_delete['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                    <?php echo ucfirst($user_to_delete['status']); ?>
                                </span>
                            </td>
                        </tr>
                    </table>
                </div>

                <div class="col-md-6">
                    <h5 class="mb-3">Deletion Options</h5>

                    <form method="POST" action="" id="deleteForm">
                        <div class="mb-4">
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="radio" name="delete_type" id="deactivate" value="deactivate" checked>
                                <label class="form-check-label" for="deactivate">
                                    <strong>Deactivate Account (Recommended)</strong>
                                    <br>
                                    <small class="text-muted">
                                        User will be unable to login but their data will be preserved.
                                        Account can be reactivated later.
                                    </small>
                                </label>
                            </div>

                            <div class="form-check mb-3">
                                <input class="form-check-input" type="radio" name="delete_type" id="permanent" value="permanent">
                                <label class="form-check-label" for="permanent">
                                    <strong class="text-danger">Permanent Deletion</strong>
                                    <br>
                                    <small class="text-muted">
                                        User account will be permanently deleted from the database.
                                        This action CANNOT be undone.
                                    </small>
                                </label>
                            </div>
                        </div>

                        <div class="alert alert-danger" id="permanentWarning" style="display: none;">
                            <i class="bi bi-exclamation-octagon-fill"></i>
                            <strong>PERMANENT DELETION WARNING</strong>
                            <p class="mb-0">This action is irreversible. All user data will be permanently removed from the database.</p>
                        </div>

                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" id="confirmCheck" required>
                            <label class="form-check-label" for="confirmCheck">
                                I understand the consequences and want to proceed
                            </label>
                        </div>

                        <input type="hidden" name="confirm_delete" value="1">

                        <div class="d-flex justify-content-between mt-4">
                            <a href="../../users.php" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-danger" id="submitBtn">
                                <i class="bi bi-trash"></i> <span id="btnText">Deactivate User</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const deactivateRadio = document.getElementById('deactivate');
    const permanentRadio = document.getElementById('permanent');
    const permanentWarning = document.getElementById('permanentWarning');
    const submitBtn = document.getElementById('submitBtn');
    const btnText = document.getElementById('btnText');
    const deleteForm = document.getElementById('deleteForm');

    function updateUI() {
        if (permanentRadio.checked) {
            permanentWarning.style.display = 'block';
            submitBtn.classList.remove('btn-danger');
            submitBtn.classList.add('btn-danger');
            btnText.textContent = 'Permanently Delete User';
        } else {
            permanentWarning.style.display = 'none';
            submitBtn.classList.remove('btn-danger');
            submitBtn.classList.add('btn-danger');
            btnText.textContent = 'Deactivate User';
        }
    }

    deactivateRadio.addEventListener('change', updateUI);
    permanentRadio.addEventListener('change', updateUI);

    deleteForm.addEventListener('submit', function(e) {
        if (permanentRadio.checked) {
            if (!confirm('Are you ABSOLUTELY SURE you want to PERMANENTLY DELETE this user?\n\nThis action CANNOT be undone!')) {
                e.preventDefault();
                return false;
            }
        } else {
            if (!confirm('Are you sure you want to deactivate this user?')) {
                e.preventDefault();
                return false;
            }
        }
    });
});
</script>

<?php include '../../includes/footer.php'; ?>
