<?php
/**
 * Add New User
 * Create new system user accounts with appropriate permissions
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_once '../../includes/error_logger.php';

require_authentication();

$page_title = 'Add New User';
$current_page = 'users';

// Check permission - minimum level 90 (HR Director)
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 90) {
    log_error('UNAUTHORIZED_ACCESS', 'User ' . $current_user['username'] . ' attempted to access add user page', [
        'user_id' => $current_user['id'],
        'permission_level' => $current_user['permission_level']
    ]);
    die('Access denied. HR Director permission (Level 90) or higher required.');
}

$errors = [];
$success = false;

// Permission level definitions
$permission_levels = [
    100 => 'System Administrator',
    90 => 'HR Director',
    80 => 'HR Manager',
    70 => 'Department Manager',
    60 => 'Supervisor',
    50 => 'Staff',
    40 => 'Read Only',
    30 => 'Limited'
];

// Fetch workers for optional linking
$workers = [];
try {
    $stmt = $pdo->query("SELECT id, employee_number, full_name, email
                         FROM " . DB_PREFIX . "sponsored_workers
                         WHERE status = 'active'
                         ORDER BY full_name ASC");
    $workers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    log_error('DATABASE_ERROR', 'Failed to fetch workers list', [
        'error' => $e->getMessage(),
        'file' => __FILE__,
        'line' => __LINE__
    ]);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate input
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $full_name = trim($_POST['full_name'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $user_role = trim($_POST['user_role'] ?? '');
    $permission_level = (int)($_POST['permission_level'] ?? 0);
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    $worker_id = !empty($_POST['worker_id']) ? (int)$_POST['worker_id'] : null;
    $status = $_POST['status'] ?? 'active';

    // Validation
    if (empty($username)) $errors[] = 'Username is required';
    if (empty($email)) $errors[] = 'Email is required';
    if (empty($full_name)) $errors[] = 'Full name is required';
    if (empty($user_role)) $errors[] = 'Role title is required';
    if (empty($permission_level)) $errors[] = 'Permission level is required';
    if (empty($password)) $errors[] = 'Password is required';

    // Username validation
    if (!empty($username)) {
        if (strlen($username) < 3) {
            $errors[] = 'Username must be at least 3 characters';
        }
        if (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
            $errors[] = 'Username can only contain letters, numbers, and underscores';
        }
    }

    // Email validation
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'Invalid email format';
    }

    // Password validation
    if (!empty($password)) {
        if (strlen($password) < 8) {
            $errors[] = 'Password must be at least 8 characters';
        }
        if ($password !== $confirm_password) {
            $errors[] = 'Passwords do not match';
        }
    }

    // Permission level validation - can't create user with higher permission than self
    if ($permission_level > $current_user['permission_level']) {
        $errors[] = 'You cannot create a user with higher permission level than your own';
        log_error('PERMISSION_VIOLATION', 'User attempted to create account with higher permission', [
            'creator_id' => $current_user['id'],
            'creator_level' => $current_user['permission_level'],
            'requested_level' => $permission_level
        ]);
    }

    // Check for duplicate username
    if (empty($errors) && !empty($username)) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "users WHERE username = ? LIMIT 1");
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                $errors[] = 'Username already exists';
            }
        } catch (PDOException $e) {
            log_error('DATABASE_ERROR', 'Failed to check duplicate username', [
                'username' => $username,
                'error' => $e->getMessage()
            ]);
            $errors[] = 'Database error occurred';
        }
    }

    // Check for duplicate email
    if (empty($errors) && !empty($email)) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "users WHERE email = ? LIMIT 1");
            $stmt->execute([$email]);
            if ($stmt->fetch()) {
                $errors[] = 'Email already exists';
            }
        } catch (PDOException $e) {
            log_error('DATABASE_ERROR', 'Failed to check duplicate email', [
                'email' => $email,
                'error' => $e->getMessage()
            ]);
            $errors[] = 'Database error occurred';
        }
    }

    // Create user if no errors
    if (empty($errors)) {
        try {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);

            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "users
                (username, email, password_hash, full_name, phone, user_role, permission_level,
                 worker_id, status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");

            $stmt->execute([
                $username,
                $email,
                $password_hash,
                $full_name,
                $phone,
                $user_role,
                $permission_level,
                $worker_id,
                $status
            ]);

            $new_user_id = $pdo->lastInsertId();

            log_error('USER_CREATED', 'New user account created', [
                'new_user_id' => $new_user_id,
                'username' => $username,
                'created_by' => $current_user['username'],
                'permission_level' => $permission_level
            ], 'info');

            $success = true;
            header('Location: ../../users.php?success=user_created');
            exit;

        } catch (PDOException $e) {
            log_error('DATABASE_ERROR', 'Failed to create user account', [
                'username' => $username,
                'email' => $email,
                'error' => $e->getMessage(),
                'file' => __FILE__,
                'line' => __LINE__
            ]);
            $errors[] = 'Failed to create user account. Please try again.';
        }
    }
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-person-plus-fill"></i> Add New User
            </h1>
            <p class="welcome-subtitle">Create a new system user account</p>
        </div>
        <div class="welcome-actions">
            <a href="../../users.php" class="btn btn-outline-secondary">
                <i class="bi bi-arrow-left"></i> Back to Users
            </a>
        </div>
    </div>

    <?php if (!empty($errors)): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle-fill"></i> <strong>Please correct the following errors:</strong>
            <ul class="mb-0 mt-2">
                <?php foreach ($errors as $error): ?>
                    <li><?php echo htmlspecialchars($error); ?></li>
                <?php endforeach; ?>
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <div class="modern-card">
        <div class="card-body-modern">
            <form method="POST" action="">
                <div class="row">
                    <!-- Account Information -->
                    <div class="col-md-6">
                        <h5 class="mb-3"><i class="bi bi-person-circle"></i> Account Information</h5>

                        <div class="mb-3">
                            <label class="form-label">Username <span class="text-danger">*</span></label>
                            <input type="text" name="username" class="form-control" required
                                   value="<?php echo htmlspecialchars($_POST['username'] ?? ''); ?>"
                                   pattern="[a-zA-Z0-9_]{3,}" title="3+ characters, letters, numbers, underscores only">
                            <small class="text-muted">3+ characters, letters, numbers, and underscores only</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Full Name <span class="text-danger">*</span></label>
                            <input type="text" name="full_name" class="form-control" required
                                   value="<?php echo htmlspecialchars($_POST['full_name'] ?? ''); ?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Email <span class="text-danger">*</span></label>
                            <input type="email" name="email" class="form-control" required
                                   value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Phone</label>
                            <input type="tel" name="phone" class="form-control"
                                   value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Status <span class="text-danger">*</span></label>
                            <select name="status" class="form-select" required>
                                <option value="active" <?php echo ($_POST['status'] ?? 'active') === 'active' ? 'selected' : ''; ?>>Active</option>
                                <option value="inactive" <?php echo ($_POST['status'] ?? '') === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                            </select>
                        </div>
                    </div>

                    <!-- Permissions and Role -->
                    <div class="col-md-6">
                        <h5 class="mb-3"><i class="bi bi-shield-check"></i> Permissions & Role</h5>

                        <div class="mb-3">
                            <label class="form-label">Permission Level <span class="text-danger">*</span></label>
                            <select name="permission_level" id="permission_level" class="form-select" required>
                                <option value="">Select permission level...</option>
                                <?php foreach ($permission_levels as $level => $name): ?>
                                    <?php if ($level <= $current_user['permission_level']): ?>
                                        <option value="<?php echo $level; ?>"
                                            <?php echo ($_POST['permission_level'] ?? '') == $level ? 'selected' : ''; ?>>
                                            Level <?php echo $level; ?> - <?php echo htmlspecialchars($name); ?>
                                        </option>
                                    <?php endif; ?>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">You can only assign permission levels up to your own (<?php echo $current_user['permission_level']; ?>)</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Role Title <span class="text-danger">*</span></label>
                            <input type="text" name="user_role" id="user_role" class="form-control" required
                                   value="<?php echo htmlspecialchars($_POST['user_role'] ?? ''); ?>"
                                   placeholder="e.g., HR Manager, System Admin, etc.">
                            <small class="text-muted">Job title or role description</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Link to Worker (Optional)</label>
                            <select name="worker_id" class="form-select">
                                <option value="">Not linked to any worker</option>
                                <?php foreach ($workers as $worker): ?>
                                    <option value="<?php echo $worker['id']; ?>"
                                        <?php echo ($_POST['worker_id'] ?? '') == $worker['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($worker['full_name']); ?>
                                        (<?php echo htmlspecialchars($worker['employee_number']); ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Link this user account to an existing worker record</small>
                        </div>
                    </div>
                </div>

                <hr class="my-4">

                <!-- Password Section -->
                <div class="row">
                    <div class="col-md-12">
                        <h5 class="mb-3"><i class="bi bi-key-fill"></i> Password</h5>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label">Password <span class="text-danger">*</span></label>
                            <input type="password" name="password" class="form-control" required minlength="8">
                            <small class="text-muted">Minimum 8 characters</small>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label">Confirm Password <span class="text-danger">*</span></label>
                            <input type="password" name="confirm_password" class="form-control" required minlength="8">
                        </div>
                    </div>
                </div>

                <hr class="my-4">

                <!-- Submit Buttons -->
                <div class="d-flex justify-content-between">
                    <a href="../../users.php" class="btn btn-secondary">
                        <i class="bi bi-x-circle"></i> Cancel
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-check-circle"></i> Create User
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Auto-populate role title based on permission level
document.getElementById('permission_level').addEventListener('change', function() {
    const level = parseInt(this.value);
    const roleField = document.getElementById('user_role');

    // Only auto-populate if field is empty
    if (roleField.value.trim() === '') {
        const roleNames = {
            100: 'System Administrator',
            90: 'HR Director',
            80: 'HR Manager',
            70: 'Department Manager',
            60: 'Supervisor',
            50: 'Staff Member',
            40: 'Viewer',
            30: 'Limited User'
        };

        if (roleNames[level]) {
            roleField.value = roleNames[level];
        }
    }
});
</script>

<?php include '../../includes/footer.php'; ?>
