<?php
/**
 * Module Settings Page
 * Handles settings for individual modules
 */

session_start();
require_once 'config/database.php';
require_once 'includes/database.php';
require_once 'includes/session_manager.php';
require_once 'includes/ModuleManager.php';

// Require authentication
require_authentication();

// Check admin permission
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 100) {
    die('Access denied. Admin permission required.');
}

// Get module ID from URL
$module_id = $_GET['module'] ?? '';

if (empty($module_id)) {
    header('Location: settings.php?tab=modules');
    exit;
}

// Initialize Module Manager
$moduleManager = new ModuleManager($pdo);
$module = null;

// Get module information
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "modules_registry WHERE module_id = ?");
$stmt->execute([$module_id]);
$module = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$module) {
    die('Module not found');
}

$page_title = $module['name'] . ' Settings';
$current_page = 'settings';

// Check if module has custom settings page
$custom_settings_file = "modules/{$module_id}/settings.php";
if (file_exists($custom_settings_file)) {
    // Redirect to module's custom settings page
    header("Location: modules/{$module_id}/settings.php");
    exit;
}

// Handle form submission for generic settings
$message = '';
$message_type = 'success';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $_POST['action'] === 'save_settings') {
    try {
        $settings = $_POST['settings'] ?? [];

        // Check if sidebar_display_name is being updated
        if (isset($settings['sidebar_display_name'])) {
            $new_sidebar_name = trim($settings['sidebar_display_name']);
            if (!empty($new_sidebar_name)) {
                // Update module_name in modules_registry table for sidebar
                $update_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "modules_registry
                                              SET module_name = ?
                                              WHERE module_id = ?");
                $update_stmt->execute([$new_sidebar_name, $module_id]);
            }
        }

        // Save each setting
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                              VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");

        foreach ($settings as $key => $value) {
            // Prefix with module ID to avoid conflicts
            $setting_key = $module_id . '_' . $key;

            // Handle checkboxes
            if (is_array($value)) {
                $value = implode(',', $value);
            }

            $stmt->execute([$setting_key, $value]);
        }

        $message = "Settings saved successfully";
        $message_type = 'success';

    } catch (PDOException $e) {
        $message = "Error saving settings: " . $e->getMessage();
        $message_type = 'danger';
    }
}

// Get current settings for this module
$settings = [];
$stmt = $pdo->prepare("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings
                      WHERE setting_key LIKE ?");
$stmt->execute([$module_id . '_%']);
while ($row = $stmt->fetch()) {
    // Remove module prefix from key
    $key = str_replace($module_id . '_', '', $row['setting_key']);
    $settings[$key] = $row['setting_value'];
}

// Add current sidebar display name from modules_registry
if (!isset($settings['sidebar_display_name'])) {
    $settings['sidebar_display_name'] = $module['module_name'] ?? $module['name'] ?? '';
}

// Get module info for settings fields
$module_info = $moduleManager->getModuleInfo($module_id);

include 'includes/header.php';
?>

<div class="dashboard-container">
    <!-- Page Header -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi <?= htmlspecialchars($module['icon'] ?? 'bi-gear') ?>"></i>
                <?= htmlspecialchars($module['name']) ?> Settings
            </h1>
            <p class="welcome-subtitle">Configure settings for <?= htmlspecialchars($module['name']) ?> module</p>
        </div>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-<?= $message_type ?> alert-dismissible fade show" role="alert">
        <i class="bi bi-<?= $message_type === 'success' ? 'check-circle' : 'exclamation-triangle' ?>-fill"></i>
        <?= htmlspecialchars($message) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <div class="modern-card">
        <div class="card-body-modern">

            <?php if (!empty($module_info['settings_fields'])): ?>
            <!-- Module has defined settings -->
            <form method="POST">
                <input type="hidden" name="action" value="save_settings">

                <?php foreach ($module_info['settings_fields'] as $field): ?>
                <div class="mb-4">
                    <?php if ($field['type'] === 'section'): ?>
                        <h5 class="border-bottom pb-2 mb-3">
                            <?php if (isset($field['icon'])): ?>
                                <i class="bi <?= htmlspecialchars($field['icon']) ?>"></i>
                            <?php endif; ?>
                            <?= htmlspecialchars($field['label']) ?>
                        </h5>
                        <?php if (isset($field['description'])): ?>
                            <p class="text-muted"><?= htmlspecialchars($field['description']) ?></p>
                        <?php endif; ?>

                    <?php elseif ($field['type'] === 'text'): ?>
                        <label class="form-label">
                            <strong><?= htmlspecialchars($field['label']) ?></strong>
                            <?php if (isset($field['required']) && $field['required']): ?>
                                <span class="text-danger">*</span>
                            <?php endif; ?>
                        </label>
                        <input type="text"
                               class="form-control"
                               name="settings[<?= htmlspecialchars($field['name']) ?>]"
                               value="<?= htmlspecialchars($settings[$field['name']] ?? $field['default'] ?? '') ?>"
                               <?= (isset($field['required']) && $field['required']) ? 'required' : '' ?>>
                        <?php if (isset($field['help'])): ?>
                            <small class="text-muted"><?= htmlspecialchars($field['help']) ?></small>
                        <?php endif; ?>

                    <?php elseif ($field['type'] === 'number'): ?>
                        <label class="form-label">
                            <strong><?= htmlspecialchars($field['label']) ?></strong>
                            <?php if (isset($field['required']) && $field['required']): ?>
                                <span class="text-danger">*</span>
                            <?php endif; ?>
                        </label>
                        <input type="number"
                               class="form-control"
                               name="settings[<?= htmlspecialchars($field['name']) ?>]"
                               value="<?= htmlspecialchars($settings[$field['name']] ?? $field['default'] ?? '') ?>"
                               <?= isset($field['min']) ? 'min="' . $field['min'] . '"' : '' ?>
                               <?= isset($field['max']) ? 'max="' . $field['max'] . '"' : '' ?>
                               <?= (isset($field['required']) && $field['required']) ? 'required' : '' ?>>
                        <?php if (isset($field['help'])): ?>
                            <small class="text-muted"><?= htmlspecialchars($field['help']) ?></small>
                        <?php endif; ?>

                    <?php elseif ($field['type'] === 'checkbox'): ?>
                        <div class="form-check">
                            <input type="checkbox"
                                   class="form-check-input"
                                   id="<?= htmlspecialchars($field['name']) ?>"
                                   name="settings[<?= htmlspecialchars($field['name']) ?>]"
                                   value="1"
                                   <?= (!empty($settings[$field['name']])) ? 'checked' : '' ?>>
                            <label class="form-check-label" for="<?= htmlspecialchars($field['name']) ?>">
                                <strong><?= htmlspecialchars($field['label']) ?></strong>
                            </label>
                        </div>
                        <?php if (isset($field['help'])): ?>
                            <small class="text-muted d-block mt-1"><?= htmlspecialchars($field['help']) ?></small>
                        <?php endif; ?>

                    <?php elseif ($field['type'] === 'select'): ?>
                        <label class="form-label">
                            <strong><?= htmlspecialchars($field['label']) ?></strong>
                            <?php if (isset($field['required']) && $field['required']): ?>
                                <span class="text-danger">*</span>
                            <?php endif; ?>
                        </label>
                        <select class="form-select"
                                name="settings[<?= htmlspecialchars($field['name']) ?>]"
                                <?= (isset($field['required']) && $field['required']) ? 'required' : '' ?>>
                            <?php foreach ($field['options'] as $value => $label): ?>
                                <option value="<?= htmlspecialchars($value) ?>"
                                        <?= ($settings[$field['name']] ?? $field['default'] ?? '') == $value ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($label) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <?php if (isset($field['help'])): ?>
                            <small class="text-muted"><?= htmlspecialchars($field['help']) ?></small>
                        <?php endif; ?>

                    <?php elseif ($field['type'] === 'textarea'): ?>
                        <label class="form-label">
                            <strong><?= htmlspecialchars($field['label']) ?></strong>
                            <?php if (isset($field['required']) && $field['required']): ?>
                                <span class="text-danger">*</span>
                            <?php endif; ?>
                        </label>
                        <textarea class="form-control"
                                  name="settings[<?= htmlspecialchars($field['name']) ?>]"
                                  rows="<?= $field['rows'] ?? 3 ?>"
                                  <?= (isset($field['required']) && $field['required']) ? 'required' : '' ?>><?= htmlspecialchars($settings[$field['name']] ?? $field['default'] ?? '') ?></textarea>
                        <?php if (isset($field['help'])): ?>
                            <small class="text-muted"><?= htmlspecialchars($field['help']) ?></small>
                        <?php endif; ?>

                    <?php endif; ?>
                </div>
                <?php endforeach; ?>

                <div class="d-flex justify-content-between">
                    <a href="settings.php?tab=modules" class="btn btn-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Modules
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Settings
                    </button>
                </div>
            </form>

            <?php else: ?>
            <!-- No settings defined for this module -->
            <div class="text-center py-5">
                <i class="bi bi-gear" style="font-size: 3rem; color: var(--text-muted);"></i>
                <p class="mt-3">This module does not have configurable settings.</p>

                <?php if (file_exists("modules/{$module_id}/")): ?>
                    <div class="mt-4">
                        <a href="modules/<?= htmlspecialchars($module_id) ?>/list.php" class="btn btn-primary">
                            <i class="bi bi-box-arrow-up-right"></i> Go to Module
                        </a>
                    </div>
                <?php endif; ?>

                <div class="mt-3">
                    <a href="settings.php?tab=modules" class="btn btn-secondary">
                        <i class="bi bi-arrow-left"></i> Back to Modules
                    </a>
                </div>
            </div>
            <?php endif; ?>

        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>