<?php
/**
 * Comprehensive Workers Table Migration
 * Adds all UK Home Office compliance fields
 */

require_once 'config/database.php';
require_once 'includes/database.php';

echo "Migrating workers table to comprehensive UK Home Office compliance structure...\n\n";

try {
    // Add Person/Legal fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS legal_name VARCHAR(255) AFTER full_name,
        ADD COLUMN IF NOT EXISTS aka_names TEXT AFTER legal_name,
        ADD COLUMN IF NOT EXISTS title VARCHAR(10) AFTER employee_number,
        ADD COLUMN IF NOT EXISTS first_name VARCHAR(100) AFTER title,
        ADD COLUMN IF NOT EXISTS middle_names VARCHAR(200) AFTER first_name,
        ADD COLUMN IF NOT EXISTS last_name VARCHAR(100) AFTER middle_names
    ");
    echo "✓ Person/Legal name fields added\n";

    // Add Contact & Address fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS mobile VARCHAR(20) AFTER phone,
        ADD COLUMN IF NOT EXISTS address_line1 VARCHAR(255) AFTER email,
        ADD COLUMN IF NOT EXISTS address_line2 VARCHAR(255) AFTER address_line1,
        ADD COLUMN IF NOT EXISTS city VARCHAR(100) AFTER address_line2,
        ADD COLUMN IF NOT EXISTS county VARCHAR(100) AFTER city,
        ADD COLUMN IF NOT EXISTS postcode VARCHAR(20) AFTER county,
        ADD COLUMN IF NOT EXISTS country VARCHAR(100) DEFAULT 'United Kingdom' AFTER postcode
    ");
    echo "✓ Contact & Address fields added\n";

    // Add Employment fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS soc_code VARCHAR(20) AFTER job_title,
        ADD COLUMN IF NOT EXISTS site_locations TEXT AFTER department,
        ADD COLUMN IF NOT EXISTS employment_start_date DATE AFTER start_date,
        ADD COLUMN IF NOT EXISTS employment_status ENUM('active','inactive','suspended','on_leave','terminated') DEFAULT 'active' AFTER status,
        ADD COLUMN IF NOT EXISTS contracted_hours_per_week DECIMAL(5,2) AFTER employment_status,
        ADD COLUMN IF NOT EXISTS working_pattern VARCHAR(100) AFTER contracted_hours_per_week,
        ADD COLUMN IF NOT EXISTS line_manager_id INT AFTER working_pattern
    ");
    echo "✓ Employment fields added\n";

    // Add Pay fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS base_salary DECIMAL(10,2) AFTER contracted_hours_per_week,
        ADD COLUMN IF NOT EXISTS pay_frequency ENUM('hourly','weekly','monthly','annually') DEFAULT 'annually' AFTER base_salary,
        ADD COLUMN IF NOT EXISTS allowances TEXT AFTER pay_frequency,
        ADD COLUMN IF NOT EXISTS overtime_eligible TINYINT(1) DEFAULT 0 AFTER allowances
    ");
    echo "✓ Pay fields added\n";

    // Add Right to Work (RTW) fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS rtw_route ENUM('List A','List B Group 1','List B Group 2') AFTER overtime_eligible,
        ADD COLUMN IF NOT EXISTS rtw_method ENUM('manual','online','idsp') AFTER rtw_route,
        ADD COLUMN IF NOT EXISTS rtw_document_type VARCHAR(100) AFTER rtw_method,
        ADD COLUMN IF NOT EXISTS rtw_document_number VARCHAR(100) AFTER rtw_document_type,
        ADD COLUMN IF NOT EXISTS rtw_share_code VARCHAR(20) AFTER rtw_document_number,
        ADD COLUMN IF NOT EXISTS rtw_issue_date DATE AFTER rtw_share_code,
        ADD COLUMN IF NOT EXISTS rtw_expiry_date DATE AFTER rtw_issue_date,
        ADD COLUMN IF NOT EXISTS rtw_check_date DATE AFTER rtw_expiry_date,
        ADD COLUMN IF NOT EXISTS rtw_checker_name VARCHAR(100) AFTER rtw_check_date,
        ADD COLUMN IF NOT EXISTS rtw_result_ref VARCHAR(100) AFTER rtw_checker_name,
        ADD COLUMN IF NOT EXISTS rtw_restrictions TEXT AFTER rtw_result_ref,
        ADD COLUMN IF NOT EXISTS rtw_follow_up_required TINYINT(1) DEFAULT 0 AFTER rtw_restrictions,
        ADD COLUMN IF NOT EXISTS rtw_follow_up_date DATE AFTER rtw_follow_up_required,
        ADD COLUMN IF NOT EXISTS rtw_document_copy_path VARCHAR(500) AFTER rtw_follow_up_date
    ");
    echo "✓ Right to Work fields added\n";

    // Add Sponsorship fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS is_sponsored TINYINT(1) DEFAULT 0 AFTER rtw_document_copy_path,
        ADD COLUMN IF NOT EXISTS cos_route VARCHAR(100) AFTER is_sponsored,
        ADD COLUMN IF NOT EXISTS cos_status ENUM('assigned','used','withdrawn','expired') AFTER cos_route,
        ADD COLUMN IF NOT EXISTS cos_assigned_date DATE AFTER cos_status,
        ADD COLUMN IF NOT EXISTS cos_used_date DATE AFTER cos_assigned_date,
        ADD COLUMN IF NOT EXISTS cos_soc_code VARCHAR(20) AFTER cos_used_date,
        ADD COLUMN IF NOT EXISTS cos_job_description TEXT AFTER cos_soc_code,
        ADD COLUMN IF NOT EXISTS cos_salary DECIMAL(10,2) AFTER cos_job_description,
        ADD COLUMN IF NOT EXISTS cos_going_rate_basis VARCHAR(100) AFTER cos_salary,
        ADD COLUMN IF NOT EXISTS cos_work_locations TEXT AFTER cos_going_rate_basis,
        ADD COLUMN IF NOT EXISTS visa_category VARCHAR(100) AFTER cos_work_locations,
        ADD COLUMN IF NOT EXISTS visa_permission_start DATE AFTER visa_category,
        ADD COLUMN IF NOT EXISTS visa_permission_expiry DATE AFTER visa_permission_start,
        ADD COLUMN IF NOT EXISTS visa_conditions TEXT AFTER visa_permission_expiry,
        ADD COLUMN IF NOT EXISTS recruitment_evidence TEXT AFTER visa_conditions,
        ADD COLUMN IF NOT EXISTS qualifications_on_file TEXT AFTER recruitment_evidence
    ");
    echo "✓ Sponsorship fields added\n";

    // Add Absence tracking fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS first_day_absence_date DATE AFTER qualifications_on_file,
        ADD COLUMN IF NOT EXISTS absence_reported_to_ukvi TINYINT(1) DEFAULT 0 AFTER first_day_absence_date,
        ADD COLUMN IF NOT EXISTS absence_report_date DATE AFTER absence_reported_to_ukvi
    ");
    echo "✓ Absence tracking fields added\n";

    // Add Audit fields
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD COLUMN IF NOT EXISTS created_by INT AFTER created_at,
        ADD COLUMN IF NOT EXISTS updated_at TIMESTAMP NULL AFTER created_by,
        ADD COLUMN IF NOT EXISTS updated_by INT AFTER updated_at,
        ADD COLUMN IF NOT EXISTS retention_end_date DATE AFTER updated_by
    ");
    echo "✓ Audit fields added\n";

    // Create indexes for performance
    $pdo->exec("ALTER TABLE scms_sponsored_workers
        ADD INDEX idx_rtw_follow_up (rtw_follow_up_date),
        ADD INDEX idx_visa_expiry (visa_permission_expiry),
        ADD INDEX idx_cos_status (cos_status),
        ADD INDEX idx_employment_status (employment_status),
        ADD INDEX idx_is_sponsored (is_sponsored)
    ");
    echo "✓ Performance indexes added\n";

    echo "\n✅ Migration completed successfully!\n";
    echo "\nNew table supports:\n";
    echo "- Core worker profiles (legal name, DOB, NI, contacts)\n";
    echo "- Employment details (job, SOC, hours, pay)\n";
    echo "- Right to Work compliance (List A/B, checks, follow-ups)\n";
    echo "- Sponsorship tracking (CoS, visa, recruitment evidence)\n";
    echo "- Absence monitoring (UKVI reporting)\n";
    echo "- Full audit trail\n";

} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
}
