<?php
/**
 * Holidays Module - List View
 * Shows UK bank holidays and allows management
 */

$page_title = 'Holiday Management';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

// Get current year or from URL
$year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
$year = max(2020, min(2030, $year));

// Fetch bank holidays for the year
$holidays_sql = "SELECT * FROM " . DB_PREFIX . "bank_holidays WHERE YEAR(date) = ? ORDER BY date ASC";
$stmt = $pdo->prepare($holidays_sql);
$stmt->execute([$year]);
$holidays = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get holiday statistics
$stats_sql = "SELECT
    COUNT(*) as total_holidays,
    SUM(CASE WHEN date >= CURDATE() THEN 1 ELSE 0 END) as upcoming_holidays,
    SUM(CASE WHEN date < CURDATE() THEN 1 ELSE 0 END) as past_holidays
FROM " . DB_PREFIX . "bank_holidays WHERE YEAR(date) = ?";
$stats_stmt = $pdo->prepare($stats_sql);
$stats_stmt->execute([$year]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>

<div class="stat-card">
    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h5><i class="bi bi-calendar-event"></i> UK Bank Holidays <?= $year ?></h5>
            <p class="text-muted mb-0">Manage statutory bank holidays</p>
        </div>
        <div class="d-flex gap-2">
            <?php if (($_SESSION['permission_level'] ?? 0) >= 90): ?>
            <a href="add.php?year=<?= $year ?>" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Add Holiday
            </a>
            <?php endif; ?>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h6 class="card-title">Total Holidays</h6>
                    <h2 class="mb-0"><?= $stats['total_holidays'] ?? 0 ?></h2>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <h6 class="card-title">Upcoming</h6>
                    <h2 class="mb-0"><?= $stats['upcoming_holidays'] ?? 0 ?></h2>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card bg-secondary text-white">
                <div class="card-body">
                    <h6 class="card-title">Past Holidays</h6>
                    <h2 class="mb-0"><?= $stats['past_holidays'] ?? 0 ?></h2>
                </div>
            </div>
        </div>
    </div>

    <!-- Year Navigation -->
    <div class="d-flex justify-content-center mb-4">
        <div class="btn-group">
            <a href="?year=<?= $year - 1 ?>" class="btn btn-outline-secondary">
                <i class="bi bi-chevron-left"></i> <?= $year - 1 ?>
            </a>
            <button class="btn btn-outline-primary" disabled><?= $year ?></button>
            <a href="?year=<?= $year + 1 ?>" class="btn btn-outline-secondary">
                <?= $year + 1 ?> <i class="bi bi-chevron-right"></i>
            </a>
        </div>
    </div>

    <!-- Holidays Table -->
    <?php if (empty($holidays)): ?>
        <div class="alert alert-info text-center">
            <i class="bi bi-calendar-x" style="font-size: 48px;"></i>
            <p class="mt-3 mb-2"><strong>No Bank Holidays Defined for <?= $year ?></strong></p>
            <p class="text-muted">Add UK statutory bank holidays for this year.</p>
            <?php if (($_SESSION['permission_level'] ?? 0) >= 90): ?>
            <a href="add.php?year=<?= $year ?>" class="btn btn-primary mt-2">
                <i class="bi bi-plus-circle"></i> Add First Holiday
            </a>
            <?php endif; ?>
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-hover">
                <thead class="table-light">
                    <tr>
                        <th width="15%">Date</th>
                        <th width="20%">Day</th>
                        <th width="40%">Holiday Name</th>
                        <th width="15%">Status</th>
                        <?php if (($_SESSION['permission_level'] ?? 0) >= 90): ?>
                        <th width="10%">Actions</th>
                        <?php endif; ?>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($holidays as $holiday):
                        $date = new DateTime($holiday['date']);
                        $day_name = $date->format('l');
                        $is_past = $date < new DateTime('today');
                        $is_today = $date->format('Y-m-d') === date('Y-m-d');
                        $formatted_date = $date->format('d M Y');
                    ?>
                    <tr class="<?= $is_today ? 'table-warning' : '' ?>">
                        <td>
                            <strong><?= htmlspecialchars($formatted_date) ?></strong>
                        </td>
                        <td>
                            <span class="badge bg-secondary"><?= htmlspecialchars($day_name) ?></span>
                        </td>
                        <td>
                            <i class="bi bi-calendar-event"></i>
                            <?= htmlspecialchars($holiday['name']) ?>
                        </td>
                        <td>
                            <?php if ($is_today): ?>
                                <span class="badge bg-warning">Today</span>
                            <?php elseif ($is_past): ?>
                                <span class="badge bg-secondary">Past</span>
                            <?php else: ?>
                                <span class="badge bg-success">Upcoming</span>
                            <?php endif; ?>
                        </td>
                        <?php if (($_SESSION['permission_level'] ?? 0) >= 90): ?>
                        <td>
                            <a href="edit.php?id=<?= $holiday['id'] ?>"
                               class="btn btn-sm btn-outline-warning"
                               title="Edit">
                                <i class="bi bi-pencil"></i>
                            </a>
                            <a href="delete.php?id=<?= $holiday['id'] ?>"
                               class="btn btn-sm btn-outline-danger"
                               title="Delete"
                               onclick="return confirm('Are you sure you want to delete this holiday?')">
                                <i class="bi bi-trash"></i>
                            </a>
                        </td>
                        <?php endif; ?>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>

    <!-- Information Box -->
    <div class="alert alert-info mt-4">
        <h6><i class="bi bi-info-circle"></i> About UK Bank Holidays</h6>
        <small>
            Bank holidays are public holidays in the United Kingdom when banks and most businesses are closed.
            These dates are used throughout the system for:
        </small>
        <ul class="small mt-2 mb-0">
            <li>Attendance tracking - automatically marking holidays</li>
            <li>Leave calculations - excluding bank holidays from annual leave</li>
            <li>Working days calculations in various modules</li>
            <li>Payroll and scheduling systems</li>
        </ul>
    </div>
</div>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>
