<?php
/**
 * RIU Integration Example
 * How to integrate RIU into your SCMS installer
 */

// Include RIU configuration and client
require_once __DIR__ . '/../includes/riu_config.php';
require_once __DIR__ . '/../includes/RIUClient.php';

/**
 * STEP 1: LICENSE VERIFICATION WITH RIU
 * Replace your existing license verification with this
 */
function verify_license_with_riu($license_key, $domain)
{
    // Initialize RIU Client
    $riu = new RIUClient(
        RIU_CONTROL_PLANE_URL,
        RIU_PRODUCT_ID,
        $license_key,
        $domain
    );

    // Verify license
    $result = $riu->verifyLicense();

    if ($result['success']) {
        // License is valid, store access token in session
        $_SESSION['liu_verified'] = true;
        $_SESSION['riu_client'] = serialize($riu);

        return [
            'success' => true,
            'message' => 'License verified successfully',
            'instance_id' => $riu->getInstanceId(),
            'access_token' => $riu->getAccessToken()
        ];
    } else {
        return [
            'success' => false,
            'error' => $result['error'] ?? 'License verification failed'
        ];
    }
}

/**
 * STEP 2: CHOOSE INSTALLATION MODE
 * Add this to your installation mode selection (Step 2.5)
 */
function show_installation_mode_selection()
{
    ?>
    <div class="installation-mode-selection">
        <h3>Choose Installation Method</h3>

        <div class="mode-option">
            <input type="radio" name="install_mode" value="local" id="mode_local">
            <label for="mode_local">
                <strong>Install from Local Files</strong>
                <p>Files are already uploaded to this server</p>
            </label>
        </div>

        <div class="mode-option">
            <input type="radio" name="install_mode" value="remote" id="mode_remote" checked>
            <label for="mode_remote">
                <strong>Install from Remote Server (RIU)</strong>
                <p>Download files from update server (recommended)</p>
            </label>
        </div>
    </div>
    <?php
}

/**
 * STEP 3: DOWNLOAD FILES FROM RIU
 * Use this when user selects "remote" installation
 */
function download_files_from_riu($version = null)
{
    $version = $version ?? RIU_DEFAULT_VERSION;

    // Restore RIU client from session
    if (!isset($_SESSION['riu_client'])) {
        return ['success' => false, 'error' => 'Not authenticated. Please verify license first.'];
    }

    $riu = unserialize($_SESSION['riu_client']);

    // Step 1: Get manifest
    $manifest_result = $riu->getManifest($version);

    if (!$manifest_result['success']) {
        return [
            'success' => false,
            'error' => 'Failed to get release manifest',
            'details' => $manifest_result['error'] ?? 'Unknown error'
        ];
    }

    $manifest = $manifest_result['manifest'];
    $files = $manifest['files'] ?? [];
    $total_files = count($files);

    // Option A: Download individual files (with progress tracking)
    return download_files_individually($riu, $version, $files);

    // Option B: Download as archive (faster but no progress tracking)
    // return download_as_archive($riu, $version);
}

/**
 * Download files one by one with progress tracking
 */
function download_files_individually($riu, $version, $files)
{
    $total_files = count($files);
    $downloaded = 0;
    $failed = [];

    // Get installation root (one level up from install directory)
    $install_root = dirname(__DIR__);

    foreach ($files as $file) {
        $file_path = $file['path'];
        $destination = $install_root . '/' . $file_path;

        // Download file
        $result = $riu->downloadFile($version, $file_path, $destination);

        if ($result['success']) {
            $downloaded++;

            // Set file permissions if specified
            if (isset($file['permissions'])) {
                chmod($destination, octdec($file['permissions']));
            }

            // Send progress update (for AJAX)
            if (function_exists('send_progress_update')) {
                send_progress_update([
                    'current' => $downloaded,
                    'total' => $total_files,
                    'file' => $file_path,
                    'percentage' => round(($downloaded / $total_files) * 100)
                ]);
            }
        } else {
            $failed[] = [
                'file' => $file_path,
                'error' => $result['error'] ?? 'Unknown error'
            ];
        }
    }

    if (count($failed) > 0) {
        return [
            'success' => false,
            'error' => 'Some files failed to download',
            'downloaded' => $downloaded,
            'total' => $total_files,
            'failed' => $failed
        ];
    }

    return [
        'success' => true,
        'message' => "Downloaded {$downloaded} files successfully",
        'total' => $total_files
    ];
}

/**
 * Download as complete archive (faster)
 */
function download_as_archive($riu, $version)
{
    // Create temp directory
    $temp_dir = RIU_TEMP_DIR;
    if (!is_dir($temp_dir)) {
        mkdir($temp_dir, 0755, true);
    }

    $archive_path = $temp_dir . '/scms-' . $version . '.tar.gz';

    // Download archive
    $result = $riu->downloadArchive($version, $archive_path);

    if (!$result['success']) {
        return [
            'success' => false,
            'error' => 'Failed to download archive',
            'details' => $result['error'] ?? 'Unknown error'
        ];
    }

    // Extract archive to installation root
    $install_root = dirname(__DIR__);
    $extract_result = $riu->extractArchive($archive_path, $install_root);

    if (!$extract_result['success']) {
        return [
            'success' => false,
            'error' => 'Failed to extract archive',
            'details' => $extract_result['error'] ?? 'Unknown error'
        ];
    }

    // Clean up
    unlink($archive_path);

    return [
        'success' => true,
        'message' => 'Files downloaded and extracted successfully'
    ];
}

/**
 * STEP 4: AJAX ENDPOINT FOR PROGRESS TRACKING
 * Add this to handle AJAX requests during download
 */
if (isset($_GET['ajax']) && $_GET['ajax'] === 'download_progress') {
    session_start();
    header('Content-Type: application/json');

    $action = $_POST['action'] ?? '';

    if ($action === 'start_download') {
        $version = $_POST['version'] ?? RIU_DEFAULT_VERSION;

        // Start download in background or use chunked response
        $result = download_files_from_riu($version);

        echo json_encode($result);
        exit;
    }

    echo json_encode(['success' => false, 'error' => 'Invalid action']);
    exit;
}

/**
 * COMPLETE INSTALLATION FLOW WITH RIU
 */
function complete_installation_flow()
{
    // Step 1: Verify License
    $license_key = $_POST['license_key'];
    $domain = $_SERVER['HTTP_HOST'];

    $verify_result = verify_license_with_riu($license_key, $domain);

    if (!$verify_result['success']) {
        return ['error' => $verify_result['error']];
    }

    // Step 2: Choose installation mode
    $install_mode = $_POST['install_mode']; // 'local' or 'remote'

    if ($install_mode === 'remote') {
        // Step 3: Download files from RIU
        $download_result = download_files_from_riu();

        if (!$download_result['success']) {
            return ['error' => $download_result['error']];
        }
    }

    // Step 4: Continue with database setup (your existing code)
    // ... database configuration ...
    // ... run SQL migrations ...
    // ... create admin user ...

    return ['success' => true];
}

/**
 * USAGE IN YOUR EXISTING INSTALLER
 */

// In install_step1.php - License Verification
// Replace old license check with:
/*
if (isset($_POST['submit_license'])) {
    $license_key = trim($_POST['license_key']);
    $domain = extract_root_domain($_SERVER['HTTP_HOST']);

    $result = verify_license_with_riu($license_key, $domain);

    if ($result['success']) {
        header('Location: install.php?step=2');
        exit;
    } else {
        $errors[] = $result['error'];
    }
}
*/

// In install_step2_5_remote.php - Remote Installation
// Replace file download logic with:
/*
if (isset($_POST['start_download'])) {
    $version = $_POST['version'] ?? RIU_DEFAULT_VERSION;

    $result = download_files_from_riu($version);

    if ($result['success']) {
        header('Location: install.php?step=3');
        exit;
    } else {
        $errors[] = $result['error'];
    }
}
*/
