<?php
/**
 * RIU Integration for Installation Process
 * Include this file in install.php to enable progress reporting
 */

// Only load if in installation context
if (!defined('INSTALL_CONTEXT')) {
    define('INSTALL_CONTEXT', true);
}

// Load RIU helper only if database config exists
$riu_helper_loaded = false;
if (file_exists(__DIR__ . '/config/database.php') && file_exists(__DIR__ . '/includes/riu_helper.php')) {
    try {
        require_once __DIR__ . '/includes/riu_helper.php';
        $riu_helper_loaded = true;
    } catch (Exception $e) {
        // RIU helper not available yet - continue without it
        $riu_helper_loaded = false;
    }
}

// Installation progress tracking
function track_install_progress($step, $action = 'start') {
    static $steps = array(
        0 => 'System Requirements Check',
        1 => 'License Validation',
        2 => 'Installation Mode Selection',
        3 => 'File Operations',
        4 => 'Database Configuration',
        5 => 'Business Profile',
        6 => 'Admin Account Creation'
    );

    $total_steps = 7;
    $current_step_num = $step;
    $step_name = $steps[$step] ?? 'Unknown Step';

    // Try to report via RIU directly using session data
    if (isset($_SESSION['riu_instance_id']) && isset($_SESSION['riu_access_token'])) {
        try {
            // Build request data
            $data = array(
                'instance_id' => $_SESSION['riu_instance_id'],
                'type' => 'install',
                'status' => ($action === 'complete') ? 'running' : (($action === 'error') ? 'failed' : 'running'),
                'message' => ($action === 'start') ? "Starting: {$step_name}" : (($action === 'complete') ? "Completed: {$step_name}" : "Error at: {$step_name}"),
                'metadata' => array(
                    'step' => $step_name,
                    'step_number' => $current_step_num,
                    'total_steps' => $total_steps,
                    'progress' => round(($current_step_num / $total_steps) * 100),
                    'action' => $action
                )
            );

            // Send to RIU API
            $ch = curl_init(RIU_CONTROL_PLANE_URL . '/api/v1/operations/report');
            curl_setopt_array($ch, array(
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($data),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
                ),
                CURLOPT_TIMEOUT => 5
            ));
            curl_exec($ch);
            curl_close($ch);

            // Also update instance status
            $statusData = array(
                'instance_id' => $_SESSION['riu_instance_id'],
                'status' => ($action === 'error') ? 'failed' : 'installing',
                'current_version' => RIU_VERSION,
                'metadata' => array(
                    'current_step' => $step_name,
                    'step_number' => $current_step_num,
                    'total_steps' => $total_steps,
                    'progress' => round(($current_step_num / $total_steps) * 100)
                )
            );

            $ch = curl_init(RIU_CONTROL_PLANE_URL . '/api/v1/instances/heartbeat');
            curl_setopt_array($ch, array(
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($statusData),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
                ),
                CURLOPT_TIMEOUT => 5
            ));
            curl_exec($ch);
            curl_close($ch);

        } catch (Exception $e) {
            // Silent fail - don't interrupt installation
            error_log('RIU progress reporting failed: ' . $e->getMessage());
        }
    }

    // Also try legacy method if database is available
    global $riu_helper_loaded;
    if ($riu_helper_loaded) {
        if ($action === 'start') {
            if (function_exists('monitor_system_state')) {
                monitor_system_state('installing', RIU_VERSION, array(
                    'current_step' => $step_name,
                    'step_number' => $current_step_num,
                    'total_steps' => $total_steps
                ));
            }
        } elseif ($action === 'complete') {
            if (function_exists('log_system_event')) {
                log_system_event('install', 'running', "Completed: {$step_name}", array(
                    'step' => $step_name,
                    'step_number' => $current_step_num
                ));
            }
        } elseif ($action === 'error') {
            if (function_exists('log_system_event')) {
                log_system_event('install', 'failed', "Error at: {$step_name}", array(
                    'step' => $step_name,
                    'step_number' => $current_step_num
                ));
            }
        }
    }
}

// File download progress reporting (for remote install)
function report_download_progress($current, $total, $filename = '') {
    $percentage = $total > 0 ? round(($current / $total) * 100) : 0;

    log_system_event(
        'install',
        'running',
        "Downloading files: {$current}/{$total}",
        array(
            'action' => 'downloading',
            'current_file' => $filename,
            'files_downloaded' => $current,
            'total_files' => $total,
            'progress' => $percentage
        )
    );
}

// Database setup progress reporting
function report_database_progress($message, $tables_created = 0, $total_tables = 16) {
    $percentage = $total_tables > 0 ? round(($tables_created / $total_tables) * 100) : 0;

    log_system_event(
        'install',
        'running',
        $message,
        array(
            'action' => 'database_setup',
            'tables_created' => $tables_created,
            'total_tables' => $total_tables,
            'progress' => $percentage
        )
    );
}

// License verification progress
function report_license_check($status, $message = '') {
    log_system_event(
        'install',
        $status,
        "License verification: " . $message,
        array(
            'action' => 'license_check',
            'status' => $status
        )
    );
}

// Installation completion handler
function finalize_installation() {
    // Report to RIU directly first
    if (isset($_SESSION['riu_instance_id'])) {
        try {
            // Report final installation completion
            $data = array(
                'instance_id' => $_SESSION['riu_instance_id'],
                'type' => 'install',
                'status' => 'completed',
                'message' => 'Installation completed successfully',
                'metadata' => array(
                    'completed_at' => date('Y-m-d H:i:s'),
                    'version' => RIU_VERSION,
                    'step' => 'Installation Complete',
                    'step_number' => 7,
                    'total_steps' => 7,
                    'progress' => 100
                )
            );

            $ch = curl_init(RIU_CONTROL_PLANE_URL . '/api/v1/operations/report');
            curl_setopt_array($ch, array(
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($data),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => array('Content-Type: application/json'),
                CURLOPT_TIMEOUT => 5
            ));
            curl_exec($ch);
            curl_close($ch);

            // Update instance to active status
            $statusData = array(
                'instance_id' => $_SESSION['riu_instance_id'],
                'status' => 'active',
                'current_version' => RIU_VERSION,
                'metadata' => array(
                    'installed_at' => date('Y-m-d H:i:s'),
                    'completed' => true
                )
            );

            $ch = curl_init(RIU_CONTROL_PLANE_URL . '/api/v1/instances/heartbeat');
            curl_setopt_array($ch, array(
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => json_encode($statusData),
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => array('Content-Type: application/json'),
                CURLOPT_TIMEOUT => 5
            ));
            curl_exec($ch);
            curl_close($ch);

        } catch (Exception $e) {
            error_log('RIU finalization failed: ' . $e->getMessage());
        }
    }

    // Also try legacy method if database is available
    global $pdo;
    if (isset($pdo)) {
        try {
            // Get database info
            $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'license_type' LIMIT 1");
            $stmt->execute();
            $license_type = $stmt->fetchColumn();

            $stmt = $pdo->prepare("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers");
            $stmt->execute();
            $worker_count = $stmt->fetchColumn();

            // Send final status via helper if available
            if (function_exists('log_system_event')) {
                log_system_event(
                    'install',
                    'completed',
                    'Installation completed successfully',
                    array(
                        'completed_at' => date('Y-m-d H:i:s'),
                        'license_type' => $license_type,
                        'initial_workers' => $worker_count,
                        'version' => RIU_VERSION
                    )
                );
            }

            // Store last heartbeat time
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value, setting_type)
                                   VALUES ('last_heartbeat', NOW(), 'datetime')
                                   ON DUPLICATE KEY UPDATE setting_value = NOW()");
            $stmt->execute();

        } catch (Exception $e) {
            // Silent fail - installation was successful, just reporting failed
        }
    }
}

// Installation error handler
function handle_install_error($error_message, $step = null) {
    if ($step !== null) {
        track_install_progress($step, 'error');
    }

    fail_installation($error_message);
}
?>
