<?php
/**
 * SCMS Installation with RIU Stage Reporting
 * This example shows how to report installation progress to RIU Control Plane
 */

require_once __DIR__ . '/../includes/RIUClient.php';
require_once __DIR__ . '/../includes/riu_config.php';

class SCMSInstallerWithReporting
{
    private $riuClient;
    private $installMode; // 'local' or 'remote'
    private $version;

    public function __construct($licenseKey, $domain, $installMode = 'remote', $version = '1.0.0')
    {
        $this->riuClient = new RIUClient(
            RIU_CONTROL_PLANE_URL,
            RIU_PRODUCT_ID,
            $licenseKey,
            $domain
        );
        $this->installMode = $installMode;
        $this->version = $version;
    }

    /**
     * Complete installation flow with RIU reporting
     */
    public function install()
    {
        try {
            // STAGE 1: License Verification
            echo "Step 1: Verifying license...\n";
            $this->reportStage('installing', '0.0.0', ['step' => 'license_verification']);

            $licenseResult = $this->riuClient->verifyLicense();
            if (!$licenseResult['success']) {
                $this->reportStage('failed', '0.0.0', [
                    'step' => 'license_verification',
                    'error' => $licenseResult['message'] ?? 'License verification failed'
                ]);
                throw new Exception('License verification failed: ' . ($licenseResult['message'] ?? 'Unknown error'));
            }
            echo "✓ License verified\n";

            // Report installation start
            $this->riuClient->reportProgress('install', 'running', 'Installation started', [
                'mode' => $this->installMode,
                'target_version' => $this->version
            ]);

            // STAGE 2: Download/Prepare Files
            if ($this->installMode === 'remote') {
                echo "\nStep 2: Downloading files from RIU...\n";
                $this->reportStage('installing', $this->version, ['step' => 'downloading_files']);

                $result = $this->downloadFromRIU();
                if (!$result['success']) {
                    $this->reportStage('failed', '0.0.0', [
                        'step' => 'downloading_files',
                        'error' => $result['error']
                    ]);
                    throw new Exception('Download failed: ' . $result['error']);
                }
                echo "✓ Files downloaded successfully\n";
            } else {
                echo "\nStep 2: Preparing local files...\n";
                $this->reportStage('installing', $this->version, ['step' => 'preparing_local_files']);

                $result = $this->prepareLocalFiles();
                if (!$result['success']) {
                    $this->reportStage('failed', '0.0.0', [
                        'step' => 'preparing_local_files',
                        'error' => $result['error']
                    ]);
                    throw new Exception('Local preparation failed: ' . $result['error']);
                }
                echo "✓ Local files prepared\n";
            }

            // STAGE 3: Database Setup
            echo "\nStep 3: Setting up database...\n";
            $this->reportStage('installing', $this->version, ['step' => 'database_setup']);

            $dbResult = $this->setupDatabase();
            if (!$dbResult['success']) {
                $this->reportStage('failed', '0.0.0', [
                    'step' => 'database_setup',
                    'error' => $dbResult['error']
                ]);
                throw new Exception('Database setup failed: ' . $dbResult['error']);
            }
            echo "✓ Database configured\n";

            // STAGE 4: File Installation
            echo "\nStep 4: Installing files...\n";
            $this->reportStage('installing', $this->version, ['step' => 'file_installation']);

            $fileResult = $this->installFiles();
            if (!$fileResult['success']) {
                $this->reportStage('failed', '0.0.0', [
                    'step' => 'file_installation',
                    'error' => $fileResult['error']
                ]);
                throw new Exception('File installation failed: ' . $fileResult['error']);
            }
            echo "✓ Files installed\n";

            // STAGE 5: Configuration
            echo "\nStep 5: Writing configuration...\n";
            $this->reportStage('installing', $this->version, ['step' => 'configuration']);

            $configResult = $this->writeConfiguration();
            if (!$configResult['success']) {
                $this->reportStage('failed', '0.0.0', [
                    'step' => 'configuration',
                    'error' => $configResult['error']
                ]);
                throw new Exception('Configuration failed: ' . $configResult['error']);
            }
            echo "✓ Configuration saved\n";

            // STAGE 6: Post-installation
            echo "\nStep 6: Running post-installation tasks...\n";
            $this->reportStage('installing', $this->version, ['step' => 'post_installation']);

            $postResult = $this->postInstallation();
            if (!$postResult['success']) {
                $this->reportStage('failed', $this->version, [
                    'step' => 'post_installation',
                    'error' => $postResult['error']
                ]);
                throw new Exception('Post-installation failed: ' . $postResult['error']);
            }
            echo "✓ Post-installation complete\n";

            // STAGE 7: Mark as Active
            echo "\nStep 7: Finalizing installation...\n";
            $this->reportStage('active', $this->version, [
                'step' => 'completed',
                'install_mode' => $this->installMode,
                'installed_at' => date('Y-m-d H:i:s')
            ]);

            // Report installation completion
            $this->riuClient->reportProgress('install', 'completed', 'Installation completed successfully', [
                'version' => $this->version,
                'mode' => $this->installMode,
                'duration' => time() - $_SERVER['REQUEST_TIME']
            ]);

            echo "\n✅ Installation completed successfully!\n";
            return ['success' => true];

        } catch (Exception $e) {
            echo "\n❌ Installation failed: " . $e->getMessage() . "\n";

            // Report failure
            $this->riuClient->reportProgress('install', 'failed', $e->getMessage(), [
                'version' => $this->version,
                'mode' => $this->installMode
            ]);

            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Download files from RIU Control Plane
     */
    private function downloadFromRIU()
    {
        // Get manifest
        $manifestResult = $this->riuClient->getManifest($this->version);
        if (!$manifestResult['success']) {
            return ['success' => false, 'error' => 'Failed to get manifest'];
        }

        $manifest = $manifestResult['manifest'];
        $totalFiles = count($manifest['files']);
        $downloadedFiles = 0;

        echo "Downloading {$totalFiles} files...\n";

        foreach ($manifest['files'] as $file) {
            $destinationPath = __DIR__ . '/../' . $file['path'];
            $destinationDir = dirname($destinationPath);

            if (!is_dir($destinationDir)) {
                mkdir($destinationDir, 0755, true);
            }

            $result = $this->riuClient->downloadFile($this->version, $file['path'], $destinationPath);

            if (!$result['success']) {
                return ['success' => false, 'error' => 'Failed to download ' . $file['path']];
            }

            $downloadedFiles++;

            // Report progress every 10 files
            if ($downloadedFiles % 10 === 0) {
                $this->riuClient->reportProgress('install', 'running', "Downloaded {$downloadedFiles}/{$totalFiles} files", [
                    'progress' => round(($downloadedFiles / $totalFiles) * 100, 2)
                ]);
            }
        }

        return ['success' => true];
    }

    /**
     * Prepare local files (when installation files are already on server)
     */
    private function prepareLocalFiles()
    {
        // In local mode, files are already present
        // Just verify they exist and report to RIU
        $sourceDir = __DIR__ . '/../';

        if (!is_dir($sourceDir)) {
            return ['success' => false, 'error' => 'Source directory not found'];
        }

        // Still need to register with RIU even in local mode
        return ['success' => true];
    }

    /**
     * Setup database
     */
    private function setupDatabase()
    {
        // Your database setup logic here
        // Example:
        // - Create database
        // - Import schema
        // - Create admin user

        sleep(1); // Simulate work
        return ['success' => true];
    }

    /**
     * Install/copy files to target location
     */
    private function installFiles()
    {
        // Your file installation logic here
        // Example:
        // - Copy files to target directories
        // - Set permissions

        sleep(1); // Simulate work
        return ['success' => true];
    }

    /**
     * Write configuration file
     */
    private function writeConfiguration()
    {
        // Your configuration logic here
        // Example:
        // - Write config.php with database credentials
        // - Save RIU settings

        $config = [
            'version' => $this->version,
            'installed_at' => date('Y-m-d H:i:s'),
            'riu_instance_id' => $this->riuClient->getInstanceId(),
        ];

        // Save to file
        file_put_contents(
            __DIR__ . '/../config/version.json',
            json_encode($config, JSON_PRETTY_PRINT)
        );

        return ['success' => true];
    }

    /**
     * Post-installation tasks
     */
    private function postInstallation()
    {
        // Your post-installation logic here
        // Example:
        // - Clear caches
        // - Generate keys
        // - Send welcome email

        sleep(1); // Simulate work
        return ['success' => true];
    }

    /**
     * Helper to report stage with error handling
     */
    private function reportStage($stage, $version, $metadata = [])
    {
        try {
            $this->riuClient->reportStage($stage, $version, $metadata);
        } catch (Exception $e) {
            // Don't fail installation if reporting fails
            error_log('RIU reporting failed: ' . $e->getMessage());
        }
    }
}

// Example usage
if (php_sapi_name() === 'cli') {
    // CLI usage example
    $licenseKey = $argv[1] ?? 'demo-license-key';
    $domain = $argv[2] ?? 'example.com';
    $mode = $argv[3] ?? 'remote'; // 'local' or 'remote'

    $installer = new SCMSInstallerWithReporting($licenseKey, $domain, $mode);
    $result = $installer->install();

    exit($result['success'] ? 0 : 1);
}
