<?php
/**
 * Step 3: File Operations
 * Downloads files from RIU (remote) or verifies local files
 * Shows progress bar during operation
 */

// Check install mode
$install_mode = $_SESSION['install_mode'] ?? '';

if (empty($install_mode)) {
    header('Location: install.php?step=2');
    exit;
}

// RIU Configuration for remote mode
$riu_server = 'https://supix.co.uk/sys/riu';
$product_id = 'scms_v3';
$version = '3.0.0';

// Define expected files
$required_files = array(
    'includes/core.php',
    'includes/helpers.php',
    'includes/database.php',
    'includes/system_health.php',
    'includes/session_manager.php',
    'includes/resource_manager.php',
    'config/database.sample.php',
    'admin/login.php',
    'admin/dashboard.php',
    'admin/header.php',
    'admin/footer.php',
    'admin/sidebar.php'
);

$total_files = count($required_files);
?>

<h3 class="mb-4">
    <i class="bi bi-<?php echo $install_mode === 'remote' ? 'cloud-download-fill' : 'hdd-fill'; ?>"></i>
    <?php echo $install_mode === 'remote' ? 'Downloading Files from RIU' : 'Verifying Local Files'; ?>
</h3>

<div class="alert alert-info">
    <i class="bi bi-info-circle-fill"></i>
    <?php if ($install_mode === 'remote'): ?>
        <strong>Remote Installation:</strong> Downloading files from RIU server at <?php echo $riu_server; ?>
    <?php else: ?>
        <strong>Local Installation:</strong> Verifying that all required files are present on your server.
    <?php endif; ?>
</div>

<!-- Progress Container -->
<div class="card">
    <div class="card-body">
        <!-- Overall Progress Bar -->
        <div class="mb-4">
            <div class="d-flex justify-content-between mb-2">
                <span><strong>Overall Progress</strong></span>
                <span id="overall_percent">0%</span>
            </div>
            <div class="progress" style="height: 30px;">
                <div class="progress-bar progress-bar-striped progress-bar-animated bg-<?php echo $install_mode === 'remote' ? 'success' : 'info'; ?>"
                     role="progressbar"
                     id="overall_progress"
                     style="width: 0%">
                    0%
                </div>
            </div>
        </div>

        <!-- Current File -->
        <div class="mb-4">
            <div class="d-flex justify-content-between mb-2">
                <span><strong>Current Task:</strong></span>
                <span id="current_status" class="badge bg-secondary">Initializing</span>
            </div>
            <p id="current_file" class="text-primary mb-0 font-monospace">
                <i class="bi bi-hourglass-split"></i> Starting operation...
            </p>
        </div>

        <!-- Statistics -->
        <div class="row mb-4 text-center">
            <div class="col-md-3">
                <div class="card bg-primary text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="total_files"><?php echo $total_files; ?></h4>
                        <small>Total Files</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-success text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="processed_files">0</h4>
                        <small><?php echo $install_mode === 'remote' ? 'Downloaded' : 'Verified'; ?></small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-info text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="size_mb">0.0 MB</h4>
                        <small>Size</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card bg-warning text-white">
                    <div class="card-body py-2">
                        <h4 class="mb-0" id="eta">--:--</h4>
                        <small>Time Remaining</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Details Log (Collapsible) -->
        <div class="card bg-light">
            <div class="card-header">
                <strong>Operation Details</strong>
                <button type="button" class="btn btn-sm btn-outline-secondary float-end" onclick="toggleLog()">
                    <i class="bi bi-eye" id="log_icon"></i> <span id="log_toggle_text">Show Details</span>
                </button>
            </div>
            <div class="card-body p-0" id="log_container" style="display: none;">
                <div id="operation_log" style="max-height: 300px; overflow-y: auto; font-family: 'Courier New', monospace; font-size: 12px; padding: 15px; background: #f8f9fa;">
                    <div class="text-muted">[00:00:00] Starting operation...</div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success Message -->
<div id="operation_complete" class="alert alert-success mt-4" style="display: none;">
    <i class="bi bi-check-circle-fill"></i> <strong>Operation Complete!</strong><br>
    All files processed successfully.
</div>

<!-- Continue Button -->
<div id="continue_button" class="d-grid gap-2 mt-4" style="display: none;">
    <button class="btn btn-primary btn-lg" onclick="proceedToNextStep()">
        <i class="bi bi-arrow-right-circle-fill"></i> Continue to Database Configuration
    </button>
</div>

<!-- Error Message -->
<div id="operation_error" class="alert alert-danger mt-4" style="display: none;">
    <i class="bi bi-x-circle-fill"></i> <strong>Operation Failed!</strong><br>
    <span id="error_message"></span>
    <div class="mt-3">
        <button class="btn btn-outline-danger" onclick="window.location.href='install.php?step=2'">
            <i class="bi bi-arrow-left"></i> Go Back
        </button>
        <button class="btn btn-outline-primary" onclick="location.reload()">
            <i class="bi bi-arrow-clockwise"></i> Retry
        </button>
    </div>
</div>

<script>
const installMode = '<?php echo $install_mode; ?>';
const riuServer = '<?php echo $riu_server; ?>';
const productId = '<?php echo $product_id; ?>';
const version = '<?php echo $version; ?>';
const licenseKey = '<?php echo $_SESSION['license_key'] ?? ''; ?>';
const domain = '<?php echo $_SESSION['domain'] ?? ''; ?>';

let startTime = Date.now();
let processedFiles = 0;
let totalFiles = <?php echo $total_files; ?>;
let totalSize = 0;

function toggleLog() {
    const logContainer = document.getElementById('log_container');
    const toggleText = document.getElementById('log_toggle_text');
    const icon = document.getElementById('log_icon');

    if (logContainer.style.display === 'none') {
        logContainer.style.display = 'block';
        toggleText.textContent = 'Hide Details';
        icon.className = 'bi bi-eye-slash';
    } else {
        logContainer.style.display = 'none';
        toggleText.textContent = 'Show Details';
        icon.className = 'bi bi-eye';
    }
}

function addLog(message, type = 'info') {
    const timestamp = new Date().toLocaleTimeString();
    const logDiv = document.getElementById('operation_log');
    const icons = {
        'info': '🔍',
        'success': '✓',
        'error': '✗',
        'warning': '⚠'
    };
    const colors = {
        'info': 'text-muted',
        'success': 'text-success',
        'error': 'text-danger',
        'warning': 'text-warning'
    };

    const logEntry = `<div class="${colors[type]}">[${timestamp}] ${icons[type]} ${message}</div>`;
    logDiv.innerHTML += logEntry;
    logDiv.scrollTop = logDiv.scrollHeight;
}

function updateProgress(percent, file, status = '') {
    document.getElementById('overall_progress').style.width = percent + '%';
    document.getElementById('overall_progress').textContent = Math.round(percent) + '%';
    document.getElementById('overall_percent').textContent = Math.round(percent) + '%';
    document.getElementById('current_file').innerHTML = '<i class="bi bi-file-earmark-text"></i> ' + file;

    if (status) {
        const statusBadge = document.getElementById('current_status');
        statusBadge.textContent = status;
        statusBadge.className = 'badge bg-' + (status === 'Running' ? 'primary' : 'success');
    }
}

function updateStats(processed, size) {
    document.getElementById('processed_files').textContent = processed;
    document.getElementById('size_mb').textContent = (size / 1024 / 1024).toFixed(2) + ' MB';

    // Calculate ETA
    const elapsed = (Date.now() - startTime) / 1000;
    const rate = processed / elapsed;
    const remaining = totalFiles - processed;
    const eta = remaining / rate;

    if (isFinite(eta) && eta > 0) {
        const minutes = Math.floor(eta / 60);
        const seconds = Math.floor(eta % 60);
        document.getElementById('eta').textContent = minutes + ':' + (seconds < 10 ? '0' : '') + seconds;
    }
}

async function startOperation() {
    addLog(`Starting ${installMode} installation mode`, 'info');

    if (installMode === 'remote') {
        await performRemoteDownload();
    } else {
        await performLocalVerification();
    }
}

async function performRemoteDownload() {
    try {
        addLog(`Connecting to RIU server: ${riuServer}`, 'info');
        updateProgress(5, 'Connecting to RIU server...', 'Running');

        // Step 1: Fetch manifest from RIU
        addLog('Fetching file manifest from RIU...', 'info');

        let files = [];
        try {
            const manifestResponse = await fetch(`${riuServer}/api/manifest.php?product=${productId}&version=${version}&license=${licenseKey}&domain=${domain}`);
            const manifestData = await manifestResponse.json();

            if (manifestData.success && manifestData.files) {
                files = manifestData.files;
                totalFiles = files.length;
                document.getElementById('total_files').textContent = totalFiles;
                addLog(`Manifest loaded: ${files.length} files to download`, 'success');
            } else {
                throw new Error(manifestData.message || 'Failed to fetch manifest from RIU');
            }
        } catch (manifestError) {
            addLog('Failed to fetch RIU manifest, using default file list', 'warning');
            files = <?php echo json_encode($required_files); ?>;
        }

        updateProgress(10, 'Starting downloads...', 'Running');

        // Step 2: Download files
        for (let i = 0; i < files.length; i++) {
            const file = files[i];
            const progress = 10 + ((i + 1) / files.length) * 90;

            updateProgress(progress, file, 'Running');
            addLog(`Downloading: ${file}`, 'info');

            // Download file from RIU
            try {
                const downloadResponse = await fetch(`${riuServer}/api/download.php?product=${productId}&version=${version}&file=${encodeURIComponent(file)}&license=${licenseKey}`);

                if (!downloadResponse.ok) {
                    throw new Error(`Failed to download ${file}`);
                }

                const fileContent = await downloadResponse.text();
                const fileSize = new Blob([fileContent]).size;

                // Save file via backend
                const saveResponse = await fetch('install_save_file.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                    body: `file=${encodeURIComponent(file)}&content=${encodeURIComponent(fileContent)}`
                });

                const saveResult = await saveResponse.json();

                if (saveResult.success) {
                    addLog(`✓ Downloaded: ${file} (${(fileSize / 1024).toFixed(1)} KB)`, 'success');
                    totalSize += fileSize;
                } else {
                    addLog(`✗ Save failed: ${file}`, 'error');
                }

            } catch (downloadError) {
                addLog(`⚠ Download error: ${file} - ${downloadError.message}`, 'warning');
            }

            processedFiles++;
            updateStats(processedFiles, totalSize);

            await new Promise(resolve => setTimeout(resolve, 100));
        }

        // Stop animation
        document.getElementById('overall_progress').classList.remove('progress-bar-animated');

        addLog('All files downloaded successfully!', 'success');
        updateProgress(100, 'Download complete!', 'Complete');

        document.getElementById('operation_complete').style.display = 'block';
        document.getElementById('continue_button').style.display = 'block';

    } catch (error) {
        addLog('ERROR: ' + error.message, 'error');
        document.getElementById('overall_progress').classList.remove('progress-bar-animated');
        document.getElementById('operation_error').style.display = 'block';
        document.getElementById('error_message').textContent = error.message;
    }
}

async function performLocalVerification() {
    try {
        addLog('Starting local file verification...', 'info');

        // First, fetch manifest from RIU
        addLog('Fetching file manifest from RIU...', 'info');
        updateProgress(5, 'Fetching manifest from RIU...', 'Running');

        let files = [];
        let hasErrors = false;
        let errorFiles = [];

        try {
            const manifestResponse = await fetch(`${riuServer}/api/manifest.php?product=${productId}&version=${version}`);
            const manifestData = await manifestResponse.json();

            if (manifestData.success && manifestData.files) {
                files = manifestData.files;
                totalFiles = files.length;
                document.getElementById('total_files').textContent = totalFiles;
                addLog(`Manifest loaded: ${files.length} files to verify`, 'success');
            } else {
                throw new Error('Failed to fetch manifest from RIU');
            }
        } catch (manifestError) {
            addLog('Failed to fetch RIU manifest, using default file list', 'warning');
            files = <?php echo json_encode($required_files); ?>;
        }

        updateProgress(10, 'Starting file verification...', 'Running');

        for (let i = 0; i < files.length; i++) {
            const file = files[i];
            const progress = 10 + ((i + 1) / files.length) * 90;

            updateProgress(progress, file, 'Running');

            // Verify file exists
            const response = await fetch(`install_verify.php?file=${encodeURIComponent(file)}`);
            const result = await response.json();

            if (result.exists) {
                addLog(`✓ Found: ${file} (${(result.size / 1024).toFixed(1)} KB)`, 'success');
                totalSize += result.size || 0;
            } else {
                addLog(`✗ MISSING: ${file}`, 'error');
                hasErrors = true;
                errorFiles.push(file);
            }

            processedFiles++;
            updateStats(processedFiles, totalSize);

            await new Promise(resolve => setTimeout(resolve, 50));
        }

        // Stop animation at 100%
        document.getElementById('overall_progress').classList.remove('progress-bar-animated');
        updateProgress(100, 'Verification complete', 'Complete');

        if (hasErrors) {
            addLog(`Verification failed! ${errorFiles.length} files are missing.`, 'error');

            document.getElementById('operation_error').style.display = 'block';
            document.getElementById('error_message').innerHTML =
                `<strong>${errorFiles.length} required files are missing:</strong><br><ul class="mt-2 mb-0">` +
                errorFiles.map(f => `<li>${f}</li>`).join('') +
                '</ul><br><strong>Please upload all required files via FTP or use Remote Installation instead.</strong>';
        } else {
            addLog('All files verified successfully!', 'success');

            document.getElementById('operation_complete').style.display = 'block';
            document.getElementById('continue_button').style.display = 'block';
        }

    } catch (error) {
        addLog('ERROR: ' + error.message, 'error');
        document.getElementById('overall_progress').classList.remove('progress-bar-animated');
        document.getElementById('operation_error').style.display = 'block';
        document.getElementById('error_message').textContent = error.message;
    }
}

// Proceed to next step (update session and redirect)
async function proceedToNextStep() {
    try {
        // Update session to allow Step 4 access
        const response = await fetch('install.php?action=complete_files', {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'}
        });

        const result = await response.json();

        if (result.success) {
            window.location.href = 'install.php?step=4';
        } else {
            alert('Failed to proceed. Please try again.');
        }
    } catch (error) {
        console.error('Error:', error);
        // Fallback: try to proceed anyway
        window.location.href = 'install.php?step=4';
    }
}

// Auto-start operation when page loads
window.addEventListener('load', () => {
    setTimeout(startOperation, 500);
});
</script>

<style>
.font-monospace {
    font-family: 'Courier New', monospace;
}
</style>
