<?php
/**
 * SCMS Update Checker
 * Checks for pending updates from RIU Control Plane and applies them
 * This should be run via cron job (e.g., every hour or daily)
 */

require_once __DIR__ . '/RIUClient.php';
require_once __DIR__ . '/riu_config.php';

class SCMSUpdateChecker
{
    private $riuClient;
    private $currentVersion;
    private $configPath;

    public function __construct()
    {
        // Load current installation config
        $this->configPath = __DIR__ . '/../config/version.json';
        $this->loadCurrentVersion();

        // Initialize RIU Client from saved credentials
        $config = $this->loadConfig();

        $this->riuClient = new RIUClient(
            RIU_CONTROL_PLANE_URL,
            RIU_PRODUCT_ID,
            $config['license_key'] ?? '',
            $config['domain'] ?? $_SERVER['HTTP_HOST']
        );

        // Restore access token and instance ID from session/config
        if (isset($config['riu_access_token'])) {
            $_SESSION['riu_access_token'] = $config['riu_access_token'];
        }
        if (isset($config['riu_instance_id'])) {
            $_SESSION['riu_instance_id'] = $config['riu_instance_id'];
        }
    }

    /**
     * Check and apply pending updates
     */
    public function checkAndUpdate()
    {
        echo "[" . date('Y-m-d H:i:s') . "] Checking for pending updates...\n";

        // Send heartbeat to show we're alive
        $this->riuClient->heartbeat([
            'current_version' => $this->currentVersion,
            'check_type' => 'automated'
        ]);

        // Check for pending operations
        $response = $this->riuClient->checkPendingOperations();

        if (!$response['success']) {
            echo "Error checking for updates: " . ($response['error'] ?? 'Unknown error') . "\n";
            return ['success' => false];
        }

        $operations = $response['operations'] ?? [];

        if (empty($operations)) {
            echo "No pending operations\n";
            return ['success' => true, 'message' => 'No updates available'];
        }

        echo "Found " . count($operations) . " pending operation(s)\n";

        foreach ($operations as $operation) {
            $this->processOperation($operation);
        }

        return ['success' => true];
    }

    /**
     * Process a single operation
     */
    private function processOperation($operation)
    {
        $operationId = $operation['id'];
        $type = $operation['type'];
        $targetVersion = $operation['release']['version'] ?? null;

        echo "\nProcessing operation #{$operationId}: {$type}\n";

        try {
            switch ($type) {
                case 'update':
                    $this->performUpdate($operation);
                    break;

                case 'maintenance':
                    $this->enterMaintenanceMode($operation);
                    break;

                case 'rollback':
                    $this->performRollback($operation);
                    break;

                default:
                    throw new Exception("Unknown operation type: {$type}");
            }

            // Mark operation as completed
            $this->riuClient->completeOperation($operationId, 'completed');
            echo "✓ Operation #{$operationId} completed successfully\n";

        } catch (Exception $e) {
            echo "✗ Operation #{$operationId} failed: " . $e->getMessage() . "\n";

            // Mark operation as failed
            $this->riuClient->completeOperation($operationId, 'failed', $e->getMessage());
        }
    }

    /**
     * Perform update to new version
     */
    private function performUpdate($operation)
    {
        $targetVersion = $operation['release']['version'];
        $fromVersion = $this->currentVersion;

        echo "Updating from v{$fromVersion} to v{$targetVersion}\n";

        // Report update started
        $this->riuClient->reportStage('installing', $targetVersion, [
            'update_type' => 'automatic',
            'from_version' => $fromVersion
        ]);

        $this->riuClient->reportProgress('update', 'running', "Updating to v{$targetVersion}", [
            'from_version' => $fromVersion,
            'to_version' => $targetVersion
        ]);

        // Step 1: Download manifest
        echo "  - Downloading manifest...\n";
        $manifestResult = $this->riuClient->getManifest($targetVersion);

        if (!$manifestResult['success']) {
            throw new Exception('Failed to download manifest');
        }

        $manifest = $manifestResult['manifest'];

        // Step 2: Backup current version
        echo "  - Creating backup...\n";
        $this->createBackup($fromVersion);

        // Step 3: Download new files
        echo "  - Downloading " . count($manifest['files']) . " files...\n";
        $downloadResult = $this->downloadFiles($targetVersion, $manifest['files']);

        if (!$downloadResult['success']) {
            throw new Exception('Failed to download files: ' . $downloadResult['error']);
        }

        // Step 4: Run database migrations if any
        echo "  - Running migrations...\n";
        $this->runMigrations($targetVersion);

        // Step 5: Update version file
        echo "  - Updating version...\n";
        $this->updateVersion($targetVersion);

        // Step 6: Clear caches
        echo "  - Clearing caches...\n";
        $this->clearCaches();

        // Report update completed
        $this->riuClient->reportStage('active', $targetVersion, [
            'updated_from' => $fromVersion,
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        $this->riuClient->reportProgress('update', 'completed', "Updated successfully to v{$targetVersion}", [
            'from_version' => $fromVersion,
            'to_version' => $targetVersion
        ]);

        echo "✓ Update completed successfully\n";
    }

    /**
     * Download update files
     */
    private function downloadFiles($version, $files)
    {
        $total = count($files);
        $downloaded = 0;

        foreach ($files as $file) {
            $destinationPath = __DIR__ . '/../' . $file['path'];
            $destinationDir = dirname($destinationPath);

            if (!is_dir($destinationDir)) {
                mkdir($destinationDir, 0755, true);
            }

            $result = $this->riuClient->downloadFile($version, $file['path'], $destinationPath);

            if (!$result['success']) {
                return ['success' => false, 'error' => 'Failed to download ' . $file['path']];
            }

            $downloaded++;

            // Report progress
            if ($downloaded % 10 === 0 || $downloaded === $total) {
                $this->riuClient->reportProgress('update', 'running', "Downloaded {$downloaded}/{$total} files", [
                    'progress' => round(($downloaded / $total) * 100, 2)
                ]);
            }
        }

        return ['success' => true];
    }

    /**
     * Create backup of current version
     */
    private function createBackup($version)
    {
        $backupDir = __DIR__ . '/../backups/v' . $version . '_' . date('YmdHis');

        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }

        // Copy critical files/directories
        // In production, you'd want to backup everything
        $itemsToBackup = ['includes', 'config', 'install'];

        foreach ($itemsToBackup as $item) {
            $source = __DIR__ . '/../' . $item;
            if (file_exists($source)) {
                $this->recursiveCopy($source, $backupDir . '/' . $item);
            }
        }

        return $backupDir;
    }

    /**
     * Run database migrations
     */
    private function runMigrations($version)
    {
        // Check if migration file exists for this version
        $migrationFile = __DIR__ . '/../migrations/migrate_to_' . str_replace('.', '_', $version) . '.php';

        if (file_exists($migrationFile)) {
            require_once $migrationFile;
            // Run migration
            echo "    Running migration for v{$version}\n";
        }
    }

    /**
     * Update version in config
     */
    private function updateVersion($newVersion)
    {
        $config = $this->loadConfig();
        $config['version'] = $newVersion;
        $config['updated_at'] = date('Y-m-d H:i:s');

        file_put_contents(
            $this->configPath,
            json_encode($config, JSON_PRETTY_PRINT)
        );

        $this->currentVersion = $newVersion;
    }

    /**
     * Clear application caches
     */
    private function clearCaches()
    {
        // Clear your application's caches here
        // Example:
        $cacheDir = __DIR__ . '/../cache';
        if (is_dir($cacheDir)) {
            $this->deleteDirectory($cacheDir);
            mkdir($cacheDir, 0755);
        }
    }

    /**
     * Enter maintenance mode
     */
    private function enterMaintenanceMode($operation)
    {
        echo "Entering maintenance mode\n";

        // Create maintenance flag file
        file_put_contents(__DIR__ . '/../maintenance.flag', date('Y-m-d H:i:s'));

        // Report to RIU
        $this->riuClient->reportStage('maintenance', $this->currentVersion, [
            'reason' => $operation['metadata']['reason'] ?? 'Scheduled maintenance'
        ]);
    }

    /**
     * Perform rollback to previous version
     */
    private function performRollback($operation)
    {
        $targetVersion = $operation['metadata']['target_version'] ?? null;

        if (!$targetVersion) {
            throw new Exception('No target version specified for rollback');
        }

        echo "Rolling back to v{$targetVersion}\n";

        // Similar to update, but restoring from backup
        // Implementation depends on your backup strategy
    }

    /**
     * Load current version from config
     */
    private function loadCurrentVersion()
    {
        if (!file_exists($this->configPath)) {
            $this->currentVersion = '0.0.0';
            return;
        }

        $config = json_decode(file_get_contents($this->configPath), true);
        $this->currentVersion = $config['version'] ?? '0.0.0';
    }

    /**
     * Load full configuration
     */
    private function loadConfig()
    {
        if (!file_exists($this->configPath)) {
            return [];
        }

        return json_decode(file_get_contents($this->configPath), true) ?? [];
    }

    /**
     * Recursive copy helper
     */
    private function recursiveCopy($source, $dest)
    {
        if (is_dir($source)) {
            if (!is_dir($dest)) {
                mkdir($dest, 0755, true);
            }

            $files = scandir($source);
            foreach ($files as $file) {
                if ($file !== '.' && $file !== '..') {
                    $this->recursiveCopy($source . '/' . $file, $dest . '/' . $file);
                }
            }
        } else {
            copy($source, $dest);
        }
    }

    /**
     * Delete directory recursively
     */
    private function deleteDirectory($dir)
    {
        if (!is_dir($dir)) {
            return;
        }

        $files = scandir($dir);
        foreach ($files as $file) {
            if ($file !== '.' && $file !== '..') {
                $path = $dir . '/' . $file;
                if (is_dir($path)) {
                    $this->deleteDirectory($path);
                } else {
                    unlink($path);
                }
            }
        }
        rmdir($dir);
    }
}

// CLI Usage
if (php_sapi_name() === 'cli') {
    session_start();
    $checker = new SCMSUpdateChecker();
    $result = $checker->checkAndUpdate();
    exit($result['success'] ? 0 : 1);
}
