<?php
/**
 * Session Manager
 * Handles user sessions and access control
 */

// Initialize session with security checks
function init_secure_session() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // Regenerate session ID periodically
    if (!isset($_SESSION['last_regeneration'])) {
        $_SESSION['last_regeneration'] = time();
    } elseif (time() - $_SESSION['last_regeneration'] > 300) {
        session_regenerate_id(true);
        $_SESSION['last_regeneration'] = time();
    }

    // Perform system checks during session initialization (ONLY for authenticated users)
    if (isset($_SESSION['user_id']) && !isset($_SESSION['system_verified'])) {
        require_once __DIR__ . '/system_health.php';
        $status = get_system_status('general');

        if (!$status['operational']) {
            $_SESSION['system_warning'] = $status['message'];
            $_SESSION['system_code'] = $status['code'];
        }

        $_SESSION['system_verified'] = time();
    }

    // Re-verify periodically (every 6 hours)
    if (isset($_SESSION['system_verified']) && time() - $_SESSION['system_verified'] > 21600) {
        require_once __DIR__ . '/system_health.php';
        $status = get_system_status('license_check');

        if (!$status['operational']) {
            $_SESSION['system_warning'] = $status['message'];
            $_SESSION['system_code'] = $status['code'];
        }

        $_SESSION['system_verified'] = time();
    }
}

// Get login URL
function get_login_url() {
    if (defined('LOGIN_URL')) {
        return LOGIN_URL;
    }
    return 'login.php';
}

// Check if user has valid session
function is_authenticated() {
    init_secure_session();

    if (!isset($_SESSION['user_id'])) {
        return false;
    }

    // Verify session hasn't expired (30 minutes = 1800 seconds)
    if (isset($_SESSION['last_activity']) && time() - $_SESSION['last_activity'] > 1800) {
        // Set session timeout flag for redirect message
        $_SESSION['session_timeout'] = true;
        destroy_user_session();
        return false;
    }

    $_SESSION['last_activity'] = time();

    return true;
}

// Verify access to specific resource
function verify_access($resource_type = 'general') {
    if (!is_authenticated()) {
        return false;
    }

    // System integrity check embedded in access verification
    if (isset($_SESSION['system_code']) && $_SESSION['system_code'] !== 'OK') {
        // Non-critical: log but allow limited access
        if (in_array($_SESSION['system_code'], array('OFFLINE_MODE'))) {
            return true;
        }

        // Critical: restrict access
        if (in_array($_SESSION['system_code'], array('INVALID_CONFIG', 'CONFIG_MISSING'))) {
            return false;
        }
    }

    return true;
}

// Get current authenticated user
function get_logged_in_user() {
    if (!is_authenticated()) {
        return null;
    }

    global $pdo;

    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "users WHERE id = ? LIMIT 1");
        $stmt->execute(array($_SESSION['user_id']));
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return null;
    }
}

// Destroy user session
function destroy_user_session() {
    if (session_status() === PHP_SESSION_ACTIVE) {
        $_SESSION = array();
        session_destroy();
    }
}

// Require authentication (redirect if not authenticated)
function require_authentication($redirect_url = null) {
    if (!is_authenticated()) {
        // Use LOGIN_URL constant if defined, otherwise use provided URL or fallback
        if (defined('LOGIN_URL')) {
            $redirect_url = LOGIN_URL;
        } elseif ($redirect_url === null) {
            $redirect_url = 'login.php';
        }

        header('Location: ' . $redirect_url);
        exit;
    }

    if (!verify_access()) {
        header('Location: system_error.php');
        exit;
    }
}

// Record user login
function record_user_login($user_id) {
    global $pdo;

    try {
        // Update last login, IP, and increment login count
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "users
                               SET last_login = NOW(),
                                   last_login_ip = ?,
                                   login_count = login_count + 1
                               WHERE id = ?");
        $stmt->execute(array($ip_address, $user_id));

        // Initialize session
        $_SESSION['user_id'] = $user_id;
        $_SESSION['last_activity'] = time();
        $_SESSION['login_time'] = time();

        // Trigger system health check on login (validates license with server)
        require_once __DIR__ . '/system_health.php';
        $health = check_system_integrity('login');

        // Check license status
        if (!$health['status']) {
            // License validation failed
            if ($health['code'] === 'INVALID_CONFIG') {
                // License is invalid/suspended/expired
                if (isset($health['details']['status'])) {
                    $license_status = $health['details']['status'];

                    if ($license_status === 'suspended') {
                        // License is suspended - block login
                        $_SESSION['license_status'] = 'suspended';
                        $_SESSION['license_message'] = $health['msg'] ?? 'Your license has been suspended. Please contact support.';
                        destroy_user_session();
                        header('Location: license_suspended.php');
                        exit;
                    } elseif ($license_status === 'expired') {
                        // License has expired - block login
                        $_SESSION['license_status'] = 'expired';
                        $_SESSION['license_message'] = $health['msg'] ?? 'Your license has expired. Please renew your license.';
                        destroy_user_session();
                        header('Location: license_expired.php');
                        exit;
                    }
                }

                // Generic invalid license
                $_SESSION['system_warning'] = $health['code'];
                $_SESSION['system_code'] = $health['code'];
                destroy_user_session();
                header('Location: login.php?error=invalid_license');
                exit;
            } elseif ($health['code'] === 'CONFIG_MISSING') {
                // License configuration missing - request new key
                $_SESSION['system_warning'] = $health['code'];
                destroy_user_session();
                header('Location: login.php?error=license_missing');
                exit;
            } else {
                // Other errors (non-critical warnings)
                $_SESSION['system_warning'] = $health['code'];
            }
        } else {
            // License valid - store details in session
            if (isset($health['details'])) {
                $_SESSION['license_details'] = $health['details'];
                $_SESSION['license_validated_at'] = time();
            }
            // Set system code to OK so verify_access() passes
            $_SESSION['system_code'] = 'OK';
            $_SESSION['system_warning'] = null;
        }

        // Get worker count for validation
        require_once __DIR__ . '/resource_manager.php';
        $worker_count = get_resource_usage();

        // Silent validation with current usage
        verify_resource_allocation(0, 'login');

    } catch (Exception $e) {
        // Log error
        error_log('Login recording error: ' . $e->getMessage());
    }
}
?>
