<?php
/**
 * RIU Helper Functions
 * Wrapper functions for RIU integration with obfuscated names
 */

require_once __DIR__ . '/riu_config.php';
require_once __DIR__ . '/RIUClient.php';

// Initialize RIU client
function init_monitoring_client() {
    if (!isset($_SESSION['license_key']) || !isset($_SESSION['domain'])) {
        return null;
    }

    // Only initialize if we have valid RIU tokens
    if (empty($_SESSION['riu_access_token']) || empty($_SESSION['riu_instance_id'])) {
        return null;
    }

    try {
        $client = new RIUClient(
            RIU_CONTROL_PLANE_URL,
            RIU_PRODUCT_ID,
            $_SESSION['license_key'],
            $_SESSION['domain']
        );

        // Restore tokens from session using reflection (since properties are private)
        $reflection = new ReflectionClass($client);

        $accessTokenProp = $reflection->getProperty('accessToken');
        $accessTokenProp->setAccessible(true);
        $accessTokenProp->setValue($client, $_SESSION['riu_access_token']);

        $instanceIdProp = $reflection->getProperty('instanceId');
        $instanceIdProp->setAccessible(true);
        $instanceIdProp->setValue($client, $_SESSION['riu_instance_id']);

        return $client;
    } catch (Exception $e) {
        return null;
    }
}

// Report system status (disguised as "monitor system state")
function monitor_system_state($state, $version = null, $details = array()) {
    $client = init_monitoring_client();
    if (!$client) return false;

    try {
        if ($version === null) {
            $version = RIU_VERSION;
        }

        return $client->reportStage($state, $version, $details);
    } catch (Exception $e) {
        return false;
    }
}

// Report operation progress (disguised as "log system event")
function log_system_event($event_type, $event_status, $event_message, $event_data = array()) {
    $client = init_monitoring_client();
    if (!$client) return false;

    try {
        return $client->reportProgress($event_type, $event_status, $event_message, $event_data);
    } catch (Exception $e) {
        return false;
    }
}

// Send heartbeat (disguised as "system health ping")
function send_health_ping($health_data = array()) {
    $client = init_monitoring_client();
    if (!$client) return false;

    try {
        return $client->heartbeat($health_data);
    } catch (Exception $e) {
        return false;
    }
}

// Check for updates (disguised as "check system notifications")
function check_system_notifications() {
    $client = init_monitoring_client();
    if (!$client) return array('pending' => false);

    try {
        return $client->checkPendingOperations();
    } catch (Exception $e) {
        return array('pending' => false);
    }
}

// Complete operation (disguised as "acknowledge system notification")
function acknowledge_system_notification($notification_id, $ack_status, $ack_message = null) {
    $client = init_monitoring_client();
    if (!$client) return false;

    try {
        return $client->completeOperation($notification_id, $ack_status, $ack_message);
    } catch (Exception $e) {
        return false;
    }
}

// Installation stage reporting wrapper
function report_installation_stage($step_name, $step_number, $total_steps) {
    $percentage = round(($step_number / $total_steps) * 100);

    log_system_event(
        'install',
        'running',
        "Installation Step {$step_number}/{$total_steps}: {$step_name}",
        array(
            'step' => $step_name,
            'step_number' => $step_number,
            'total_steps' => $total_steps,
            'progress' => $percentage
        )
    );
}

// Mark installation as complete
function complete_installation($version) {
    monitor_system_state('active', $version, array(
        'installation_completed' => date('Y-m-d H:i:s'),
        'installer_version' => $version
    ));
}

// Mark installation as failed
function fail_installation($error_message) {
    monitor_system_state('failed', RIU_VERSION, array(
        'error' => $error_message,
        'failed_at' => date('Y-m-d H:i:s')
    ));
}
?>
