<?php
/**
 * RIU Configuration Loader
 * Loads all configuration from database settings
 */

// Default configuration values (used during installation before database is available)
if (!defined('RIU_CONTROL_PLANE_URL')) {
    define('RIU_CONTROL_PLANE_URL', 'https://supix.co.uk/sys/riu');
}

if (!defined('RIU_PRODUCT_ID')) {
    define('RIU_PRODUCT_ID', 'scms'); // Must match product_id in RIU releases
}

if (!defined('RIU_VERSION')) {
    define('RIU_VERSION', '3.0.0');
}

// Load from database if available
if (isset($pdo)) {
    try {
        // Get RIU settings from database
        $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings
                               WHERE setting_key IN ('riu_control_plane_url', 'riu_product_id', 'riu_version',
                                                     'license_server_url', 'system_version')");
        $stmt->execute();
        $settings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        // Override constants with database values if available
        if (!empty($settings['riu_control_plane_url'])) {
            define('RIU_CONTROL_PLANE_URL_DB', $settings['riu_control_plane_url']);
        }

        if (!empty($settings['riu_product_id'])) {
            define('RIU_PRODUCT_ID_DB', $settings['riu_product_id']);
        }

        if (!empty($settings['riu_version']) || !empty($settings['system_version'])) {
            define('RIU_VERSION_DB', $settings['riu_version'] ?? $settings['system_version'] ?? RIU_VERSION);
        }

        if (!empty($settings['license_server_url'])) {
            define('LICENSE_SERVER_URL', $settings['license_server_url']);
        }

    } catch (Exception $e) {
        // Database not available yet or settings don't exist
        // Use default constants
    }
}

// Helper function to get configuration value (checks database first, then constants)
function get_riu_config($key) {
    global $pdo;

    $db_key = 'riu_' . strtolower($key);

    if (isset($pdo)) {
        try {
            $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings
                                   WHERE setting_key = ? LIMIT 1");
            $stmt->execute(array($db_key));
            $value = $stmt->fetchColumn();

            if ($value !== false) {
                return $value;
            }
        } catch (Exception $e) {
            // Fall through to constants
        }
    }

    // Fallback to constants
    $constant_name = 'RIU_' . strtoupper($key);
    if (defined($constant_name . '_DB')) {
        return constant($constant_name . '_DB');
    }
    if (defined($constant_name)) {
        return constant($constant_name);
    }

    return null;
}
