<?php
/**
 * Module Helper Functions
 * Provides easy access to module status for conditional UI
 */

/**
 * Check if a module is active (lightweight - direct DB query)
 * Usage: if (is_module_active('attendance')) { ... }
 */
function is_module_active($module_id) {
    global $pdo;

    if (!$pdo) {
        return false;
    }

    try {
        $stmt = $pdo->prepare("SELECT status FROM " . DB_PREFIX . "modules_registry WHERE module_id = ? AND status = 'active'");
        $stmt->execute([$module_id]);
        return $stmt->fetch() !== false;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Get all active modules (lightweight - direct DB query)
 * Usage: $active_modules = get_active_modules();
 */
function get_active_modules() {
    global $pdo;

    if (!$pdo) {
        return array();
    }

    try {
        $stmt = $pdo->query("SELECT module_id FROM " . DB_PREFIX . "modules_registry WHERE status = 'active' ORDER BY sort_order, name");
        $result = $stmt->fetchAll(PDO::FETCH_COLUMN);
        return $result ?: array();
    } catch (Exception $e) {
        return array();
    }
}

/**
 * Get module tabs for worker profile
 * Returns array of tabs based on active modules
 */
function get_worker_profile_tabs($worker_id) {
    $tabs = array();

    // Overview tab (always present)
    $tabs[] = array(
        'id' => 'overview',
        'label' => 'Overview',
        'icon' => 'bi-person',
        'url' => 'workers_view.php?id=' . $worker_id,
        'active' => true
    );

    // Attendance tab (if module active)
    if (is_module_active('attendance')) {
        $tabs[] = array(
            'id' => 'attendance',
            'label' => 'Attendance',
            'icon' => 'bi-calendar-check',
            'url' => 'modules/attendance/worker_attendance.php?worker_id=' . $worker_id,
            'active' => false
        );
    }

    // Holidays tab (if module active)
    if (is_module_active('holidays')) {
        $tabs[] = array(
            'id' => 'holidays',
            'label' => 'Holidays',
            'icon' => 'bi-umbrella',
            'url' => 'modules/holidays/worker_holidays.php?worker_id=' . $worker_id,
            'active' => false
        );
    }

    // Documents tab (if module active)
    if (is_module_active('documents')) {
        $tabs[] = array(
            'id' => 'documents',
            'label' => 'Documents',
            'icon' => 'bi-files',
            'url' => 'modules/documents/worker_documents.php?worker_id=' . $worker_id,
            'active' => false
        );
    }

    // RTW Checks tab (if module active)
    if (is_module_active('rtw')) {
        $tabs[] = array(
            'id' => 'rtw',
            'label' => 'RTW Checks',
            'icon' => 'bi-file-earmark-check',
            'url' => 'modules/rtw/worker_rtw.php?worker_id=' . $worker_id,
            'active' => false
        );
    }

    // CoS tab (if module active)
    if (is_module_active('cos')) {
        $tabs[] = array(
            'id' => 'cos',
            'label' => 'CoS',
            'icon' => 'bi-file-earmark-medical',
            'url' => 'modules/cos/worker_cos.php?worker_id=' . $worker_id,
            'active' => false
        );
    }

    return $tabs;
}

/**
 * Render module tabs for worker profile
 */
function render_worker_profile_tabs($worker_id, $active_tab = 'overview') {
    $tabs = get_worker_profile_tabs($worker_id);

    echo '<ul class="nav nav-tabs profile-tabs mb-4" role="tablist">';
    foreach ($tabs as $tab) {
        $is_active = ($tab['id'] === $active_tab) ? 'active' : '';
        echo '<li class="nav-item">';
        echo '<a class="nav-link ' . $is_active . '" href="' . htmlspecialchars($tab['url']) . '">';
        echo '<i class="bi ' . htmlspecialchars($tab['icon']) . '"></i> ';
        echo htmlspecialchars($tab['label']);
        echo '</a>';
        echo '</li>';
    }
    echo '</ul>';
}
?>
