<?php
/**
 * SCMS v3.0 - Module Helper Functions
 * Common functions for all modules with built-in error handling
 */

require_once __DIR__ . '/error_handler.php';

/**
 * Safe database query execution with error handling
 *
 * @param PDO $pdo Database connection
 * @param string $sql SQL query
 * @param array $params Query parameters
 * @param string $module Module name
 * @param string $operation Operation description
 * @return PDOStatement|false Statement object or false on error
 */
function safe_query($pdo, $sql, $params, $module, $operation) {
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        $error = handle_db_error($e, $module, $operation);
        $_SESSION['error_message'] = $error['error'];
        return false;
    }
}

/**
 * Safe file upload with validation and error handling
 *
 * @param array $file $_FILES array element
 * @param string $upload_dir Upload directory path
 * @param array $allowed_types Allowed MIME types
 * @param int $max_size Maximum file size in bytes
 * @param string $module Module name
 * @return array ['success' => bool, 'filename' => string|null, 'error' => string|null]
 */
function safe_upload($file, $upload_dir, $allowed_types, $max_size, $module) {
    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return handle_upload_error($file['error'], $module);
    }

    // Validate file size
    if ($file['size'] > $max_size) {
        log_error('WARNING', "File too large: {$file['size']} bytes", $module);
        return [
            'success' => false,
            'error' => 'File size exceeds maximum allowed size'
        ];
    }

    // Validate file type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime_type = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    if (!in_array($mime_type, $allowed_types)) {
        log_error('WARNING', "Invalid file type: {$mime_type}", $module);
        return [
            'success' => false,
            'error' => 'File type not allowed'
        ];
    }

    // Generate safe filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $safe_filename = uniqid() . '_' . time() . '.' . $extension;
    $target_path = $upload_dir . '/' . $safe_filename;

    // Ensure upload directory exists
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }

    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $target_path)) {
        log_error('INFO', "File uploaded successfully: {$safe_filename}", $module);
        return [
            'success' => true,
            'filename' => $safe_filename,
            'path' => $target_path
        ];
    } else {
        log_error('ERROR', "Failed to move uploaded file", $module);
        return [
            'success' => false,
            'error' => 'Failed to save uploaded file'
        ];
    }
}

/**
 * Check user permission for action
 *
 * @param int $required_level Required permission level
 * @param string $action Action description
 * @param string $module Module name
 * @return bool True if permitted, false otherwise
 */
function check_permission($required_level, $action, $module) {
    $user_level = $_SESSION['permission_level'] ?? 0;

    if ($user_level < $required_level) {
        handle_permission_error($action, $module);
        $_SESSION['error_message'] = "You do not have permission to {$action}";
        return false;
    }

    return true;
}

/**
 * Validate required POST fields
 *
 * @param array $required Array of required field names
 * @param string $module Module name
 * @return array ['valid' => bool, 'errors' => array]
 */
function validate_required($required, $module) {
    $errors = [];

    foreach ($required as $field => $label) {
        if (empty($_POST[$field]) && $_POST[$field] !== '0') {
            $errors[$field] = "{$label} is required";
        }
    }

    if (!empty($errors)) {
        log_error('WARNING', "Validation failed", $module, ['missing_fields' => array_keys($errors)]);
    }

    return [
        'valid' => empty($errors),
        'errors' => $errors
    ];
}

/**
 * Sanitize user input
 *
 * @param string $input Raw input
 * @param string $type Type of sanitization (text, email, url, int, float)
 * @return mixed Sanitized value
 */
function sanitize_input($input, $type = 'text') {
    switch ($type) {
        case 'email':
            return filter_var($input, FILTER_SANITIZE_EMAIL);
        case 'url':
            return filter_var($input, FILTER_SANITIZE_URL);
        case 'int':
            return filter_var($input, FILTER_SANITIZE_NUMBER_INT);
        case 'float':
            return filter_var($input, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION);
        case 'text':
        default:
            return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
}

/**
 * Log module activity (info level)
 *
 * @param string $action Action performed
 * @param string $module Module name
 * @param array $details Additional details
 */
function log_activity($action, $module, $details = []) {
    log_error('INFO', $action, $module, $details);
}

/**
 * Get safe redirect URL (prevent open redirect)
 *
 * @param string $url URL to redirect to
 * @param string $default Default URL if validation fails
 * @return string Safe redirect URL
 */
function get_safe_redirect($url, $default = 'dashboard.php') {
    // Only allow relative URLs within the application
    if (strpos($url, 'http://') === 0 || strpos($url, 'https://') === 0) {
        return $default;
    }

    // Remove any path traversal attempts
    $url = str_replace(['../', '..\\'], '', $url);

    return $url;
}

/**
 * Format error message for display
 *
 * @param string $message Error message
 * @param string $type Type (danger, warning, info, success)
 * @return string HTML alert
 */
function format_alert($message, $type = 'danger') {
    $icons = [
        'danger' => 'exclamation-triangle-fill',
        'warning' => 'exclamation-triangle',
        'info' => 'info-circle-fill',
        'success' => 'check-circle-fill'
    ];

    $icon = $icons[$type] ?? 'info-circle';

    return sprintf(
        '<div class="alert alert-%s alert-dismissible fade show" role="alert">
            <i class="bi bi-%s"></i> %s
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>',
        $type,
        $icon,
        htmlspecialchars($message)
    );
}

/**
 * AJAX response helper
 *
 * @param bool $success Success status
 * @param mixed $data Response data
 * @param string|null $error Error message
 */
function ajax_response($success, $data = null, $error = null) {
    header('Content-Type: application/json');
    echo json_encode([
        'success' => $success,
        'data' => $data,
        'error' => $error
    ]);
    exit;
}

/**
 * Check if request is AJAX
 *
 * @return bool
 */
function is_ajax_request() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Get pagination data
 *
 * @param int $total_records Total number of records
 * @param int $per_page Records per page
 * @param int $current_page Current page number
 * @return array Pagination data
 */
function get_pagination($total_records, $per_page = 20, $current_page = 1) {
    $total_pages = ceil($total_records / $per_page);
    $current_page = max(1, min($current_page, $total_pages));
    $offset = ($current_page - 1) * $per_page;

    return [
        'total_records' => $total_records,
        'per_page' => $per_page,
        'current_page' => $current_page,
        'total_pages' => $total_pages,
        'offset' => $offset,
        'has_prev' => $current_page > 1,
        'has_next' => $current_page < $total_pages
    ];
}

/**
 * Render pagination HTML
 *
 * @param array $pagination Pagination data from get_pagination()
 * @param string $base_url Base URL for pagination links
 * @return string HTML pagination
 */
function render_pagination($pagination, $base_url) {
    if ($pagination['total_pages'] <= 1) {
        return '';
    }

    $html = '<nav><ul class="pagination justify-content-center">';

    // Previous button
    if ($pagination['has_prev']) {
        $prev_page = $pagination['current_page'] - 1;
        $html .= '<li class="page-item"><a class="page-link" href="' . $base_url . '&page=' . $prev_page . '">Previous</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">Previous</span></li>';
    }

    // Page numbers
    $start = max(1, $pagination['current_page'] - 2);
    $end = min($pagination['total_pages'], $pagination['current_page'] + 2);

    if ($start > 1) {
        $html .= '<li class="page-item"><a class="page-link" href="' . $base_url . '&page=1">1</a></li>';
        if ($start > 2) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
    }

    for ($i = $start; $i <= $end; $i++) {
        $active = $i === $pagination['current_page'] ? 'active' : '';
        $html .= '<li class="page-item ' . $active . '"><a class="page-link" href="' . $base_url . '&page=' . $i . '">' . $i . '</a></li>';
    }

    if ($end < $pagination['total_pages']) {
        if ($end < $pagination['total_pages'] - 1) {
            $html .= '<li class="page-item disabled"><span class="page-link">...</span></li>';
        }
        $html .= '<li class="page-item"><a class="page-link" href="' . $base_url . '&page=' . $pagination['total_pages'] . '">' . $pagination['total_pages'] . '</a></li>';
    }

    // Next button
    if ($pagination['has_next']) {
        $next_page = $pagination['current_page'] + 1;
        $html .= '<li class="page-item"><a class="page-link" href="' . $base_url . '&page=' . $next_page . '">Next</a></li>';
    } else {
        $html .= '<li class="page-item disabled"><span class="page-link">Next</span></li>';
    }

    $html .= '</ul></nav>';

    return $html;
}

?>
