<?php
/**
 * SCMS Localization Configuration
 * British English as Default
 */

// Locale settings
define('LOCALE_LANGUAGE', 'en-GB'); // British English
define('LOCALE_TIMEZONE', 'Europe/London');
define('LOCALE_CURRENCY', 'GBP');
define('LOCALE_CURRENCY_SYMBOL', '£');

// Date and time formats (British)
define('LOCALE_DATE_FORMAT', 'd/m/Y'); // 18/10/2025
define('LOCALE_DATE_SHORT', 'd/m/y'); // 18/10/25
define('LOCALE_DATE_LONG', 'j F Y'); // 18 October 2025
define('LOCALE_DATETIME_FORMAT', 'd/m/Y H:i'); // 18/10/2025 14:30
define('LOCALE_TIME_FORMAT', 'H:i'); // 14:30 (24-hour)
define('LOCALE_TIME_12H', 'g:i A'); // 2:30 PM (12-hour)

// Database date formats
define('DB_DATE_FORMAT', 'Y-m-d'); // MySQL format
define('DB_DATETIME_FORMAT', 'Y-m-d H:i:s'); // MySQL format

// Number formats
define('LOCALE_DECIMAL_SEPARATOR', '.');
define('LOCALE_THOUSANDS_SEPARATOR', ',');
define('LOCALE_DECIMAL_PLACES', 2);

// British English spelling corrections
$british_spellings = [
    'color' => 'colour',
    'colors' => 'colours',
    'favorite' => 'favourite',
    'favorites' => 'favourites',
    'center' => 'centre',
    'centers' => 'centres',
    'license' => 'licence', // noun form
    'organize' => 'organise',
    'organized' => 'organised',
    'organization' => 'organisation',
    'realize' => 'realise',
    'realized' => 'realised',
    'honor' => 'honour',
    'honors' => 'honours',
    'labor' => 'labour',
    'theater' => 'theatre',
    'meter' => 'metre',
    'analyze' => 'analyse',
    'catalog' => 'catalogue',
    'defense' => 'defence',
    'offense' => 'offence',
    'program' => 'programme', // except computer programs
];

/**
 * Format date for display (British format)
 *
 * @param string $date Date string or timestamp
 * @param string $format Format type: 'short', 'long', 'datetime', 'time'
 * @return string Formatted date
 */
function format_date($date, $format = 'short') {
    if (empty($date) || $date === '0000-00-00' || $date === '0000-00-00 00:00:00') {
        return '';
    }

    $timestamp = is_numeric($date) ? $date : strtotime($date);

    switch ($format) {
        case 'short':
            return date(LOCALE_DATE_SHORT, $timestamp);
        case 'long':
            return date(LOCALE_DATE_LONG, $timestamp);
        case 'datetime':
            return date(LOCALE_DATETIME_FORMAT, $timestamp);
        case 'time':
            return date(LOCALE_TIME_FORMAT, $timestamp);
        case 'time_12h':
            return date(LOCALE_TIME_12H, $timestamp);
        default:
            return date(LOCALE_DATE_FORMAT, $timestamp);
    }
}

/**
 * Format date for database (MySQL format)
 *
 * @param string $date Date in British format (dd/mm/yyyy)
 * @return string Date in MySQL format (yyyy-mm-dd)
 */
function format_date_for_db($date) {
    if (empty($date)) {
        return null;
    }

    // Check if already in MySQL format
    if (preg_match('/^\d{4}-\d{2}-\d{2}/', $date)) {
        return $date;
    }

    // Convert British format to MySQL
    $parts = explode('/', $date);
    if (count($parts) === 3) {
        // Handle 2-digit year
        $year = strlen($parts[2]) === 2 ? '20' . $parts[2] : $parts[2];
        return sprintf('%04d-%02d-%02d', $year, $parts[1], $parts[0]);
    }

    // Try to parse with strtotime
    $timestamp = strtotime($date);
    if ($timestamp !== false) {
        return date(DB_DATE_FORMAT, $timestamp);
    }

    return null;
}

/**
 * Format currency (British Pounds)
 *
 * @param float $amount Amount to format
 * @param bool $show_symbol Show currency symbol
 * @return string Formatted currency
 */
function format_currency($amount, $show_symbol = true) {
    $formatted = number_format(
        $amount,
        LOCALE_DECIMAL_PLACES,
        LOCALE_DECIMAL_SEPARATOR,
        LOCALE_THOUSANDS_SEPARATOR
    );

    return $show_symbol ? LOCALE_CURRENCY_SYMBOL . $formatted : $formatted;
}

/**
 * Format number (British format)
 *
 * @param float $number Number to format
 * @param int $decimals Decimal places
 * @return string Formatted number
 */
function format_number($number, $decimals = null) {
    if ($decimals === null) {
        $decimals = LOCALE_DECIMAL_PLACES;
    }

    return number_format(
        $number,
        $decimals,
        LOCALE_DECIMAL_SEPARATOR,
        LOCALE_THOUSANDS_SEPARATOR
    );
}

/**
 * Get day name (British English)
 *
 * @param int $day_number 0 (Sunday) to 6 (Saturday)
 * @return string Day name
 */
function get_day_name($day_number, $short = false) {
    $days_long = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
    $days_short = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

    return $short ? $days_short[$day_number] : $days_long[$day_number];
}

/**
 * Get month name (British English)
 *
 * @param int $month_number 1 (January) to 12 (December)
 * @return string Month name
 */
function get_month_name($month_number, $short = false) {
    $months_long = [
        1 => 'January', 'February', 'March', 'April', 'May', 'June',
        'July', 'August', 'September', 'October', 'November', 'December'
    ];
    $months_short = [
        1 => 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
        'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'
    ];

    return $short ? $months_short[$month_number] : $months_long[$month_number];
}

/**
 * Convert American spelling to British
 *
 * @param string $text Text to convert
 * @return string Text with British spelling
 */
function british_spelling($text) {
    global $british_spellings;

    foreach ($british_spellings as $american => $british) {
        $text = str_replace($american, $british, $text);
        $text = str_replace(ucfirst($american), ucfirst($british), $text);
    }

    return $text;
}

/**
 * Get locale configuration for JavaScript
 *
 * @return array Locale configuration
 */
function get_locale_config() {
    return [
        'language' => LOCALE_LANGUAGE,
        'timezone' => LOCALE_TIMEZONE,
        'currency' => LOCALE_CURRENCY,
        'currencySymbol' => LOCALE_CURRENCY_SYMBOL,
        'dateFormat' => 'DD/MM/YYYY',
        'dateFormatShort' => 'DD/MM/YY',
        'dateFormatLong' => 'D MMMM YYYY',
        'timeFormat' => 'HH:mm',
        'timeFormat12h' => 'h:mm A',
        'datetimeFormat' => 'DD/MM/YYYY HH:mm',
        'firstDayOfWeek' => 1, // Monday
        'decimalSeparator' => LOCALE_DECIMAL_SEPARATOR,
        'thousandsSeparator' => LOCALE_THOUSANDS_SEPARATOR,
    ];
}

// Set PHP locale
setlocale(LC_TIME, 'en_GB.UTF-8', 'en_GB', 'English_United Kingdom.1252');

// Set timezone
date_default_timezone_set(LOCALE_TIMEZONE);
?>
