<?php
/**
 * SCMS v3.0 - Centralized Error Handler
 * Provides unified error logging, handling, and reporting across all modules
 */

// Error log file location
define('ERROR_LOG_FILE', __DIR__ . '/../logs/scms_errors.log');
define('ERROR_LOG_MAX_SIZE', 5 * 1024 * 1024); // 5MB max log size

// Ensure logs directory exists
if (!file_exists(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

/**
 * Log error to centralized log file
 *
 * @param string $level Error level (ERROR, WARNING, INFO, DEBUG)
 * @param string $message Error message
 * @param string $module Module name (e.g., 'workers', 'rtw', 'cos')
 * @param array $context Additional context data
 * @param Exception|null $exception Exception object if available
 */
function log_error($level, $message, $module = 'system', $context = [], $exception = null) {
    // Rotate log if too large
    if (file_exists(ERROR_LOG_FILE) && filesize(ERROR_LOG_FILE) > ERROR_LOG_MAX_SIZE) {
        rotate_error_log();
    }

    // Prepare log entry
    $timestamp = date('Y-m-d H:i:s');
    $user_id = $_SESSION['user_id'] ?? 'guest';
    $ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    $request_uri = $_SERVER['REQUEST_URI'] ?? 'CLI';

    $log_entry = sprintf(
        "[%s] [%s] [%s] [User:%s] [IP:%s] %s",
        $timestamp,
        strtoupper($level),
        strtoupper($module),
        $user_id,
        $ip,
        $message
    );

    // Add context if provided
    if (!empty($context)) {
        $log_entry .= " | Context: " . json_encode($context);
    }

    // Add exception details if provided
    if ($exception instanceof Exception) {
        $log_entry .= sprintf(
            " | Exception: %s in %s:%d | Stack: %s",
            $exception->getMessage(),
            $exception->getFile(),
            $exception->getLine(),
            $exception->getTraceAsString()
        );
    }

    $log_entry .= " | URI: " . $request_uri . PHP_EOL;

    // Write to log file
    file_put_contents(ERROR_LOG_FILE, $log_entry, FILE_APPEND | LOCK_EX);

    // For critical errors, also log to PHP error log
    if (in_array(strtoupper($level), ['ERROR', 'CRITICAL', 'ALERT', 'EMERGENCY'])) {
        error_log($log_entry);
    }
}

/**
 * Rotate error log file
 */
function rotate_error_log() {
    if (file_exists(ERROR_LOG_FILE)) {
        $backup_file = ERROR_LOG_FILE . '.' . date('Ymd_His');
        rename(ERROR_LOG_FILE, $backup_file);

        // Keep only last 5 backup files
        $log_dir = dirname(ERROR_LOG_FILE);
        $backup_files = glob($log_dir . '/scms_errors.log.*');
        if (count($backup_files) > 5) {
            usort($backup_files, function($a, $b) {
                return filemtime($a) - filemtime($b);
            });
            array_splice($backup_files, 0, count($backup_files) - 5);
            foreach ($backup_files as $old_file) {
                unlink($old_file);
            }
        }
    }
}

/**
 * Handle database errors uniformly
 *
 * @param PDOException $e PDO exception
 * @param string $module Module name
 * @param string $operation Operation being performed
 * @return array Error response array
 */
function handle_db_error($e, $module, $operation) {
    $error_code = $e->getCode();
    $error_msg = $e->getMessage();

    // Log the full error
    log_error('ERROR', "Database error during {$operation}", $module, [
        'code' => $error_code,
        'message' => $error_msg
    ], $e);

    // Return user-friendly message
    $user_message = "A database error occurred";

    // Provide specific messages for common errors
    if (strpos($error_msg, 'Duplicate entry') !== false) {
        $user_message = "This record already exists";
    } elseif (strpos($error_msg, 'foreign key constraint fails') !== false) {
        $user_message = "Cannot delete: record is referenced by other data";
    } elseif (strpos($error_msg, 'Column not found') !== false) {
        $user_message = "Database schema error - please contact administrator";
    }

    return [
        'success' => false,
        'error' => $user_message,
        'error_code' => $error_code,
        'logged' => true
    ];
}

/**
 * Handle validation errors
 *
 * @param array $errors Array of validation errors
 * @param string $module Module name
 * @return array Error response array
 */
function handle_validation_error($errors, $module) {
    log_error('WARNING', "Validation failed", $module, ['errors' => $errors]);

    return [
        'success' => false,
        'error' => 'Please correct the following errors',
        'validation_errors' => $errors
    ];
}

/**
 * Handle file upload errors
 *
 * @param int $error_code PHP file upload error code
 * @param string $module Module name
 * @return array Error response array
 */
function handle_upload_error($error_code, $module) {
    $messages = [
        UPLOAD_ERR_INI_SIZE => 'File is too large (exceeds server limit)',
        UPLOAD_ERR_FORM_SIZE => 'File is too large (exceeds form limit)',
        UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
        UPLOAD_ERR_NO_FILE => 'No file was uploaded',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
        UPLOAD_ERR_EXTENSION => 'File upload blocked by extension'
    ];

    $message = $messages[$error_code] ?? 'Unknown upload error';

    log_error('ERROR', "File upload failed: {$message}", $module, ['error_code' => $error_code]);

    return [
        'success' => false,
        'error' => $message
    ];
}

/**
 * Handle permission errors
 *
 * @param string $action Action being attempted
 * @param string $module Module name
 * @return array Error response array
 */
function handle_permission_error($action, $module) {
    log_error('WARNING', "Permission denied for action: {$action}", $module, [
        'user_permission' => $_SESSION['permission_level'] ?? 'none'
    ]);

    return [
        'success' => false,
        'error' => 'You do not have permission to perform this action'
    ];
}

/**
 * Get recent errors from log
 *
 * @param int $limit Number of recent errors to retrieve
 * @param string|null $level Filter by error level
 * @param string|null $module Filter by module
 * @return array Array of log entries
 */
function get_recent_errors($limit = 100, $level = null, $module = null) {
    if (!file_exists(ERROR_LOG_FILE)) {
        return [];
    }

    $lines = file(ERROR_LOG_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $lines = array_reverse($lines);
    $results = [];

    foreach ($lines as $line) {
        if (count($results) >= $limit) {
            break;
        }

        // Parse log line
        if (preg_match('/\[(.*?)\] \[(.*?)\] \[(.*?)\] \[User:(.*?)\] \[IP:(.*?)\] (.*)/', $line, $matches)) {
            $entry = [
                'timestamp' => $matches[1],
                'level' => $matches[2],
                'module' => $matches[3],
                'user_id' => $matches[4],
                'ip' => $matches[5],
                'message' => $matches[6]
            ];

            // Apply filters
            if ($level && strcasecmp($entry['level'], $level) !== 0) {
                continue;
            }
            if ($module && strcasecmp($entry['module'], $module) !== 0) {
                continue;
            }

            $results[] = $entry;
        }
    }

    return $results;
}

/**
 * Clear error log
 *
 * @return bool Success status
 */
function clear_error_log() {
    if (file_exists(ERROR_LOG_FILE)) {
        // Backup before clearing
        $backup_file = ERROR_LOG_FILE . '.cleared.' . date('Ymd_His');
        copy(ERROR_LOG_FILE, $backup_file);

        // Clear the log
        file_put_contents(ERROR_LOG_FILE, '');

        log_error('INFO', 'Error log cleared', 'system');
        return true;
    }
    return false;
}

/**
 * Set custom error handler
 */
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    // Don't log suppressed errors (@)
    if (!(error_reporting() & $errno)) {
        return false;
    }

    $level = 'ERROR';
    switch ($errno) {
        case E_WARNING:
        case E_USER_WARNING:
            $level = 'WARNING';
            break;
        case E_NOTICE:
        case E_USER_NOTICE:
            $level = 'NOTICE';
            break;
        case E_DEPRECATED:
        case E_USER_DEPRECATED:
            $level = 'DEPRECATED';
            break;
    }

    log_error($level, $errstr, 'php', [
        'file' => $errfile,
        'line' => $errline,
        'type' => $errno
    ]);

    // Don't execute PHP internal error handler
    return true;
});

/**
 * Set custom exception handler
 */
set_exception_handler(function($exception) {
    log_error('CRITICAL', 'Uncaught exception', 'system', [], $exception);

    // Show generic error page
    if (!headers_sent()) {
        http_response_code(500);
        if (file_exists(__DIR__ . '/../error_pages/500.html')) {
            include __DIR__ . '/../error_pages/500.html';
        } else {
            echo "An unexpected error occurred. Please try again later.";
        }
    }
});

/**
 * Register shutdown function to catch fatal errors
 */
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        log_error('CRITICAL', $error['message'], 'php', [
            'file' => $error['file'],
            'line' => $error['line'],
            'type' => $error['type']
        ]);
    }
});

?>
