<?php
/**
 * Email Helper for SCMS
 * Based on proven L Server implementation
 * Supports direct SMTP using fsockopen
 */

class EmailHelper {
    private $pdo;
    private $settings = array();
    public $last_error = '';

    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->loadSettings();
    }

    /**
     * Load settings from database
     */
    private function loadSettings() {
        try {
            $stmt = $this->pdo->query("SELECT setting_name, setting_value FROM " . DB_PREFIX . "settings");
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach ($results as $row) {
                $this->settings[$row['setting_name']] = $row['setting_value'];
            }
        } catch (PDOException $e) {
            error_log("EmailHelper: Failed to load settings - " . $e->getMessage());
        }
    }

    /**
     * Get setting value
     */
    private function getSetting($key, $default = '') {
        return isset($this->settings[$key]) ? $this->settings[$key] : $default;
    }

    /**
     * Send email immediately via SMTP
     */
    public function sendEmail($to_email, $to_name, $subject, $body, $is_html = true) {
        // Check if SMTP is enabled
        if ($this->getSetting('smtp_enabled', '0') !== '1') {
            $this->last_error = 'SMTP is not enabled in settings';
            return false;
        }

        // Wrap body in template if HTML
        if ($is_html) {
            $body = $this->wrapEmailTemplate($body);
        }

        // Send via direct SMTP
        return $this->sendViaSMTP($to_email, $to_name, $subject, $body, $is_html);
    }

    /**
     * Queue email for later sending
     */
    public function queueEmail($to_email, $to_name, $subject, $body, $type = 'general', $related_id = null) {
        try {
            $stmt = $this->pdo->prepare(
                "INSERT INTO " . DB_PREFIX . "email_queue
                (recipient_email, recipient_name, subject, body, type, related_id, status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, 'pending', NOW())"
            );

            $stmt->execute([$to_email, $to_name, $subject, $body, $type, $related_id]);

            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            $this->last_error = "Failed to queue email: " . $e->getMessage();
            return false;
        }
    }

    /**
     * Send via direct SMTP using fsockopen
     */
    private function sendViaSMTP($to_email, $to_name, $subject, $body, $is_html = true) {
        try {
            // Get SMTP settings
            $smtp_host = $this->getSetting('smtp_host', '');
            $smtp_port = (int)$this->getSetting('smtp_port', '465');
            $smtp_user = $this->getSetting('smtp_username', '');
            $smtp_pass = $this->getSetting('smtp_password', '');
            $smtp_encryption = strtolower($this->getSetting('smtp_encryption', 'ssl'));
            $from_email = $this->getSetting('smtp_from_email', 'noreply@' . $_SERVER['SERVER_NAME']);
            $from_name = $this->getSetting('smtp_from_name', 'SCMS System');

            // Validate required settings
            if (empty($smtp_host) || empty($smtp_user) || empty($smtp_pass)) {
                $this->last_error = 'SMTP credentials not configured';
                return false;
            }

            // Determine connection type
            if ($smtp_port == 465 || $smtp_encryption === 'ssl') {
                $host_connect = 'ssl://' . $smtp_host;
                $use_starttls = false;
            } else {
                $host_connect = $smtp_host;
                $use_starttls = ($smtp_port == 587 || $smtp_encryption === 'tls');
            }

            // Connect to SMTP server
            $smtp = @fsockopen($host_connect, $smtp_port, $errno, $errstr, 30);

            if (!$smtp) {
                $this->last_error = "Connection failed: $errstr ($errno)";
                return false;
            }

            // Read greeting
            $response = '';
            while ($line = fgets($smtp, 515)) {
                $response .= $line;
                if (preg_match('/^220 /', $line)) {
                    break;
                }
            }

            if (substr($response, 0, 3) !== '220') {
                fclose($smtp);
                $this->last_error = "Invalid greeting: $response";
                return false;
            }

            // Send EHLO
            fputs($smtp, "EHLO " . $_SERVER['SERVER_NAME'] . "\r\n");
            $response = '';
            while ($line = fgets($smtp, 515)) {
                $response .= $line;
                if (substr($line, 3, 1) === ' ') break;
            }

            // STARTTLS if needed
            if ($use_starttls) {
                fputs($smtp, "STARTTLS\r\n");
                $response = fgets($smtp, 515);
                if (substr($response, 0, 3) !== '220') {
                    fclose($smtp);
                    $this->last_error = "STARTTLS failed: $response";
                    return false;
                }

                stream_socket_enable_crypto($smtp, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);

                // EHLO again after STARTTLS
                fputs($smtp, "EHLO " . $_SERVER['SERVER_NAME'] . "\r\n");
                $response = '';
                while ($line = fgets($smtp, 515)) {
                    $response .= $line;
                    if (substr($line, 3, 1) === ' ') break;
                }
            }

            // AUTH LOGIN
            fputs($smtp, "AUTH LOGIN\r\n");
            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '334') {
                fclose($smtp);
                $this->last_error = "AUTH LOGIN failed: $response";
                return false;
            }

            // Send username
            fputs($smtp, base64_encode($smtp_user) . "\r\n");
            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '334') {
                fclose($smtp);
                $this->last_error = "Username rejected: $response";
                return false;
            }

            // Send password
            fputs($smtp, base64_encode($smtp_pass) . "\r\n");
            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '235') {
                fclose($smtp);
                $this->last_error = "Password rejected: $response";
                return false;
            }

            // MAIL FROM
            fputs($smtp, "MAIL FROM: <$from_email>\r\n");
            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '250') {
                fclose($smtp);
                $this->last_error = "MAIL FROM rejected: $response";
                return false;
            }

            // RCPT TO
            fputs($smtp, "RCPT TO: <$to_email>\r\n");
            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '250') {
                fclose($smtp);
                $this->last_error = "RCPT TO rejected: $response";
                return false;
            }

            // DATA
            fputs($smtp, "DATA\r\n");
            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '354') {
                fclose($smtp);
                $this->last_error = "DATA rejected: $response";
                return false;
            }

            // Build message headers
            $headers = "From: $from_name <$from_email>\r\n";
            $headers .= "To: $to_name <$to_email>\r\n";
            $headers .= "Subject: $subject\r\n";
            $headers .= "MIME-Version: 1.0\r\n";

            if ($is_html) {
                $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
                $headers .= "Content-Transfer-Encoding: quoted-printable\r\n";
                $encoded_body = quoted_printable_encode($body);
            } else {
                $headers .= "Content-Type: text/plain; charset=UTF-8\r\n";
                $headers .= "Content-Transfer-Encoding: 8bit\r\n";
                $encoded_body = $body;
            }

            $headers .= "Date: " . date('r') . "\r\n";
            $headers .= "X-Mailer: SCMS Email System\r\n";
            $headers .= "Reply-To: " . $this->getSetting('smtp_reply_to', $from_email) . "\r\n";

            // Send message
            $message = $headers . "\r\n" . $encoded_body . "\r\n.\r\n";
            fputs($smtp, $message);

            $response = fgets($smtp, 515);
            if (substr($response, 0, 3) !== '250') {
                fclose($smtp);
                $this->last_error = "Message rejected: $response";
                return false;
            }

            // QUIT
            fputs($smtp, "QUIT\r\n");
            fclose($smtp);

            // Log success
            $this->logEmail($to_email, $subject, 'sent');

            return true;

        } catch (Exception $e) {
            $this->last_error = 'Exception: ' . $e->getMessage();
            return false;
        }
    }

    /**
     * Process email queue
     */
    public function processQueue($limit = 10) {
        $sent = 0;
        $failed = 0;

        try {
            $stmt = $this->pdo->prepare(
                "SELECT * FROM " . DB_PREFIX . "email_queue
                WHERE status = 'pending'
                ORDER BY created_at ASC
                LIMIT ?"
            );
            $stmt->execute([$limit]);
            $emails = $stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach ($emails as $email) {
                $result = $this->sendEmail(
                    $email['recipient_email'],
                    $email['recipient_name'],
                    $email['subject'],
                    $email['body']
                );

                if ($result) {
                    $this->updateQueueStatus($email['id'], 'sent');
                    $sent++;
                } else {
                    $this->updateQueueStatus($email['id'], 'failed', $this->last_error);
                    $failed++;
                }
            }
        } catch (PDOException $e) {
            error_log("Email queue processing error: " . $e->getMessage());
        }

        return ['sent' => $sent, 'failed' => $failed];
    }

    /**
     * Update queue status
     */
    private function updateQueueStatus($id, $status, $error = '') {
        try {
            $stmt = $this->pdo->prepare(
                "UPDATE " . DB_PREFIX . "email_queue
                SET status = ?, error_message = ?, sent_at = IF(? = 'sent', NOW(), NULL)
                WHERE id = ?"
            );
            $stmt->execute([$status, $error, $status, $id]);
        } catch (PDOException $e) {
            error_log("Failed to update queue status: " . $e->getMessage());
        }
    }

    /**
     * Log email activity
     */
    private function logEmail($to, $subject, $status) {
        try {
            $stmt = $this->pdo->prepare(
                "INSERT INTO " . DB_PREFIX . "email_log
                (recipient, subject, status, sent_at)
                VALUES (?, ?, ?, NOW())"
            );
            $stmt->execute([$to, $subject, $status]);
        } catch (PDOException $e) {
            // Table might not exist, silently fail
        }
    }

    /**
     * Wrap email in responsive HTML template
     */
    public function wrapEmailTemplate($body) {
        $company_name = $this->getSetting('company_name', 'SCMS');
        $company_color = $this->getSetting('primary_color', '#667eea');

        $template = '
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
<head>
    <meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>' . htmlspecialchars($company_name) . '</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
    <table border="0" cellpadding="0" cellspacing="0" width="100%" style="background-color: #f4f4f4;">
        <tr>
            <td align="center" style="padding: 20px 0;">
                <table border="0" cellpadding="0" cellspacing="0" width="600" style="background-color: #ffffff; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                    <!-- Header -->
                    <tr>
                        <td align="center" style="background-color: ' . htmlspecialchars($company_color) . '; padding: 40px 30px; border-radius: 8px 8px 0 0;">
                            <h1 style="margin: 0; color: #ffffff; font-size: 28px; font-weight: bold;">
                                ' . htmlspecialchars($company_name) . '
                            </h1>
                            <p style="margin: 10px 0 0 0; color: #e0e7ff; font-size: 14px;">
                                Staff Compliance Management System
                            </p>
                        </td>
                    </tr>

                    <!-- Content -->
                    <tr>
                        <td style="padding: 40px 30px; color: #333333; font-size: 16px; line-height: 1.8;">
                            ' . $body . '
                        </td>
                    </tr>

                    <!-- Footer -->
                    <tr>
                        <td align="center" style="background-color: #f8f9fa; padding: 30px; border-radius: 0 0 8px 8px; border-top: 1px solid #e9ecef;">
                            <p style="margin: 0; font-size: 12px; color: #6c757d; line-height: 1.6;">
                                © ' . date('Y') . ' ' . htmlspecialchars($company_name) . '. All rights reserved.<br/>
                                <span style="color: #adb5bd;">This is an automated message. Please do not reply to this email.</span>
                            </p>
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';

        return $template;
    }

    /**
     * Send test email
     */
    public function sendTestEmail($to_email) {
        $subject = "Test Email from SCMS";
        $body = "<h3>Test Email Successful!</h3>
                 <p>This is a test email from your SCMS installation.</p>
                 <p>If you're receiving this, your SMTP settings are configured correctly.</p>
                 <hr>
                 <p><strong>Configuration Details:</strong></p>
                 <ul>
                    <li>SMTP Host: " . $this->getSetting('smtp_host') . "</li>
                    <li>SMTP Port: " . $this->getSetting('smtp_port') . "</li>
                    <li>Encryption: " . $this->getSetting('smtp_encryption') . "</li>
                    <li>From Email: " . $this->getSetting('smtp_from_email') . "</li>
                 </ul>
                 <p>Sent at: " . date('Y-m-d H:i:s') . "</p>";

        return $this->sendEmail($to_email, 'Test Recipient', $subject, $body);
    }

    /**
     * RTW specific email templates
     */
    public function sendRTWExpiryAlert($worker_email, $worker_name, $days_until_expiry) {
        $subject = "RTW Check Expiry Alert - Action Required";

        if ($days_until_expiry < 0) {
            $message = "Your Right to Work check has <strong style='color: #dc3545;'>EXPIRED</strong>.";
            $action = "Please contact HR immediately to update your documentation.";
        } elseif ($days_until_expiry == 0) {
            $message = "Your Right to Work check <strong style='color: #dc3545;'>expires TODAY</strong>.";
            $action = "Please submit your updated documents immediately.";
        } elseif ($days_until_expiry <= 7) {
            $message = "Your Right to Work check will expire in <strong style='color: #ffc107;'>$days_until_expiry days</strong>.";
            $action = "Please prepare your documents for renewal.";
        } else {
            $message = "Your Right to Work check will expire in <strong>$days_until_expiry days</strong>.";
            $action = "Please ensure your documents are ready for renewal.";
        }

        $body = "<h3>Dear $worker_name,</h3>
                 <p>$message</p>
                 <div style='background-color: #f8f9fa; padding: 15px; border-left: 4px solid #dc3545; margin: 20px 0;'>
                    <p style='margin: 0; font-weight: bold;'>$action</p>
                 </div>
                 <p><strong>What you need to do:</strong></p>
                 <ul>
                    <li>Prepare your updated RTW documents</li>
                    <li>Contact your HR department</li>
                    <li>Schedule a document verification appointment</li>
                 </ul>
                 <p>If you have any questions, please contact the HR team.</p>";

        return $this->queueEmail($worker_email, $worker_name, $subject, $body, 'rtw_expiry', null);
    }

    /**
     * Send RTW check confirmation
     */
    public function sendRTWCheckConfirmation($worker_email, $worker_name, $check_date, $outcome, $next_check_date = null) {
        $subject = "RTW Check Completed - " . ucfirst($outcome);

        $outcome_message = $outcome == 'pass'
            ? "<span style='color: #28a745;'>PASSED</span>"
            : "<span style='color: #dc3545;'>REQUIRES ATTENTION</span>";

        $body = "<h3>Dear $worker_name,</h3>
                 <p>Your Right to Work check has been completed on <strong>" . date('d/m/Y', strtotime($check_date)) . "</strong>.</p>
                 <p>Check Result: <strong>$outcome_message</strong></p>";

        if ($next_check_date) {
            $body .= "<p>Next check due: <strong>" . date('d/m/Y', strtotime($next_check_date)) . "</strong></p>";
        }

        $body .= "<p>A copy of this verification has been filed in your employment records.</p>
                  <p>If you have any questions, please contact the HR department.</p>";

        return $this->sendEmail($worker_email, $worker_name, $subject, $body);
    }
}

// Helper function for quick email sending
function send_email($pdo, $to, $subject, $body, $to_name = '') {
    $emailHelper = new EmailHelper($pdo);
    return $emailHelper->sendEmail($to, $to_name, $subject, $body);
}

// Helper function for queuing emails
function queue_email($pdo, $to, $subject, $body, $to_name = '', $type = 'general') {
    $emailHelper = new EmailHelper($pdo);
    return $emailHelper->queueEmail($to, $to_name, $subject, $body, $type);
}