<?php
/**
 * Database Connection Handler
 * Provides PDO connection and helper functions
 */

// Load database configuration
if (!defined('DB_HOST')) {
    if (file_exists(__DIR__ . '/../config/database.php')) {
        require_once __DIR__ . '/../config/database.php';
    } else {
        die('Database configuration not found. Please run the installer.');
    }
}

// Initialize PDO connection
try {
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
    $pdo = new PDO($dsn, DB_USER, DB_PASS, array(
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
    ));
} catch (PDOException $e) {
    error_log('Database connection failed: ' . $e->getMessage());
    die('Database connection failed. Please check your configuration.');
}

/**
 * Get database connection (for compatibility)
 */
function get_database_connection() {
    global $pdo;
    return $pdo;
}

/**
 * Execute a prepared statement with parameters
 */
function db_execute($sql, $params = array()) {
    global $pdo;
    try {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        error_log('Database query failed: ' . $e->getMessage());
        return false;
    }
}

/**
 * Fetch a single row
 */
function db_fetch($sql, $params = array()) {
    $stmt = db_execute($sql, $params);
    return $stmt ? $stmt->fetch() : false;
}

/**
 * Fetch all rows
 */
function db_fetch_all($sql, $params = array()) {
    $stmt = db_execute($sql, $params);
    return $stmt ? $stmt->fetchAll() : array();
}

/**
 * Get last insert ID
 */
function db_last_insert_id() {
    global $pdo;
    return $pdo->lastInsertId();
}

/**
 * Begin transaction
 */
function db_begin_transaction() {
    global $pdo;
    return $pdo->beginTransaction();
}

/**
 * Commit transaction
 */
function db_commit() {
    global $pdo;
    return $pdo->commit();
}

/**
 * Rollback transaction
 */
function db_rollback() {
    global $pdo;
    return $pdo->rollBack();
}

/**
 * Escape string for LIKE queries
 */
function db_escape_like($string) {
    return str_replace(array('%', '_'), array('\\%', '\\_'), $string);
}

/**
 * Check if table exists
 */
function db_table_exists($table_name) {
    global $pdo;
    try {
        $result = $pdo->query("SHOW TABLES LIKE '" . DB_PREFIX . $table_name . "'");
        return $result->rowCount() > 0;
    } catch (PDOException $e) {
        return false;
    }
}

/**
 * Get database version
 */
function db_version() {
    global $pdo;
    return $pdo->query('SELECT VERSION()')->fetchColumn();
}

// Load module helper functions
require_once __DIR__ . '/module_helpers.php';
