<?php
/**
 * Module Manager
 * Handles registration, activation, deactivation of modules
 */

class ModuleManager {
    private $pdo;
    private $modules_dir;
    private $registered_modules = array();

    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->modules_dir = dirname(__DIR__) . '/modules';
        $this->loadRegisteredModules();
    }

    /**
     * Load all registered modules from database
     */
    private function loadRegisteredModules() {
        try {
            $stmt = $this->pdo->query("SELECT * FROM " . DB_PREFIX . "modules_registry ORDER BY sort_order, module_name");
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $this->registered_modules[$row['module_id']] = $row;
            }
        } catch (Exception $e) {
            error_log("ModuleManager: Failed to load modules - " . $e->getMessage());
        }
    }

    /**
     * Scan modules directory and register new modules
     */
    public function scanAndRegister() {
        if (!is_dir($this->modules_dir)) {
            mkdir($this->modules_dir, 0755, true);
        }

        $discovered = array();
        $dirs = scandir($this->modules_dir);

        foreach ($dirs as $dir) {
            if ($dir === '.' || $dir === '..') continue;

            $module_path = $this->modules_dir . '/' . $dir;
            $info_file = $module_path . '/module.json';

            if (is_dir($module_path) && file_exists($info_file)) {
                $info = json_decode(file_get_contents($info_file), true);

                if ($info && isset($info['id']) && isset($info['name'])) {
                    $discovered[] = array(
                        'path' => $module_path,
                        'info' => $info
                    );
                }
            }
        }

        // Register new modules
        foreach ($discovered as $module) {
            $this->registerModule($module['info'], $module['path']);
        }

        return count($discovered);
    }

    /**
     * Register a single module
     */
    private function registerModule($info, $path) {
        $module_id = $info['id'];

        // Check if already registered
        if (isset($this->registered_modules[$module_id])) {
            // Update if version changed
            if ($this->registered_modules[$module_id]['version'] !== $info['version']) {
                $this->updateModule($module_id, $info);
            }
            return;
        }

        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO " . DB_PREFIX . "modules_registry
                (module_id, module_name, description, version, author, icon, status, settings, installed_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");

            $stmt->execute([
                $module_id,
                $info['name'],
                $info['description'] ?? '',
                $info['version'],
                $info['author'] ?? 'SCMS',
                $info['icon'] ?? 'bi-puzzle',
                'inactive', // Default to inactive
                json_encode($info['default_settings'] ?? array())
            ]);

            // Run SQL migration if exists
            $sql_file = $path . '/install.sql';
            if (file_exists($sql_file)) {
                $this->executeSQLFile($sql_file);
            }

            // Run install script if exists
            $install_file = $path . '/install.php';
            if (file_exists($install_file)) {
                require_once $install_file;
                if (function_exists('install_' . $module_id)) {
                    call_user_func('install_' . $module_id, $this->pdo);
                }
            }

            $this->registered_modules[$module_id] = array(
                'module_id' => $module_id,
                'module_name' => $info['name'],
                'version' => $info['version'],
                'status' => 'inactive'
            );

        } catch (Exception $e) {
            error_log("ModuleManager: Failed to register module {$module_id} - " . $e->getMessage());
        }
    }

    /**
     * Update existing module
     */
    private function updateModule($module_id, $info) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE " . DB_PREFIX . "modules_registry
                SET version = ?, description = ?, updated_at = NOW()
                WHERE module_id = ?
            ");

            $stmt->execute([
                $info['version'],
                $info['description'] ?? '',
                $module_id
            ]);

            // Run upgrade script if exists
            $upgrade_file = $this->modules_dir . '/' . $module_id . '/upgrade.php';
            if (file_exists($upgrade_file)) {
                require_once $upgrade_file;
                if (function_exists('upgrade_' . $module_id)) {
                    call_user_func('upgrade_' . $module_id, $this->pdo, $info['version']);
                }
            }

        } catch (Exception $e) {
            error_log("ModuleManager: Failed to update module {$module_id} - " . $e->getMessage());
        }
    }

    /**
     * Activate a module
     */
    public function activate($module_id) {
        if (!isset($this->registered_modules[$module_id])) {
            return array('success' => false, 'message' => 'Module not found');
        }

        // Check dependencies before activation
        $info = $this->getModuleInfo($module_id);
        if ($info && isset($info['requires']['modules'])) {
            $missing_deps = array();
            foreach ($info['requires']['modules'] as $required_module) {
                if (!isset($this->registered_modules[$required_module])) {
                    $missing_deps[] = $required_module . ' (not installed)';
                } elseif ($this->registered_modules[$required_module]['status'] !== 'active') {
                    $missing_deps[] = $required_module . ' (not active)';
                }
            }

            if (!empty($missing_deps)) {
                return array(
                    'success' => false,
                    'message' => 'Cannot activate: Missing dependencies - ' . implode(', ', $missing_deps)
                );
            }
        }

        try {
            $stmt = $this->pdo->prepare("
                UPDATE " . DB_PREFIX . "modules_registry
                SET status = 'active', activated_at = NOW()
                WHERE module_id = ?
            ");

            $stmt->execute([$module_id]);

            // Run activation hook
            $activate_file = $this->modules_dir . '/' . $module_id . '/activate.php';
            if (file_exists($activate_file)) {
                require_once $activate_file;
                if (function_exists('activate_' . $module_id)) {
                    call_user_func('activate_' . $module_id, $this->pdo);
                }
            }

            return array('success' => true, 'message' => 'Module activated successfully');

        } catch (Exception $e) {
            return array('success' => false, 'message' => $e->getMessage());
        }
    }

    /**
     * Deactivate a module
     */
    public function deactivate($module_id) {
        if (!isset($this->registered_modules[$module_id])) {
            return array('success' => false, 'message' => 'Module not found');
        }

        // Check if any active modules depend on this one
        $dependent_modules = array();
        foreach ($this->registered_modules as $id => $module) {
            if ($module['status'] === 'active' && $id !== $module_id) {
                $info = $this->getModuleInfo($id);
                if ($info && isset($info['requires']['modules'])) {
                    if (in_array($module_id, $info['requires']['modules'])) {
                        $dependent_modules[] = $module['module_name'] ?? $module['name'] ?? $id;
                    }
                }
            }
        }

        if (!empty($dependent_modules)) {
            return array(
                'success' => false,
                'message' => 'Cannot deactivate: Required by active modules - ' . implode(', ', $dependent_modules)
            );
        }

        try {
            $stmt = $this->pdo->prepare("
                UPDATE " . DB_PREFIX . "modules_registry
                SET status = 'inactive'
                WHERE module_id = ?
            ");

            $stmt->execute([$module_id]);

            // Run deactivation hook
            $deactivate_file = $this->modules_dir . '/' . $module_id . '/deactivate.php';
            if (file_exists($deactivate_file)) {
                require_once $deactivate_file;
                if (function_exists('deactivate_' . $module_id)) {
                    call_user_func('deactivate_' . $module_id, $this->pdo);
                }
            }

            return array('success' => true, 'message' => 'Module deactivated successfully');

        } catch (Exception $e) {
            return array('success' => false, 'message' => $e->getMessage());
        }
    }

    /**
     * Uninstall a module (remove completely)
     */
    public function uninstall($module_id) {
        if (!isset($this->registered_modules[$module_id])) {
            return array('success' => false, 'message' => 'Module not found');
        }

        try {
            // Run uninstall script
            $uninstall_file = $this->modules_dir . '/' . $module_id . '/uninstall.php';
            if (file_exists($uninstall_file)) {
                require_once $uninstall_file;
                if (function_exists('uninstall_' . $module_id)) {
                    call_user_func('uninstall_' . $module_id, $this->pdo);
                }
            }

            // Remove from registry
            $stmt = $this->pdo->prepare("DELETE FROM " . DB_PREFIX . "modules_registry WHERE module_id = ?");
            $stmt->execute([$module_id]);

            unset($this->registered_modules[$module_id]);

            return array('success' => true, 'message' => 'Module uninstalled successfully');

        } catch (Exception $e) {
            return array('success' => false, 'message' => $e->getMessage());
        }
    }

    /**
     * Get all registered modules
     */
    public function getAllModules() {
        return $this->registered_modules;
    }

    /**
     * Check if a module is active
     */
    public function isActive($module_id) {
        return isset($this->registered_modules[$module_id]) &&
               $this->registered_modules[$module_id]['status'] === 'active';
    }

    /**
     * Get all active module IDs
     */
    public function getActiveModules() {
        $active = array();
        foreach ($this->registered_modules as $id => $module) {
            if ($module['status'] === 'active') {
                $active[] = $id;
            }
        }
        return $active;
    }

    /**
     * Get module settings
     */
    public function getSettings($module_id) {
        if (!isset($this->registered_modules[$module_id])) {
            return array();
        }

        $settings = $this->registered_modules[$module_id]['settings'];
        return is_string($settings) ? json_decode($settings, true) : $settings;
    }

    /**
     * Update module settings
     */
    public function updateSettings($module_id, $settings) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE " . DB_PREFIX . "modules_registry
                SET settings = ?, updated_at = NOW()
                WHERE module_id = ?
            ");

            $stmt->execute([json_encode($settings), $module_id]);

            $this->registered_modules[$module_id]['settings'] = json_encode($settings);

            return array('success' => true, 'message' => 'Settings updated successfully');

        } catch (Exception $e) {
            return array('success' => false, 'message' => $e->getMessage());
        }
    }

    /**
     * Get module info
     */
    public function getModuleInfo($module_id) {
        $info_file = $this->modules_dir . '/' . $module_id . '/module.json';

        if (file_exists($info_file)) {
            return json_decode(file_get_contents($info_file), true);
        }

        return null;
    }

    /**
     * Execute SQL file for module installation
     */
    private function executeSQLFile($file_path) {
        try {
            $sql = file_get_contents($file_path);

            // Replace DB_PREFIX placeholder if exists
            $sql = str_replace('{{DB_PREFIX}}', DB_PREFIX, $sql);
            $sql = str_replace('{DB_PREFIX}', DB_PREFIX, $sql);

            // Split by semicolon and execute each statement
            $statements = array_filter(
                array_map('trim', explode(';', $sql)),
                function($stmt) {
                    return !empty($stmt) && !preg_match('/^--/', $stmt);
                }
            );

            foreach ($statements as $statement) {
                if (!empty($statement)) {
                    $this->pdo->exec($statement);
                }
            }

            return true;
        } catch (Exception $e) {
            error_log("ModuleManager: SQL execution failed - " . $e->getMessage());
            return false;
        }
    }
}
